//  Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef LISTEROLA_WINDOW_H
#define LISTEROLA_WINDOW_H

#include <gtkmm.h>
#include <string>
#include <sigc++/connection.h>
#include "event-date.h"


class Listerola;
class FrameBox;

class ListerolaWindow
{
 public:
    ListerolaWindow(Listerola &l, double num_row_channels, double tick_speed, 
                    double row_pause, double flipahead_interval, 
                    std::string advertisement_image_filename,
                    double vertical_scroll_step, double row_height, 
                    double hour_width, double channel_width);

    void set_number_of_channels(double num_row_channels);
    void set_scrolling_speed(double tick_speed);
    void set_scrolling_row_pause(double row_pause);
    void set_column_flipahead_time(double interval);
    void set_advertisement_image(std::string image_filename, std::string wipestyle);

    Gtk::Window *get_window() {return window;};
    bool run();
    void hide();
    
 private:
    Listerola &listerola;
    Gtk::Window *window;
    Gtk::Image *advertisement_image;
    Gtk::Viewport *listerola_viewport;
    Gtk::Adjustment hadj;
    Gtk::Adjustment vadj;
    Gtk::HBox *listing_hbox;
    Gtk::VBox *listing_vbox;
    Gtk::VBox *header_vbox;
    Gtk::HBox *header_hbox;
    Gtk::HBox *scroll_hbox;
    Gtk::VBox *scroll_vbox;
    Gtk::HBox *ad_hbox;
    Gtk::VBox *ad_vbox;
  
    FrameBox* timebox;

    //! The filename of the file that holds the channel listing data.
    std::string listing_filename;

    //! The number of channels shown at a time.
    double row_channels;

    //! The number of hours shown at a time.
    double column_hours;

    //! The speed at which we scroll rows.  milliseconds pause between pixels.
    double scroll_speed;

    //! The amount of time we wait while showing a row.
    double scroll_row_pause;

    //! How far in advance we show the next half hour.
    double flipahead_interval;

    int row_height;

    //! use this variable to add the on-deck hbox the first time through.
    bool add_more_rows;


    //! The filename of the file that holds the current advertisement.
    std::string advertisement_image_filename;

    //! The timer connection for the scrolling mechanism.
    sigc::connection scroll_tick_handler;

    //! the timer connection that runs every second.
    sigc::connection second_handler;

    //must be a multiple of row_height
    double vertical_scroll_step;

    //how many pixels an hour takes up.
    double hour_width;

    //how many pixels a channel box takes up.
    double channel_width;

    //helpers
    void update_listing_file(std::string listing_file);
    void update_advertisement_image(std::string advertisement_image_file, std::string wipestyle);
    void update_scroll_box_size(double rows, double columns);
    void update_scrolling_speed(double tick_speed);
    Gtk::HBox *get_next_row();
    void fill_headers();

    //! Callbacks
    bool on_vertical_scroll_tick();
    bool on_vertical_scroll_pause();
    bool on_vertical_scroll_resume();
    void on_row_scrolled(Gtk::HBox *row);
    bool on_second_passed();
    void on_column_scrolled();


};

#endif
