//  Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include "config.h"
#include <assert.h>
#include <sigc++/functors/mem_fun.h>
#include <sigc++/bind.h>
#include <errno.h>
#include <string.h>

#include <gtkmm.h>
#include "main.h"
#include "driver.h"
#include "listerola.h"
#include "listerola-config-loader.h"
#include "listerola-config-saver.h"

Driver::Driver(std::string cfg_filename)
{
  config_filename = cfg_filename;
  preferences_window = NULL;

  listerola_opts = NULL;
  listerola_view_opts = NULL;
  bool config_found = ListerolaConfigLoader::load_from_file
    (config_filename, &listerola_opts, &listerola_view_opts);
  if (!config_found)
    {
      //open the preferences window with some defaults.
      listerola_opts = new ListerolaOptions();
      listerola_view_opts = new ListerolaViewOptions();
      ListerolaConfigSaver::save_to_file(config_filename, 
                                         listerola_opts, listerola_view_opts);
    }

  listerola = new Listerola(*listerola_opts);
  run();
  Pango::AttrList event_attrs;
  Pango::Attribute event_attr = 
    Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
  Pango::AttrList host_attrs;
  Pango::Attribute host_attr = 
    Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
  host_attrs.insert(host_attr);
  Pango::AttrList header_attrs;
  Pango::Attribute header_attr = 
    Pango::Attribute::create_attr_weight(Pango::WEIGHT_BOLD);
  header_attrs.insert(header_attr);
  listerola_window = new ListerolaWindow(*listerola, *listerola_view_opts, 
                                         event_attrs, host_attrs, header_attrs);
  listerola_window->quit_requested.connect
    (sigc::mem_fun(*this, &Driver::on_quit_requested));
  listerola_window->open_preferences_requested.connect
    (sigc::mem_fun(*this, &Driver::on_preferences_requested));
  listerola_window->show();
  if (!config_found)
    {
      std::list<std::string> uncoloured_genres;
      std::list<std::string> host_names;
      on_preferences_requested(uncoloured_genres, host_names);
    }
}

void Driver::run()
{
  //here we react to any command-line options that may have been given.
  //for now we don't have any.
  return;
}

Driver::~Driver()
{
  if (listerola_window)
    delete listerola_window;
  if (preferences_window)
    delete preferences_window;
  if (listerola)
    delete listerola;
  if (listerola_opts)
    delete listerola_opts;
  if (listerola_view_opts)
    delete listerola_view_opts;
}

void Driver::on_preferences_requested(std::list<std::string> uncoloured_genres,
                                      std::list<std::string> host_names)
{
  if (preferences_window == NULL)
    {
      preferences_window = new PreferencesWindow(*listerola_opts,
                                                 *listerola_view_opts,
                                                 uncoloured_genres,
                                                 host_names);
    
      listerola_window->update_genres_in_preferences_requested.connect
        (sigc::mem_fun(*preferences_window, &PreferencesWindow::set_genres));
      listerola_window->update_hosts_in_preferences_requested.connect
        (sigc::mem_fun(*preferences_window, &PreferencesWindow::set_host_names));
      listerola->bad_listing_file.connect
        (sigc::mem_fun(*preferences_window, &PreferencesWindow::on_bad_listing_file_loaded));
      preferences_window->save_configuration_requested.connect
        (sigc::mem_fun(*this, &Driver::on_save_configuration_requested));
                   
    }
  preferences_window->show();
}

void Driver::on_quit_requested()
{
  //screw you guys, i'm going home.
  if (listerola_window)
    listerola_window->hide();
  if (preferences_window)
    preferences_window->hide();
      
  Main::instance().stop_main_loop();
}
        
void Driver::on_save_configuration_requested()
{
  if (listerola_opts && listerola_view_opts)
    ListerolaConfigSaver::save_to_file(config_filename, listerola_opts, 
                                       listerola_view_opts);
}
