//  Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>

#include <gtkmm/builder.h>
#include "preferences-window.h"
#include "file.h"
#include "add-genre-colour-dialog.h"

PreferencesWindow::PreferencesWindow (ListerolaOptions &listerola_opts,
                                      ListerolaViewOptions &listerola_view_opts,
                                      std::list<std::string> avail_genres,
                                      std::list<std::string> hosts)
  : listerola_options(listerola_opts), 
    listerola_view_options(listerola_view_opts), 
    name_column("Name", name_renderer),
    host_name_column("Name", host_name_renderer),
    host_number_column("Number", host_number_renderer)
{
  uncoloured_genres = avail_genres;
  host_names = hosts;
  Glib::RefPtr < Gtk::Builder > xml =
    Gtk::Builder::create_from_file (File::get_glade_path () + 
                                    "/listerola-config.gtk");

  xml->get_widget ("window", window);
  window->set_icon_from_file(File::get_data_path() + "/listerola.png");
  xml->get_widget ("close_button", close_button);
  close_button->signal_clicked().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_close_clicked));
  xml->get_widget ("num_rows_spinbutton", num_rows_spinbutton);
  num_rows_spinbutton->set_value(listerola_options.get_num_rows());
  num_rows_spinbutton->signal_changed().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_num_rows_changed));
  num_rows_spinbutton->signal_insert_text().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_num_rows_text_changed));
  xml->get_widget ("duration_spinbutton", duration_spinbutton);
  duration_spinbutton->set_value(listerola_options.get_duration());
  duration_spinbutton->signal_changed().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_duration_changed));
  duration_spinbutton->signal_insert_text().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_duration_text_changed));
  xml->get_widget ("interval_spinbutton", interval_spinbutton);
  interval_spinbutton->set_value(listerola_options.get_interval());
  interval_spinbutton->signal_changed().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_interval_changed));
  interval_spinbutton->signal_insert_text().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_interval_text_changed));
  xml->get_widget ("flipahead_spinbutton", flipahead_interval_spinbutton);
  flipahead_interval_spinbutton->set_value
    (listerola_options.get_flipahead_interval());
  flipahead_interval_spinbutton->signal_changed().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_flipahead_interval_changed));
  flipahead_interval_spinbutton->signal_insert_text().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_flipahead_interval_text_changed));

  xml->get_widget ("none_radiobutton", none_radiobutton);
  if (listerola_options.get_listing_kind() == ListerolaOptions::SOURCE_NONE)
    none_radiobutton->set_active(true);
  none_radiobutton->signal_toggled().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_none_toggled));
  xml->get_widget ("xmltv_radiobutton", xmltv_radiobutton);
  if (listerola_options.get_listing_kind() == ListerolaOptions::SOURCE_XMLTV)
    xmltv_radiobutton->set_active(true);
  xmltv_radiobutton->signal_toggled().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_xmltv_toggled));
  xml->get_widget ("mrbs_radiobutton", mrbs_radiobutton);
  if (listerola_options.get_listing_kind() == ListerolaOptions::SOURCE_MRBS)
    mrbs_radiobutton->set_active(true);
  mrbs_radiobutton->signal_toggled().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_mrbs_toggled));
  xml->get_widget ("xmltv_filechooserbutton", xmltv_filechooserbutton);
  if (listerola_options.get_listing_kind() == ListerolaOptions::SOURCE_XMLTV &&
      listerola_options.get_listing_file() != "")
    xmltv_filechooserbutton->set_filename(listerola_options.get_listing_file());
  xmltv_filechooserbutton->signal_selection_changed().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_xmltv_changed));
  xml->get_widget ("mrbs_filechooserbutton", mrbs_filechooserbutton);
  if (listerola_options.get_listing_kind() == ListerolaOptions::SOURCE_MRBS &&
      listerola_options.get_listing_file() != "")
    mrbs_filechooserbutton->set_filename(listerola_options.get_listing_file());
  mrbs_filechooserbutton->signal_selection_changed().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_mrbs_changed));
  if (listerola_options.get_listing_file() == "" && listerola_options.get_listing_kind() == ListerolaOptions::SOURCE_NONE)
    none_radiobutton->set_active(true);
  xml->get_widget ("refresh_checkbutton", refresh_checkbutton);
  refresh_checkbutton->set_active(listerola_view_options.get_refresh_interval() > 0.0);
  refresh_checkbutton->signal_toggled().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_refresh_toggled));
  xml->get_widget ("refresh_spinbutton", refresh_spinbutton);
  refresh_spinbutton->set_sensitive(refresh_checkbutton->get_active());
  refresh_spinbutton->set_value
    (listerola_view_options.get_refresh_interval());
  refresh_spinbutton->signal_changed().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_refresh_changed));
  refresh_spinbutton->signal_insert_text().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_refresh_text_changed));
  xml->get_widget ("speed_spinbutton", speed_spinbutton);
  speed_spinbutton->set_value (listerola_view_options.get_scroll_speed());
  speed_spinbutton->signal_changed().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_speed_changed));
  speed_spinbutton->signal_insert_text().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_speed_text_changed));
  xml->get_widget ("pause_spinbutton", pause_spinbutton);
  pause_spinbutton->set_value (listerola_view_options.get_scroll_row_pause());
  pause_spinbutton->signal_changed().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_pause_changed));
  pause_spinbutton->signal_insert_text().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_pause_text_changed));
  xml->get_widget ("step_spinbutton", step_spinbutton);
  step_spinbutton->set_value
    (listerola_view_options.get_vertical_scroll_step());
  step_spinbutton->signal_changed().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_step_changed));
  step_spinbutton->signal_insert_text().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_step_text_changed));
  xml->get_widget ("window_bg_colorbutton", window_bg_colorbutton);
  window_bg_colorbutton->set_color(listerola_view_options.get_background_colour());
  window_bg_colorbutton->signal_color_set().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_background_colour_changed));
  xml->get_widget ("host_bg_colorbutton", host_bg_colorbutton);
  host_bg_colorbutton->set_color(listerola_view_options.get_host_background_colour());
  host_bg_colorbutton->signal_color_set().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_host_background_colour_changed));
  xml->get_widget ("header_bg_colorbutton", header_bg_colorbutton);
  header_bg_colorbutton->set_color(listerola_view_options.get_header_background_colour());
  header_bg_colorbutton->signal_color_set().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_header_background_colour_changed));
  xml->get_widget ("title_bg_colorbutton", title_bg_colorbutton);
  title_bg_colorbutton->set_color(listerola_view_options.get_title_background_colour());
  title_bg_colorbutton->signal_color_set().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_title_background_colour_changed));
  xml->get_widget ("fullscreen_checkbutton", fullscreen_checkbutton);
  fullscreen_checkbutton->set_active(listerola_view_options.get_start_in_fullscreen());
  fullscreen_checkbutton->signal_toggled().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_fullscreen_toggled));
  xml->get_widget ("refresh_button", refresh_button);
  refresh_button->signal_clicked().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_refresh_clicked));
  xml->get_widget("title_checkbutton", title_checkbutton);
  title_checkbutton->set_active(listerola_view_opts.get_show_title());
  title_checkbutton->signal_toggled().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_title_toggled));
  xml->get_widget("title_entry", title_entry);
  title_entry->set_text(listerola_view_opts.get_title());
  title_entry->signal_changed().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_title_changed));
  xml->get_widget ("event_fg_colorbutton", event_fg_colorbutton);
  event_fg_colorbutton->set_color(listerola_view_options.get_event_foreground_colour());
  event_fg_colorbutton->signal_color_set().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_event_foreground_colour_changed));
  xml->get_widget ("host_fg_colorbutton", host_fg_colorbutton);
  host_fg_colorbutton->set_color(listerola_view_options.get_host_foreground_colour());
  host_fg_colorbutton->signal_color_set().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_host_foreground_colour_changed));
  xml->get_widget ("header_fg_colorbutton", header_fg_colorbutton);
  header_fg_colorbutton->set_color(listerola_view_options.get_header_foreground_colour());
  header_fg_colorbutton->signal_color_set().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_header_foreground_colour_changed));
  xml->get_widget ("title_fg_colorbutton", title_fg_colorbutton);
  title_fg_colorbutton->set_color(listerola_view_options.get_title_foreground_colour());
  title_fg_colorbutton->signal_color_set().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_title_foreground_colour_changed));
  xml->get_widget ("add_genre_button", add_genre_button);
  add_genre_button->signal_clicked().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_add_genre_colour_clicked));
  xml->get_widget ("remove_genre_button", remove_genre_button);
  remove_genre_button->signal_clicked().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_remove_genre_colour_clicked));
  xml->get_widget ("edit_genre_button", edit_genre_button);
  edit_genre_button->signal_clicked().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_edit_genre_colour_clicked));
  xml->get_widget ("unmatched_genre_bg_colorbutton", unmatched_genre_bg_colorbutton);
  unmatched_genre_bg_colorbutton->set_color(listerola_view_options.get_unmatched_genre_background_colour());
  unmatched_genre_bg_colorbutton->signal_color_set().connect
    (sigc::mem_fun(this, &PreferencesWindow::on_unmatched_genre_background_colour_changed));
  xml->get_widget("genre_treeview", genre_treeview);
  genre_list = Gtk::ListStore::create(genre_columns);
  genre_treeview->set_model(genre_list);
  name_renderer.property_editable() = true;
  name_renderer.signal_edited().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_name_edited));
  name_column.set_cell_data_func
    (name_renderer, sigc::mem_fun(*this, &PreferencesWindow::cell_data_name));
  genre_treeview->append_column(name_column);

  std::list<std::string> genres = listerola_view_opts.get_genres();
  for (std::list<std::string>::iterator it = genres.begin(); 
       it != genres.end(); it++)
    {
      Gdk::Color colour = listerola_view_opts.get_genre_colour(*it);
      add_genre(*it, colour);
    }
  genre_treeview->get_selection()->signal_changed().connect
      (sigc::mem_fun(*this, &PreferencesWindow::on_genre_selected));
  xml->get_widget ("show_host_numbers_checkbutton", show_host_numbers_checkbutton);
  show_host_numbers_checkbutton->set_active(listerola_view_options.get_show_host_numbers());
  show_host_numbers_checkbutton->signal_toggled().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_show_host_numbers_toggled));

  xml->get_widget("host_treeview", host_treeview);
  host_list = Gtk::ListStore::create(host_columns);
  host_treeview->set_model(host_list);
  host_name_renderer.property_editable() = true;
  host_name_renderer.signal_edited().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_host_name_edited));
  host_name_column.set_cell_data_func
    (host_name_renderer, sigc::mem_fun(*this, &PreferencesWindow::cell_data_host_name));
  host_treeview->append_column(host_name_column);
  host_number_renderer.property_editable() = true;
  host_number_renderer.signal_edited().connect
    (sigc::mem_fun(*this, &PreferencesWindow::on_host_number_edited));
  host_number_column.set_cell_data_func
    (host_number_renderer, sigc::mem_fun(*this, &PreferencesWindow::cell_data_host_number));
  host_treeview->append_column(host_number_column);

  for (std::list<std::string>::iterator it = host_names.begin(); 
       it != host_names.end(); it++)
    {
      int number = listerola_opts.get_host_number(*it);
      add_host(*it, number);
    }
  update_radiobuttons();
  update_genre_buttons();
}

PreferencesWindow::~PreferencesWindow ()
{
}

void PreferencesWindow::on_flipahead_interval_changed()
{
  listerola_options.set_flipahead_interval
    (flipahead_interval_spinbutton->get_value());
}

void PreferencesWindow::on_interval_changed()
{
  listerola_options.set_interval(interval_spinbutton->get_value());
}

void PreferencesWindow::on_duration_changed()
{
  listerola_options.set_duration(duration_spinbutton->get_value());
}

void PreferencesWindow::on_num_rows_changed()
{
  listerola_options.set_num_rows(num_rows_spinbutton->get_value());
}

void
PreferencesWindow::on_close_clicked()
{
  save_configuration_requested.emit();
  window->hide();
}

void
PreferencesWindow::hide ()
{
  window->hide ();
}

void
PreferencesWindow::show()
{
  window->show_all ();
  return;
}

void PreferencesWindow::update_radiobuttons()
{
  xmltv_filechooserbutton->set_sensitive(xmltv_radiobutton->get_active());
  mrbs_filechooserbutton->set_sensitive(mrbs_radiobutton->get_active());
  if (none_radiobutton->get_active())
    refresh_button->set_sensitive(false);
  else
    refresh_button->set_sensitive(true);
}

void PreferencesWindow::on_none_toggled()
{
  update_radiobuttons();
  listerola_options.clear_listing_file();
}

void PreferencesWindow::on_xmltv_toggled()
{
  update_radiobuttons();
}

void PreferencesWindow::on_mrbs_toggled()
{
  update_radiobuttons();
}

void PreferencesWindow::on_xmltv_changed()
{
  if (xmltv_filechooserbutton->get_filename() == "")
    return;
  listerola_options.set_xmltv_listing_file(xmltv_filechooserbutton->get_filename());
}

void PreferencesWindow::on_mrbs_changed()
{
  if (mrbs_filechooserbutton->get_filename() == "")
    return;
  listerola_options.set_mrbs_listing_file(mrbs_filechooserbutton->get_filename());
}

void PreferencesWindow::on_refresh_toggled()
{
  refresh_spinbutton->set_sensitive(refresh_checkbutton->get_active());
  if (refresh_checkbutton->get_active() == false)
    listerola_view_options.set_refresh_interval(0.0);
  else
    listerola_view_options.set_refresh_interval(refresh_spinbutton->get_value());
}

void PreferencesWindow::on_refresh_changed()
{
  listerola_view_options.set_refresh_interval(refresh_spinbutton->get_value());
}

void PreferencesWindow::on_num_rows_text_changed(const Glib::ustring &s, int *p)
{
  num_rows_spinbutton->set_value(atof(num_rows_spinbutton->get_text().c_str()));
  on_num_rows_changed();
}

void PreferencesWindow::on_duration_text_changed(const Glib::ustring &s, int *p)
{
  duration_spinbutton->set_value(atof(duration_spinbutton->get_text().c_str()));
  on_duration_changed();
}

void PreferencesWindow::on_interval_text_changed(const Glib::ustring &s, int *p)
{
  interval_spinbutton->set_value(atof(interval_spinbutton->get_text().c_str()));
  on_interval_changed();
}

void PreferencesWindow::on_flipahead_interval_text_changed(const Glib::ustring &s, int *p)
{
  flipahead_interval_spinbutton->set_value(atof(flipahead_interval_spinbutton->get_text().c_str()));
  on_flipahead_interval_changed();
}

void PreferencesWindow::on_refresh_text_changed(const Glib::ustring &s, int *p)
{
  refresh_spinbutton->set_value(atof(refresh_spinbutton->get_text().c_str()));
  on_refresh_changed();
}

void PreferencesWindow::on_speed_changed()
{
  listerola_view_options.set_scroll_speed(speed_spinbutton->get_value());
}

void PreferencesWindow::on_speed_text_changed(const Glib::ustring &s, int *p)
{
  speed_spinbutton->set_value(atof(speed_spinbutton->get_text().c_str()));
  on_speed_changed();
}

void PreferencesWindow::on_pause_changed()
{
  listerola_view_options.set_scroll_row_pause(pause_spinbutton->get_value());
}

void PreferencesWindow::on_pause_text_changed(const Glib::ustring &s, int *p)
{
  pause_spinbutton->set_value(atof(pause_spinbutton->get_text().c_str()));
  on_pause_changed();
}

void PreferencesWindow::on_step_changed()
{
  listerola_view_options.set_vertical_scroll_step(step_spinbutton->get_value());
}

void PreferencesWindow::on_step_text_changed(const Glib::ustring &s, int *p)
{
  step_spinbutton->set_value(atof(step_spinbutton->get_text().c_str()));
  on_step_changed();
}
    
void PreferencesWindow::on_background_colour_changed()
{
  listerola_view_options.set_background_colour(window_bg_colorbutton->get_color());
}

void PreferencesWindow::on_host_background_colour_changed()
{
  listerola_view_options.set_host_background_colour(host_bg_colorbutton->get_color());
}

void PreferencesWindow::on_header_background_colour_changed()
{
  listerola_view_options.set_header_background_colour(header_bg_colorbutton->get_color());
}

void PreferencesWindow::on_title_background_colour_changed()
{
  listerola_view_options.set_title_background_colour(title_bg_colorbutton->get_color());
}

void PreferencesWindow::on_fullscreen_toggled()
{
  listerola_view_options.set_start_in_fullscreen(fullscreen_checkbutton->get_active());
}

void PreferencesWindow::on_refresh_clicked()
{
  std::string file = listerola_options.get_listing_file();
  switch (listerola_options.get_listing_kind())
    {
    case ListerolaOptions::SOURCE_NONE:
      listerola_options.clear_listing_file();
      break;
    case ListerolaOptions::SOURCE_RANDOM:
      listerola_options.set_random_listing_file(file);
      break;
    case ListerolaOptions::SOURCE_XMLTV:
      listerola_options.set_xmltv_listing_file(file);
      break;
    case ListerolaOptions::SOURCE_MRBS:
      listerola_options.set_mrbs_listing_file(file);
      break;
    }
}

void PreferencesWindow::on_title_toggled()
{
  listerola_view_options.set_show_title(title_checkbutton->get_active());
}

void PreferencesWindow::on_title_changed()
{
  listerola_view_options.set_title(title_entry->get_text());
}

void PreferencesWindow::on_event_foreground_colour_changed()
{
  listerola_view_options.set_event_foreground_colour(event_fg_colorbutton->get_color());
  genre_treeview->hide();
  genre_treeview->show_all();
  genre_treeview->signal_composited_changed();
}

void PreferencesWindow::on_header_foreground_colour_changed()
{
  listerola_view_options.set_header_foreground_colour(header_fg_colorbutton->get_color());
}

void PreferencesWindow::on_host_foreground_colour_changed()
{
  listerola_view_options.set_host_foreground_colour(host_fg_colorbutton->get_color());
}

void PreferencesWindow::on_title_foreground_colour_changed()
{
  listerola_view_options.set_title_foreground_colour(title_fg_colorbutton->get_color());
}

void PreferencesWindow::on_add_genre_colour_clicked()
{
  AddGenreColourDialog dialog(uncoloured_genres);
  dialog.set_parent_window(*window);
  int response = dialog.run();
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      std::pair<std::string, Gdk::Color> genre_colour;
      genre_colour = dialog.get_genre_colour();
      //now add it to our treeview here.
      add_genre(genre_colour.first, genre_colour.second);
      listerola_view_options.set_genre_colour
        (genre_colour.first, genre_colour.second);
    }
}

void PreferencesWindow::on_remove_genre_colour_clicked()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = genre_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();

  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Glib::ustring genre = row[genre_columns.name];
      listerola_view_options.clear_genre_colour(genre);
      genre_list->erase(iterrow);
    }
}

void PreferencesWindow::on_unmatched_genre_background_colour_changed()
{
  listerola_view_options.set_unmatched_genre_background_colour(unmatched_genre_bg_colorbutton->get_color());
}

void
PreferencesWindow::cell_data_name(Gtk::CellRenderer *renderer, const Gtk::TreeIter &i)
{
  Gtk::CellRendererText*cell = dynamic_cast<Gtk::CellRendererText*>(renderer);
  cell->property_text() = (*i)[genre_columns.name];
  cell->property_background_gdk() = (*i)[genre_columns.colour];
  cell->property_foreground_gdk() = listerola_view_options.get_event_foreground_colour();
}

void
PreferencesWindow::on_name_edited(const Glib::ustring &path,
                                  const Glib::ustring &new_text)
{
  Gtk::TreeModel::iterator iterrow = genre_list->get_iter(Gtk::TreePath(path));
  Gtk::TreeModel::Row row = *iterrow;
  Glib::ustring old_genre = row[genre_columns.name];
  listerola_view_options.clear_genre_colour(old_genre);
  row[genre_columns.name] = new_text;
  Glib::ustring new_genre = row[genre_columns.name];
  listerola_view_options.set_genre_colour(new_genre, row[genre_columns.colour]);
}

void
PreferencesWindow::add_genre(Glib::ustring name, Gdk::Color colour)
{
  Gtk::TreeIter i = genre_list->append();
  (*i)[genre_columns.name] = name;
  (*i)[genre_columns.colour] = colour;
}

void PreferencesWindow::on_edit_genre_colour_clicked()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = genre_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();

  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Gtk::ColorSelectionDialog dialog("Choose a new Genre Color for " + 
                                       row[genre_columns.name]);
      dialog.get_colorsel()->set_current_color(row[genre_columns.colour]);
      dialog.run();
      row[genre_columns.colour] = dialog.get_colorsel()->get_current_color();
      Glib::ustring genre = row[genre_columns.name];
      listerola_view_options.set_genre_colour(genre, 
                                              row[genre_columns.colour]);
    }
}

void PreferencesWindow::update_genre_buttons()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = genre_treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();

  if (iterrow)
    {
      edit_genre_button->set_sensitive(true);
      remove_genre_button->set_sensitive(true);
    }
  else
    {
      edit_genre_button->set_sensitive(false);
      remove_genre_button->set_sensitive(false);
    }
}

void PreferencesWindow::on_genre_selected()
{
  update_genre_buttons();
}
    
void PreferencesWindow::on_show_host_numbers_toggled()
{
  listerola_view_options.set_show_host_numbers(show_host_numbers_checkbutton->get_active());
}

void
PreferencesWindow::cell_data_host_name(Gtk::CellRenderer *renderer, const Gtk::TreeIter &i)
{
  Gtk::CellRendererText*cell = dynamic_cast<Gtk::CellRendererText*>(renderer);
  if ((*i)[host_columns.name] == "")
    cell->property_text() = (*i)[host_columns.true_name];
  else
    cell->property_text() = (*i)[host_columns.name];
}

void
PreferencesWindow::on_host_name_edited(const Glib::ustring &path,
                                       const Glib::ustring &new_text)
{
  Gtk::TreeModel::iterator iterrow = host_list->get_iter(Gtk::TreePath(path));
  Gtk::TreeModel::Row row = *iterrow;
  Glib::ustring true_name = row[host_columns.true_name];
  row[host_columns.name] = new_text;
  Glib::ustring new_host = row[host_columns.name];
  if (new_host == "" || true_name == new_host)
    listerola_options.clear_host_name(true_name);
  else
    listerola_options.set_host_name(true_name, new_host);
}

void
PreferencesWindow::cell_data_host_number(Gtk::CellRenderer *renderer, const Gtk::TreeIter &i)
{
  dynamic_cast<Gtk::CellRendererSpin*>(renderer)->property_adjustment()
    = new Gtk::Adjustment((*i)[host_columns.number], 0, 10000, 1);
  char buf[32];
  int num = (*i)[host_columns.number];
  snprintf(buf,sizeof(buf), "%d", num);
  Glib::ustring new_number = buf;
  dynamic_cast<Gtk::CellRendererSpin*>(renderer)->property_text() = new_number;

}

void
PreferencesWindow::on_host_number_edited(const Glib::ustring &path,
                                         const Glib::ustring &new_text)
{
  int number = atoi(new_text.c_str());
  (*host_list->get_iter(Gtk::TreePath(path)))[host_columns.number] = number;
  Glib::ustring name =
    (*host_list->get_iter(Gtk::TreePath(path)))[host_columns.true_name];
  if (number == -1)
    listerola_options.clear_host_number(name);
  else
    listerola_options.set_host_number(name, number);
}
      
void
PreferencesWindow::add_host(Glib::ustring name, int number)
{
  Gtk::TreeIter i = host_list->append();
  (*i)[host_columns.true_name] = name;
  (*i)[host_columns.name] = listerola_options.get_host_name(name);
  (*i)[host_columns.number] = number;
}
    
void PreferencesWindow::on_bad_listing_file_loaded()
{
  Gtk::MessageDialog dialog(*window, "Unexpected error while reading file.");
  dialog.run();
  dialog.hide();
  on_none_toggled();
}
