# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 692 $
############################################################################
#    Copyright (C) 2005 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA          #
############################################################################
"""Abstract file interface for structured file types."""

#-- imports --------------------------------------------------------------------
#-- Python imports
from types import DictionaryType

#-- own imports
from Interface import ILMCFile, ILMCRecord
from Language import TR


class AbstractFile(ILMCFile):
    """Abstract file interface.

    All files that are supported should implement these methods.
    This is an abstract class. All methods related to the file
    interface raise NotImplementedError.

    A LMC module that implements a structured file type should
    raise TypeError when the file does not fit the correct
    structure.

    Header information should be accessible via the dictionary
    interface, already implemented here.
    Deletion of fields should not be permitted and raise KeyError.
    Read access to non-existing items should return None and *not* raise
    an exception.
    Write access to non-existing fields should be permitted.

    When subclassing this class, remember to implement the 6 file
    attributes accessible as data members.
    Also the methods from IFile interface need to be implemented
    and probably getRecordFactory().
    """
    def __init__(self):
        self = {}

    #-- dictionary interface
    def __delitem__(self, name):
        """del x[name] -> None. Remove item <name>"""
        raise KeyError(TR("Operation not permitted."))
    def clear(self):
        """D.clear() -> None.  Remove all items from D"""
        raise KeyError(TR("Operation not permitted."))
    def popitem(self):
        """D.popitem() -> (k, v), remove and return some (key, value) pair as a 2-tuple"""
        raise KeyError(TR("Operation not permitted."))
    def pop(self, k,d=None):
        """D.pop(k[,d]) -> v, remove specified key and return the corresponding value"""
        raise KeyError(TR("Operation not permitted."))

    def __iter__(self):
        """returns an iterator which itererates over all records"""
        self.seek(0)
        while (not self.isEOF()):
            yield self.read(1)[0]

    def __getitem__(self, name):
        """returns the value associated with a key; None if key not present"""
        if (self.has_key(name)):
            return ILMCFile.__getitem__(self,name)
        

class AbstractFileRecord(ILMCRecord):
    """Abstract Base class for file records

    A record stores all fields of a file record and provides
    the dictionary interface to access its fields (already
    implemented here).
    
    Deletion of fields should not be permitted and raise KeyError.
    Read access to non-existing fields should return None and *not*
    raise an exception.
    Write access to non-existing fields should be permitted.

    The constructor supports the copy-constructor interface, i.e.
    when a AbstractFileRecord or a dictionary is passed, all fields
    are initialised from the object or dictionary.

    Additionally supported methods that need to be implemented are:
    - dump   for returning a binary representation as string
    - load   return the binary representation as an object
    """
    def __init__(self, arg={}):
        self.__data = {}
        self.__modified = False        

        if ((type(arg) is DictionaryType) or isinstance(arg[0], AbstractFileRecord)):
            for field in arg:
                self[field] = arg[field]


    #-- dictionary interface ---------------------------------------------------
    def __delitem__(self, name):
        """del x[name] -> None. Remove item <name>"""
        raise KeyError(TR("Operation not permitted."))
    def clear(self):
        """D.clear() -> None.  Remove all items from D"""
        raise KeyError(TR("Operation not permitted."))
    def popitem(self):
        """D.popitem() -> (k, v), remove and return some (key, value) pair as a 2-tuple"""
        raise KeyError(TR("Operation not permitted."))
    def pop(self, k,d=None):
        """D.pop(k[,d]) -> v, remove specified key and return the corresponding value"""
        raise KeyError(TR("Operation not permitted."))
        
    def keys(self): return self.__data.keys()
    def values(self): return self.__data.values()
    def items(self): return self.__data.items()
    def has_key(self, k): return self.__data.has_key(k)
    def get(self, k,d=None): return self.__data.get(k, d)
    
    def setdefault(self, k,d=None):
        self.__modified = (not k in self)
        return self.__data.setdefault(k,d)

    def iterkeys(self): return self.__data.iterkeys()
    def itervalues(self): return self.__data.itervalues()
    def iteritems(self): return self.__data.iteritems()

    def update(self, E):
        for k in E:
            self.__modified |= (self[k] != E[k]) # no KeyError raised
        self.__data.update(E)          

    def __iter__(self): return iter(self.__data)
    def next(self): return self.__data.next()
    def __contains__(self, y): return (y in self.__data)
    def __len__(self): return len(self.__data)

    def __setitem__(self, name, value):
        if (name in self):
            if (self.__data[name] != value):
                self.__data[name] = value
                self.__modified = True
        else:
            self.__data[name] = value
            self.__modified = True


    def __getitem__(self, name):
        if (name in self):
            return self.__data[name]

    def copy(self):
        """returns a (deep) copy of the object"""
        return copy.deepcopy(self)

    def __repr__(self):
        """returns a (unicode) string representation of __data"""
        return unicode(self.__data)
 
    def __str__(self):
        """returns a string representation of __data and the modified flag"""
        return unicode(self.__class__)+u"(%s, %s)" %(unicode(self.__data), unicode(self.__modified))

    #-- extended interface -----------------------------------------------------
    def modified(self, changed=None):
        """returns or sets the modified flag for this record"""
        if (changed == None): return self.__modified
        self.__modified = changed


#--EOF--
