/*  Clzip - LZMA lossless data compressor
    Copyright (C) 2010, 2011, 2012, 2013 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

enum { rd_buffer_size = 16384 };

struct Range_decoder
  {
  unsigned long long partial_member_pos;
  uint8_t * buffer;		/* input buffer */
  int pos;			/* current pos in buffer */
  int stream_pos;		/* when reached, a new block must be read */
  uint32_t code;
  uint32_t range;
  int infd;			/* input file descriptor */
  bool at_stream_end;
  };

bool Rd_read_block( struct Range_decoder * const rdec );

static inline bool Rd_init( struct Range_decoder * const rdec, const int ifd )
  {
  rdec->partial_member_pos = 0;
  rdec->buffer = (uint8_t *)malloc( rd_buffer_size );
  if( !rdec->buffer ) return false;
  rdec->pos = 0;
  rdec->stream_pos = 0;
  rdec->code = 0;
  rdec->range = 0xFFFFFFFFU;
  rdec->infd = ifd;
  rdec->at_stream_end = false;
  return true;
  }

static inline void Rd_free( struct Range_decoder * const rdec )
  { free( rdec->buffer ); }

static inline bool Rd_finished( struct Range_decoder * const rdec )
  { return rdec->pos >= rdec->stream_pos && !Rd_read_block( rdec ); }

static inline unsigned long long
Rd_member_position( const struct Range_decoder * const rdec )
  { return rdec->partial_member_pos + rdec->pos; }

static inline void Rd_reset_member_position( struct Range_decoder * const rdec )
  { rdec->partial_member_pos = -rdec->pos; }

static inline uint8_t Rd_get_byte( struct Range_decoder * const rdec )
  {
  if( Rd_finished( rdec ) ) return 0xAA;	/* make code != 0 */
  return rdec->buffer[rdec->pos++];
  }

static inline int Rd_read_data( struct Range_decoder * const rdec,
                                uint8_t * const outbuf, const int size )
  {
  int rest = size;
  while( rest > 0 && !Rd_finished( rdec ) )
    {
    const int rd = min( rest, rdec->stream_pos - rdec->pos );
    memcpy( outbuf + size - rest, rdec->buffer + rdec->pos, rd );
    rdec->pos += rd;
    rest -= rd;
    }
  return size - rest;
  }

static inline void Rd_load( struct Range_decoder * const rdec )
  {
  int i;
  rdec->code = 0;
  for( i = 0; i < 5; ++i )
    rdec->code = (rdec->code << 8) | Rd_get_byte( rdec );
  rdec->range = 0xFFFFFFFFU;
  }

static inline void Rd_normalize( struct Range_decoder * const rdec )
  {
  if( rdec->range <= 0x00FFFFFFU )
    {
    rdec->range <<= 8;
    rdec->code = (rdec->code << 8) | Rd_get_byte( rdec );
    }
  }

static inline int Rd_decode( struct Range_decoder * const rdec,
                             const int num_bits )
  {
  int symbol = 0;
  int i;
  for( i = num_bits; i > 0; --i )
    {
    uint32_t mask;
    Rd_normalize( rdec );
    rdec->range >>= 1;
/*    symbol <<= 1; */
/*    if( rdec->code >= rdec->range ) { rdec->code -= rdec->range; symbol |= 1; } */
    mask = 0U - (rdec->code < rdec->range);
    rdec->code -= rdec->range;
    rdec->code += rdec->range & mask;
    symbol = (symbol << 1) + (mask + 1);
    }
  return symbol;
  }

static inline int Rd_decode_bit( struct Range_decoder * const rdec,
                                 Bit_model * const probability )
  {
  uint32_t bound;
  Rd_normalize( rdec );
  bound = ( rdec->range >> bit_model_total_bits ) * *probability;
  if( rdec->code < bound )
    {
    rdec->range = bound;
    *probability += (bit_model_total - *probability) >> bit_model_move_bits;
    return 0;
    }
  else
    {
    rdec->range -= bound;
    rdec->code -= bound;
    *probability -= *probability >> bit_model_move_bits;
    return 1;
    }
  }

static inline int Rd_decode_tree( struct Range_decoder * const rdec,
                                  Bit_model bm[], const int num_bits )
  {
  int symbol = 1;
  int i;
  for( i = num_bits; i > 0; --i )
    symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  return symbol - (1 << num_bits);
  }

static inline int Rd_decode_tree6( struct Range_decoder * const rdec,
                                   Bit_model bm[] )
  {
  int symbol = 1;
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
  return symbol - (1 << 6);
  }

static inline int Rd_decode_tree_reversed( struct Range_decoder * const rdec,
                                           Bit_model bm[], const int num_bits )
  {
  int model = 1;
  int symbol = 0;
  int i;
  for( i = 0; i < num_bits; ++i )
    {
    const bool bit = Rd_decode_bit( rdec, &bm[model] );
    model <<= 1;
    if( bit ) { ++model; symbol |= (1 << i); }
    }
  return symbol;
  }

static inline int Rd_decode_tree_reversed4( struct Range_decoder * const rdec,
                                            Bit_model bm[] )
  {
  int model = 1;
  int symbol = 0;
  int bit = Rd_decode_bit( rdec, &bm[model] );
  model = (model << 1) + bit; symbol |= bit;
  bit = Rd_decode_bit( rdec, &bm[model] );
  model = (model << 1) + bit; symbol |= (bit << 1);
  bit = Rd_decode_bit( rdec, &bm[model] );
  model = (model << 1) + bit; symbol |= (bit << 2);
  if( Rd_decode_bit( rdec, &bm[model] ) ) symbol |= 8;
  return symbol;
  }

static inline int Rd_decode_matched( struct Range_decoder * const rdec,
                                     Bit_model bm[], int match_byte )
  {
  Bit_model * const bm1 = bm + 0x100;
  int symbol = 1;
  int i;
  for( i = 7; i >= 0; --i )
    {
    int match_bit, bit;
    match_byte <<= 1;
    match_bit = match_byte & 0x100;
    bit = Rd_decode_bit( rdec, &bm1[match_bit+symbol] );
    symbol = ( symbol << 1 ) | bit;
    if( match_bit != bit << 8 )
      {
      while( symbol < 0x100 )
        symbol = ( symbol << 1 ) | Rd_decode_bit( rdec, &bm[symbol] );
      break;
      }
    }
  return symbol - 0x100;
  }

static inline int Rd_decode_len( struct Range_decoder * const rdec,
                                 struct Len_model * const lm,
                                 const int pos_state )
  {
  if( Rd_decode_bit( rdec, &lm->choice1 ) == 0 )
    return Rd_decode_tree( rdec, lm->bm_low[pos_state], len_low_bits );
  if( Rd_decode_bit( rdec, &lm->choice2 ) == 0 )
    return len_low_symbols +
           Rd_decode_tree( rdec, lm->bm_mid[pos_state], len_mid_bits );
  return len_low_symbols + len_mid_symbols +
         Rd_decode_tree( rdec, lm->bm_high, len_high_bits );
  }


struct LZ_decoder
  {
  unsigned long long partial_data_pos;
  int dictionary_size;
  int buffer_size;
  uint8_t * buffer;		/* output buffer */
  int pos;			/* current pos in buffer */
  int stream_pos;		/* first byte not yet written to file */
  uint32_t crc;
  int outfd;			/* output file descriptor */

  Bit_model bm_literal[1<<literal_context_bits][0x300];
  Bit_model bm_match[states][pos_states];
  Bit_model bm_rep[states];
  Bit_model bm_rep0[states];
  Bit_model bm_rep1[states];
  Bit_model bm_rep2[states];
  Bit_model bm_len[states][pos_states];
  Bit_model bm_dis_slot[len_states][1<<dis_slot_bits];
  Bit_model bm_dis[modeled_distances-end_dis_model];
  Bit_model bm_align[dis_align_size];

  struct Range_decoder * rdec;
  struct Len_model match_len_model;
  struct Len_model rep_len_model;
  };

void LZd_flush_data( struct LZ_decoder * const decoder );

bool LZd_verify_trailer( struct LZ_decoder * const decoder,
                         struct Pretty_print * const pp );

static inline uint8_t LZd_get_prev_byte( const struct LZ_decoder * const decoder )
  {
  const int i =
    ( ( decoder->pos > 0 ) ? decoder->pos : decoder->buffer_size ) - 1;
  return decoder->buffer[i];
  }

static inline uint8_t LZd_get_byte( const struct LZ_decoder * const decoder,
                                    const int distance )
  {
  int i = decoder->pos - distance - 1;
  if( i < 0 ) i += decoder->buffer_size;
  return decoder->buffer[i];
  }

static inline void LZd_put_byte( struct LZ_decoder * const decoder,
                                 const uint8_t b )
  {
  decoder->buffer[decoder->pos] = b;
  if( ++decoder->pos >= decoder->buffer_size ) LZd_flush_data( decoder );
  }

static inline void LZd_copy_block( struct LZ_decoder * const decoder,
                                   const int distance, int len )
  {
  int i = decoder->pos - distance - 1;
  if( i < 0 ) i += decoder->buffer_size;
  if( len < decoder->buffer_size - max( decoder->pos, i ) &&
      len <= abs( decoder->pos - i ) )		/* no wrap, no overlap */
    {
    memcpy( decoder->buffer + decoder->pos, decoder->buffer + i, len );
    decoder->pos += len;
    }
  else for( ; len > 0; --len )
    {
    decoder->buffer[decoder->pos] = decoder->buffer[i];
    if( ++decoder->pos >= decoder->buffer_size ) LZd_flush_data( decoder );
    if( ++i >= decoder->buffer_size ) i = 0;
    }
  }

static inline bool LZd_init( struct LZ_decoder * const decoder,
                             const File_header header,
                             struct Range_decoder * const rde, const int ofd )
  {
  decoder->partial_data_pos = 0;
  decoder->dictionary_size = Fh_get_dictionary_size( header );
  decoder->buffer_size = max( 65536, decoder->dictionary_size );
  decoder->buffer = (uint8_t *)malloc( decoder->buffer_size );
  if( !decoder->buffer ) return false;
  decoder->pos = 0;
  decoder->stream_pos = 0;
  decoder->crc = 0xFFFFFFFFU;
  decoder->outfd = ofd;

  Bm_array_init( decoder->bm_literal[0], (1 << literal_context_bits) * 0x300 );
  Bm_array_init( decoder->bm_match[0], states * pos_states );
  Bm_array_init( decoder->bm_rep, states );
  Bm_array_init( decoder->bm_rep0, states );
  Bm_array_init( decoder->bm_rep1, states );
  Bm_array_init( decoder->bm_rep2, states );
  Bm_array_init( decoder->bm_len[0], states * pos_states );
  Bm_array_init( decoder->bm_dis_slot[0], len_states * (1 << dis_slot_bits) );
  Bm_array_init( decoder->bm_dis, modeled_distances - end_dis_model );
  Bm_array_init( decoder->bm_align, dis_align_size );

  decoder->rdec = rde;
  Lm_init( &decoder->match_len_model );
  Lm_init( &decoder->rep_len_model );
  decoder->buffer[decoder->buffer_size-1] = 0;	/* prev_byte of first_byte */
  return true;
  }

static inline void LZd_free( struct LZ_decoder * const decoder )
  { free( decoder->buffer ); }

static inline unsigned LZd_crc( const struct LZ_decoder * const decoder )
  { return decoder->crc ^ 0xFFFFFFFFU; }

static inline unsigned long long
LZd_data_position( const struct LZ_decoder * const decoder )
  { return decoder->partial_data_pos + decoder->pos; }

int LZd_decode_member( struct LZ_decoder * const decoder,
                       struct Pretty_print * const pp );
