/*  Lunzip - Decompressor for lzip files
    Copyright (C) 2010, 2011, 2012, 2013 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/*
    Exit status: 0 for a normal exit, 1 for environmental problems
    (file not found, invalid flags, I/O errors, etc), 2 to indicate a
    corrupt or invalid input file, 3 for an internal consistency error
    (eg, bug) which caused lunzip to panic.
*/

#define _FILE_OFFSET_BITS 64

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <utime.h>
#include <sys/stat.h>
#if defined(__MSVCRT__)
#include <io.h>
#define fchmod(x,y) 0
#define fchown(x,y,z) 0
#define SIGHUP SIGTERM
#define S_ISSOCK(x) 0
#define S_IRGRP 0
#define S_IWGRP 0
#define S_IROTH 0
#define S_IWOTH 0
#endif
#if defined(__OS2__)
#include <io.h>
#endif

#include "carg_parser.h"
#include "lzip.h"
#include "decoder.h"

#if CHAR_BIT != 8
#error "Environments where CHAR_BIT != 8 are not supported."
#endif


const char * const Program_name = "Lunzip";
const char * const program_name = "lunzip";
const char * const program_year = "2013";
const char * invocation_name = 0;

#ifdef O_BINARY
const int o_binary = O_BINARY;
#else
const int o_binary = 0;
#endif

struct { const char * from; const char * to; } const known_extensions[] = {
  { ".lz",  ""     },
  { ".tlz", ".tar" },
  { 0,      0      } };

char * output_filename = 0;
int outfd = -1;
int verbosity = 0;
const mode_t usr_rw = S_IRUSR | S_IWUSR;
const mode_t all_rw = S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH;
mode_t outfd_mode = S_IRUSR | S_IWUSR;
bool delete_output_on_interrupt = false;


static void show_help( void )
  {
  printf( "%s - Decompressor for lzip files.\n", Program_name );
  printf( "\nUsage: %s [options] [files]\n", invocation_name );
  printf( "\nOptions:\n"
          "  -h, --help                 display this help and exit\n"
          "  -V, --version              output version information and exit\n"
          "  -c, --stdout               send output to standard output\n"
          "  -d, --decompress           decompress (this is the default)\n"
          "  -f, --force                overwrite existing output files\n"
          "  -k, --keep                 keep (don't delete) input files\n"
          "  -o, --output=<file>        if reading stdin, place the output into <file>\n"
          "  -q, --quiet                suppress all messages\n"
          "  -t, --test                 test compressed file integrity\n"
          "  -v, --verbose              be verbose (a 2nd -v gives more)\n"
          "If no file names are given, lunzip decompresses from standard input to\n"
          "standard output.\n"
          "\nExit status: 0 for a normal exit, 1 for environmental problems (file\n"
          "not found, invalid flags, I/O errors, etc), 2 to indicate a corrupt or\n"
          "invalid input file, 3 for an internal consistency error (eg, bug) which\n"
          "caused lunzip to panic.\n"
          "\nReport bugs to lzip-bug@nongnu.org\n"
          "Lunzip home page: http://www.nongnu.org/lzip/lunzip.html\n" );
  }


static void show_version( void )
  {
  printf( "%s %s\n", Program_name, PROGVERSION );
  printf( "Copyright (C) %s Antonio Diaz Diaz.\n", program_year );
  printf( "License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>\n"
          "This is free software: you are free to change and redistribute it.\n"
          "There is NO WARRANTY, to the extent permitted by law.\n" );
  }


static void show_header( const File_header header )
  {
  const char * const prefix[8] =
    { "Ki", "Mi", "Gi", "Ti", "Pi", "Ei", "Zi", "Yi" };
  enum { factor = 1024 };
  const char * p = "";
  const char * np = "  ";
  unsigned num = Fh_get_dictionary_size( header ), i;
  bool exact = ( num % factor == 0 );

  for( i = 0; i < 8 && ( num > 9999 || ( exact && num >= factor ) ); ++i )
    { num /= factor; if( num % factor != 0 ) exact = false;
      p = prefix[i]; np = ""; }
  fprintf( stderr, "dictionary size %s%4u %sB.  ", np, num, p );
  }


static int extension_index( const char * const name )
  {
  int i;
  for( i = 0; known_extensions[i].from; ++i )
    {
    const char * const ext = known_extensions[i].from;
    if( strlen( name ) > strlen( ext ) &&
        strncmp( name + strlen( name ) - strlen( ext ), ext, strlen( ext ) ) == 0 )
      return i;
    }
  return -1;
  }


static int open_instream( const char * const name, struct stat * const in_statsp,
                          const bool testing, const bool to_stdout )
  {
  int infd = open( name, O_RDONLY | o_binary );
  if( infd < 0 )
    {
    if( verbosity >= 0 )
      fprintf( stderr, "%s: Can't open input file '%s': %s.\n",
               program_name, name, strerror( errno ) );
    }
  else
    {
    const int i = fstat( infd, in_statsp );
    const mode_t mode = in_statsp->st_mode;
    const bool can_read = ( i == 0 &&
                            ( S_ISBLK( mode ) || S_ISCHR( mode ) ||
                              S_ISFIFO( mode ) || S_ISSOCK( mode ) ) );
    const bool no_ofile = to_stdout || testing;
    if( i != 0 || ( !S_ISREG( mode ) && ( !can_read || !no_ofile ) ) )
      {
      if( verbosity >= 0 )
        fprintf( stderr, "%s: Input file '%s' is not a regular file%s.\n",
                 program_name, name,
                 ( can_read && !no_ofile ) ?
                 " and '--stdout' was not specified" : "" );
      close( infd );
      infd = -1;
      }
    }
  return infd;
  }


/* assure at least a minimum size for buffer 'buf' */
static void * resize_buffer( void * buf, const int min_size )
  {
  if( buf ) buf = realloc( buf, min_size );
  else buf = malloc( min_size );
  if( !buf )
    {
    show_error( "Not enough memory.", 0, false );
    cleanup_and_fail( 1 );
    }
  return buf;
  }


static void set_d_outname( const char * const name, const int i )
  {
  if( i >= 0 )
    {
    const char * const from = known_extensions[i].from;
    if( strlen( name ) > strlen( from ) )
      {
      output_filename = resize_buffer( output_filename, strlen( name ) +
                                       strlen( known_extensions[0].to ) + 1 );
      strcpy( output_filename, name );
      strcpy( output_filename + strlen( name ) - strlen( from ),
              known_extensions[i].to );
      return;
      }
    }
  output_filename = resize_buffer( output_filename, strlen( name ) + 4 + 1 );
  strcpy( output_filename, name );
  strcat( output_filename, ".out" );
  if( verbosity >= 1 )
    fprintf( stderr, "%s: Can't guess original name for '%s' -- using '%s'.\n",
             program_name, name, output_filename );
  }


static bool open_outstream( const bool force )
  {
  int flags = O_CREAT | O_WRONLY | o_binary;
  if( force ) flags |= O_TRUNC; else flags |= O_EXCL;

  outfd = open( output_filename, flags, outfd_mode );
  if( outfd < 0 && verbosity >= 0 )
    {
    if( errno == EEXIST )
      fprintf( stderr, "%s: Output file '%s' already exists, skipping.\n",
               program_name, output_filename );
    else
      fprintf( stderr, "%s: Can't create output file '%s': %s.\n",
               program_name, output_filename, strerror( errno ) );
    }
  return ( outfd >= 0 );
  }


void cleanup_and_fail( const int retval )
  {
  if( delete_output_on_interrupt )
    {
    delete_output_on_interrupt = false;
    if( verbosity >= 0 )
      fprintf( stderr, "%s: Deleting output file '%s', if it exists.\n",
               program_name, output_filename );
    if( outfd >= 0 ) { close( outfd ); outfd = -1; }
    if( remove( output_filename ) != 0 && errno != ENOENT )
      show_error( "WARNING: deletion of output file (apparently) failed.", 0, false );
    }
  exit( retval );
  }


     /* Set permissions, owner and times. */
static void close_and_set_permissions( const struct stat * const in_statsp )
  {
  bool warning = false;
  if( in_statsp )
    {
    /* fchown will in many cases return with EPERM, which can be safely ignored. */
    if( ( fchown( outfd, in_statsp->st_uid, in_statsp->st_gid ) != 0 &&
          errno != EPERM ) ||
        fchmod( outfd, in_statsp->st_mode ) != 0 ) warning = true;
    }
  if( close( outfd ) != 0 ) cleanup_and_fail( 1 );
  outfd = -1;
  delete_output_on_interrupt = false;
  if( in_statsp )
    {
    struct utimbuf t;
    t.actime = in_statsp->st_atime;
    t.modtime = in_statsp->st_mtime;
    if( utime( output_filename, &t ) != 0 ) warning = true;
    }
  if( warning && verbosity >= 1 )
    show_error( "Can't change output file attributes.", 0, false );
  }


static int decompress( const int infd, struct Pretty_print * const pp,
                       const bool testing )
  {
  unsigned long long partial_file_pos = 0;
  struct Range_decoder rdec;
  int retval = 0;
  bool first_member;
  if( !Rd_init( &rdec, infd ) )
    {
    show_error( "Not enough memory. Find a machine with more memory.", 0, false );
    cleanup_and_fail( 1 );
    }

  for( first_member = true; ; first_member = false )
    {
    int result;
    File_header header;
    struct LZ_decoder decoder;
    Rd_reset_member_position( &rdec );
    Rd_read_data( &rdec, header, Fh_size );
    if( Rd_finished( &rdec ) )			/* End Of File */
      {
      if( first_member )
        { Pp_show_msg( pp, "File ends unexpectedly at member header" );
          retval = 2; }
      break;
      }
    if( !Fh_verify_magic( header ) )
      {
      if( !first_member ) break;		/* trailing garbage */
      Pp_show_msg( pp, "Bad magic number (file not in lzip format)" );
      retval = 2; break;
      }
    if( !Fh_verify_version( header ) )
      {
      if( verbosity >= 0 )
        { Pp_show_msg( pp, 0 );
          fprintf( stderr, "Version %d member format not supported.\n",
                   Fh_version( header ) ); }
      retval = 2; break;
      }
    if( Fh_get_dictionary_size( header ) < min_dictionary_size ||
        Fh_get_dictionary_size( header ) > max_dictionary_size )
      { Pp_show_msg( pp, "Invalid dictionary size in member header" );
        retval = 2; break; }

    if( verbosity >= 2 || ( verbosity == 1 && first_member ) )
      { Pp_show_msg( pp, 0 ); if( verbosity >= 3 ) show_header( header ); }

    if( !LZd_init( &decoder, header, &rdec, outfd ) )
      {
      show_error( "Not enough memory. Find a machine with more memory.", 0, false );
      cleanup_and_fail( 1 );
      }
    result = LZd_decode_member( &decoder, pp );
    partial_file_pos += Rd_member_position( &rdec );
    LZd_free( &decoder );
    if( result != 0 )
      {
      if( verbosity >= 0 && result <= 2 )
        {
        Pp_show_msg( pp, 0 );
        if( result == 2 )
          fprintf( stderr, "File ends unexpectedly at pos %llu\n",
                   partial_file_pos );
        else
          fprintf( stderr, "Decoder error at pos %llu\n", partial_file_pos );
        }
      retval = 2; break;
      }
    if( verbosity >= 2 )
      { fprintf( stderr, testing ? "ok\n" : "done\n" ); Pp_reset( pp ); }
    }
  Rd_free( &rdec );
  if( verbosity == 1 && retval == 0 )
    fprintf( stderr, testing ? "ok\n" : "done\n" );
  return retval;
  }


void signal_handler( int sig )
  {
  if( sig ) {}				/* keep compiler happy */
  show_error( "Control-C or similar caught, quitting.", 0, false );
  cleanup_and_fail( 1 );
  }


static void set_signals( void )
  {
  signal( SIGHUP, signal_handler );
  signal( SIGINT, signal_handler );
  signal( SIGTERM, signal_handler );
  }


void Pp_init( struct Pretty_print * const pp, const char * const filenames[],
              const int num_filenames )
  {
  unsigned stdin_name_len;
  int i;
  pp->name = 0;
  pp->stdin_name = "(stdin)";
  pp->longest_name = 0;
  pp->first_post = false;
  stdin_name_len = strlen( pp->stdin_name );

  for( i = 0; i < num_filenames; ++i )
    {
    const char * const s = filenames[i];
    const int len = ( (strcmp( s, "-" ) == 0) ? stdin_name_len : strlen( s ) );
    if( len > pp->longest_name ) pp->longest_name = len;
    }
  if( pp->longest_name == 0 ) pp->longest_name = stdin_name_len;
  }


void show_error( const char * const msg, const int errcode, const bool help )
  {
  if( verbosity >= 0 )
    {
    if( msg && msg[0] )
      {
      fprintf( stderr, "%s: %s", program_name, msg );
      if( errcode > 0 ) fprintf( stderr, ": %s", strerror( errcode ) );
      fprintf( stderr, "\n" );
      }
    if( help )
      fprintf( stderr, "Try '%s --help' for more information.\n",
               invocation_name );
    }
  }


void internal_error( const char * const msg )
  {
  if( verbosity >= 0 )
    fprintf( stderr, "%s: internal error: %s.\n", program_name, msg );
  exit( 3 );
  }


int main( const int argc, const char * const argv[] )
  {
  const char * input_filename = "";
  const char * default_output_filename = "";
  const char ** filenames = 0;
  int num_filenames = 0;
  int infd = -1;
  int argind = 0;
  int retval = 0;
  int i;
  bool filenames_given = false;
  bool force = false;
  bool keep_input_files = false;
  bool testing = false;
  bool to_stdout = false;
  struct Pretty_print pp;

  const struct ap_Option options[] =
    {
    { 'c', "stdout",          ap_no  },
    { 'd', "decompress",      ap_no  },
    { 'f', "force",           ap_no  },
    { 'h', "help",            ap_no  },
    { 'k', "keep",            ap_no  },
    { 'n', "threads",         ap_yes },
    { 'o', "output",          ap_yes },
    { 'q', "quiet",           ap_no  },
    { 't', "test",            ap_no  },
    { 'v', "verbose",         ap_no  },
    { 'V', "version",         ap_no  },
    {  0 ,  0,                ap_no  } };

  struct Arg_parser parser;

  invocation_name = argv[0];
  CRC32_init();

  if( !ap_init( &parser, argc, argv, options, 0 ) )
    { show_error( "Memory exhausted.", 0, false ); return 1; }
  if( ap_error( &parser ) )				/* bad option */
    { show_error( ap_error( &parser ), 0, true ); return 1; }

  for( ; argind < ap_arguments( &parser ); ++argind )
    {
    const int code = ap_code( &parser, argind );
    const char * const arg = ap_argument( &parser, argind );
    if( !code ) break;					/* no more options */
    switch( code )
      {
      case 'c': to_stdout = true; break;
      case 'd': testing = false; break;
      case 'f': force = true; break;
      case 'h': show_help(); return 0;
      case 'k': keep_input_files = true; break;
      case 'n': break;
      case 'o': default_output_filename = arg; break;
      case 'q': verbosity = -1; break;
      case 't': testing = true; break;
      case 'v': if( verbosity < 4 ) ++verbosity; break;
      case 'V': show_version(); return 0;
      default : internal_error( "uncaught option" );
      }
    } /* end process options */

#if defined(__MSVCRT__) || defined(__OS2__)
  setmode( STDIN_FILENO, O_BINARY );
  setmode( STDOUT_FILENO, O_BINARY );
#endif

  if( testing )
    outfd = -1;

  num_filenames = max( 1, ap_arguments( &parser ) - argind );
  filenames = resize_buffer( filenames, num_filenames * sizeof filenames[0] );
  filenames[0] = "-";

  for( i = 0; argind + i < ap_arguments( &parser ); ++i )
    {
    filenames[i] = ap_argument( &parser, argind + i );
    if( strcmp( filenames[i], "-" ) != 0 ) filenames_given = true;
    }

  if( !to_stdout && !testing &&
      ( filenames_given || default_output_filename[0] ) )
    set_signals();

  Pp_init( &pp, filenames, num_filenames );

  output_filename = resize_buffer( output_filename, 1 );
  for( i = 0; i < num_filenames; ++i )
    {
    int tmp;
    struct stat in_stats;
    const struct stat * in_statsp;
    output_filename[0] = 0;

    if( !filenames[i][0] || strcmp( filenames[i], "-" ) == 0 )
      {
      input_filename = "";
      infd = STDIN_FILENO;
      if( !testing )
        {
        if( to_stdout || !default_output_filename[0] )
          outfd = STDOUT_FILENO;
        else
          {
          output_filename = resize_buffer( output_filename,
                                           strlen( default_output_filename ) + 1 );
          strcpy( output_filename, default_output_filename );
          outfd_mode = all_rw;
          if( !open_outstream( force ) )
            {
            if( retval < 1 ) retval = 1;
            close( infd ); infd = -1;
            continue;
            }
          }
        }
      }
    else
      {
      input_filename = filenames[i];
      infd = open_instream( input_filename, &in_stats, testing, to_stdout );
      if( infd < 0 ) { if( retval < 1 ) retval = 1; continue; }
      if( !testing )
        {
        if( to_stdout ) outfd = STDOUT_FILENO;
        else
          {
          set_d_outname( input_filename, extension_index( input_filename ) );
          outfd_mode = usr_rw;
          if( !open_outstream( force ) )
            {
            if( retval < 1 ) retval = 1;
            close( infd ); infd = -1;
            continue;
            }
          }
        }
      }

    if( isatty( infd ) )
      {
      show_error( "I won't read compressed data from a terminal.", 0, true );
      return 1;
      }

    if( output_filename[0] && !to_stdout && !testing )
      delete_output_on_interrupt = true;
    in_statsp = input_filename[0] ? &in_stats : 0;
    Pp_set_name( &pp, input_filename );
    tmp = decompress( infd, &pp, testing );
    if( tmp > retval ) retval = tmp;
    if( tmp && !testing ) cleanup_and_fail( retval );

    if( delete_output_on_interrupt )
      close_and_set_permissions( in_statsp );
    if( input_filename[0] )
      {
      close( infd ); infd = -1;
      if( !keep_input_files && !to_stdout && !testing )
        remove( input_filename );
      }
    }
  if( outfd >= 0 && close( outfd ) != 0 )
    {
    show_error( "Can't close stdout", errno, false );
    if( retval < 1 ) retval = 1;
    }
  free( output_filename );
  free( filenames );
  ap_free( &parser );
  return retval;
  }
