/*  Tarlz - Archiver with multimember lzip compression
    Copyright (C) 2013-2018 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

enum { header_size = 512 };
typedef char Tar_header[header_size];

enum Offsets {
  name_o = 0, mode_o = 100, uid_o = 108, gid_o = 116, size_o = 124,
  mtime_o = 136, chksum_o = 148, typeflag_o = 156, linkname_o = 157,
  magic_o = 257, version_o = 263, uname_o = 265, gname_o = 297,
  devmajor_o = 329, devminor_o = 337, prefix_o = 345 };

enum Lengths {
  name_l = 100, mode_l = 8, uid_l = 8, gid_l = 8, size_l = 12,
  mtime_l = 12, chksum_l = 8, typeflag_l = 1, linkname_l = 100,
  magic_l = 6, version_l = 2, uname_l = 32, gname_l = 32,
  devmajor_l = 8, devminor_l = 8, prefix_l = 155 };

enum Typeflag {
  tf_regular = '0', tf_link = '1', tf_symlink = '2', tf_chardev = '3',
  tf_blockdev = '4', tf_directory = '5', tf_fifo = '6', tf_hiperf = '7',
  tf_extended = 'x' };

const uint8_t ustar_magic[magic_l] =
  { 0x75, 0x73, 0x74, 0x61, 0x72, 0 };			// "ustar\0"

inline bool verify_ustar_magic( const uint8_t * const buf )
  { return std::memcmp( buf + magic_o, ustar_magic, magic_l ) == 0; }


class CRC32C			// Uses CRC32-C (Castagnoli) polynomial.
  {
  uint32_t data[256];		// Table of CRCs of all 8-bit messages.

public:
  CRC32C()
    {
    for( unsigned n = 0; n < 256; ++n )
      {
      unsigned c = n;
      for( int k = 0; k < 8; ++k )
        { if( c & 1 ) c = 0x82F63B78U ^ ( c >> 1 ); else c >>= 1; }
      data[n] = c;
      }
    }

  void update_buf( uint32_t & crc, const uint8_t * const buffer,
                   const int size ) const
    {
    uint32_t c = crc;
    for( int i = 0; i < size; ++i )
      c = data[(c^buffer[i])&0xFF] ^ ( c >> 8 );
    crc = c;
    }

  // Calculates the crc of size bytes except a window of 8 bytes at pos
  uint32_t windowed_crc( const uint8_t * const buffer, const int pos,
                         const int size ) const
    {
    uint32_t crc = 0xFFFFFFFFU;
    update_buf( crc, buffer, pos );
    update_buf( crc, buffer + pos + 8, size - pos - 8 );
    return crc ^ 0xFFFFFFFFU;
    }
  };

extern const CRC32C crc32c;


// Round "size" to the next multiple of header size (512).
//
inline unsigned long long round_up( unsigned long long size )
  {
  const int rem = size % header_size;
  const int padding = rem ? header_size - rem : 0;
  return size + padding;
  }


struct Extended		// stores metadata from/for extended records
  {
  std::string linkpath;
  std::string path;
  unsigned long long size;
  bool crc_present;
  Extended() : size( 0 ), crc_present( false ) {}
  void reset()
    { linkpath.clear(); path.clear(); size = 0; crc_present = false; }
  bool empty() { return linkpath.empty() && path.empty() && size == 0; }
  bool parse( const int infd, const Tar_header header, const bool permissive );
  };

// defined in create.cc
extern int cl_owner;
extern int cl_group;
extern int cl_solid;
unsigned ustar_chksum( const uint8_t * const buf );
bool verify_ustar_chksum( const uint8_t * const buf );
class Arg_parser;
int concatenate( const std::string & archive_name, const Arg_parser & parser,
                 const int filenames );
int encode( const std::string & archive_name, const Arg_parser & parser,
            const int filenames, const int level, const bool append );

// defined in extract.cc
int decode( const std::string & archive_name, const Arg_parser & parser,
            const int filenames, const bool keep_damaged, const bool listing,
            const bool missing_crc, const bool permissive );

// defined in main.cc
extern int verbosity;
int open_instream( const std::string & name );
int open_outstream( const std::string & name, const bool create = true );
int readblock( const int fd, uint8_t * const buf, const int size );
int writeblock( const int fd, const uint8_t * const buf, const int size );
void show_error( const char * const msg, const int errcode = 0,
                 const bool help = false );
void show_file_error( const char * const filename, const char * const msg,
                      const int errcode = 0 );
void internal_error( const char * const msg );
