/*  Xlunzip - Test tool for the lunzip linux module
    Copyright (C) 2016-2018 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef max
  #define max(x,y) ((x) >= (y) ? (x) : (y))
#endif
#ifndef min
  #define min(x,y) ((x) <= (y) ? (x) : (y))
#endif

void * resize_buffer( void * buf, const unsigned min_size );

enum {
  min_dictionary_bits = 12,
  min_dictionary_size = 1 << min_dictionary_bits,
  max_dictionary_bits = 29,
  max_dictionary_size = 1 << max_dictionary_bits,
  min_member_size = 36 };


/* defined in main.c */
extern int verbosity;

struct Pretty_print
  {
  const char * name;
  char * padded_name;
  const char * stdin_name;
  unsigned longest_name;
  bool first_post;
  };

static inline void Pp_init( struct Pretty_print * const pp,
                            const char * const filenames[],
                            const int num_filenames )
  {
  unsigned stdin_name_len;
  int i;
  pp->name = 0;
  pp->padded_name = 0;
  pp->stdin_name = "(stdin)";
  pp->longest_name = 0;
  pp->first_post = false;

  if( verbosity <= 0 ) return;
  stdin_name_len = strlen( pp->stdin_name );
  for( i = 0; i < num_filenames; ++i )
    {
    const char * const s = filenames[i];
    const unsigned len = (strcmp( s, "-" ) == 0) ? stdin_name_len : strlen( s );
    if( len > pp->longest_name ) pp->longest_name = len;
    }
  if( pp->longest_name == 0 ) pp->longest_name = stdin_name_len;
  }

static inline void Pp_set_name( struct Pretty_print * const pp,
                                const char * const filename )
  {
  unsigned name_len, padded_name_len, i = 0;

  if( filename && filename[0] && strcmp( filename, "-" ) != 0 )
    pp->name = filename;
  else pp->name = pp->stdin_name;
  name_len = strlen( pp->name );
  padded_name_len = max( name_len, pp->longest_name ) + 4;
  pp->padded_name = resize_buffer( pp->padded_name, padded_name_len + 1 );
  while( i < 2 ) pp->padded_name[i++] = ' ';
  while( i < name_len + 2 ) { pp->padded_name[i] = pp->name[i-2]; ++i; }
  pp->padded_name[i++] = ':';
  while( i < padded_name_len ) pp->padded_name[i++] = ' ';
  pp->padded_name[i] = 0;
  pp->first_post = true;
  }

static inline void Pp_reset( struct Pretty_print * const pp )
  { if( pp->name && pp->name[0] ) pp->first_post = true; }
void Pp_show_msg( struct Pretty_print * const pp, const char * const msg );


static const uint8_t lzip_magic[4] = { 0x4C, 0x5A, 0x49, 0x50 }; /* "LZIP" */

typedef uint8_t Lzip_header[6];		/* 0-3 magic bytes */
					/*   4 version */
					/*   5 coded_dict_size */
enum { Lh_size = 6 };

static inline bool Lh_verify_magic( const Lzip_header data )
  { return ( memcmp( data, lzip_magic, 4 ) == 0 ); }

/* detect (truncated) header */
static inline bool Lh_verify_prefix( const Lzip_header data, const int sz )
  {
  int i; for( i = 0; i < sz && i < 4; ++i )
    if( data[i] != lzip_magic[i] ) return false;
  return ( sz > 0 );
  }

/* detect corrupt header */
static inline bool Lh_verify_corrupt( const Lzip_header data )
  {
  int matches = 0;
  int i; for( i = 0; i < 4; ++i )
    if( data[i] == lzip_magic[i] ) ++matches;
  return ( matches > 1 && matches < 4 );
  }

static inline uint8_t Lh_version( const Lzip_header data )
  { return data[4]; }

static inline bool Lh_verify_version( const Lzip_header data )
  { return ( data[4] == 1 ); }

static inline unsigned Lh_get_dictionary_size( const Lzip_header data )
  {
  unsigned sz = ( 1 << ( data[5] & 0x1F ) );
  if( sz > min_dictionary_size )
    sz -= ( sz / 16 ) * ( ( data[5] >> 5 ) & 7 );
  return sz;
  }


typedef uint8_t Lzip_trailer[20];
			/*  0-3  CRC32 of the uncompressed data */
			/*  4-11 size of the uncompressed data */
			/* 12-19 member size including header and trailer */
enum { Lt_size = 20 };

static inline unsigned Lt_get_data_crc( const Lzip_trailer data )
  {
  unsigned tmp = 0;
  int i; for( i = 3; i >= 0; --i ) { tmp <<= 8; tmp += data[i]; }
  return tmp;
  }

static inline unsigned long long Lt_get_data_size( const Lzip_trailer data )
  {
  unsigned long long tmp = 0;
  int i; for( i = 11; i >= 4; --i ) { tmp <<= 8; tmp += data[i]; }
  return tmp;
  }

static inline unsigned long long Lt_get_member_size( const Lzip_trailer data )
  {
  unsigned long long tmp = 0;
  int i; for( i = 19; i >= 12; --i ) { tmp <<= 8; tmp += data[i]; }
  return tmp;
  }


/* defined in in_place.c */
int decompress_in_place( const int infd, struct Pretty_print * const pp,
                         const bool testing );

/* defined in main.c */
int convert_retval( const int retval );
long flush( void * buf, unsigned long size );
void show_error( const char * const msg, const int errcode, const bool help );
void show_file_error( const char * const filename, const char * const msg,
                      const int errcode );
