/************************************************************************\
 * Magic Square solves magic squares.                                   *
 * Copyright (C) 2019  Asher Gordon <AsDaGo@posteo.net>                 *
 *                                                                      *
 * This file is part of Magic Square.                                   *
 *                                                                      *
 * Magic Square is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * Magic Square is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with Magic Square.  If not, see                                *
 * <https://www.gnu.org/licenses/>.                                     *
\************************************************************************/

/* main.c -- main source file */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <getopt.h>

#include "square.h"
#include "parse.h"
#include "write.h"

int square_main(int, char **);
int squarec_main(int, char **);

/* Call the correct main() function based on the name we were run as */
int main(int argc, char **argv) {
  char *progname = strdup(argv[0]); /* Make a copy since basename(3)
				       can modify its argument */
  char *bname = basename(progname);
  enum {SQUARE, SQUAREC} mode =
			   strcasecmp(bname, "squarec") ? SQUARE : SQUAREC;

  free(progname);

  return (mode == SQUARE) ?
    square_main(argc, argv) :
    (mode == SQUAREC) ?
    squarec_main(argc, argv) :
    -1;
}

int square_main(int argc, char **argv) {
  square_t square;
  FILE *file;
  char keep_going = 0;
  enum {text, binary} mode = text;

  /* Parse the options */
  while (1) {
    int c;
    static struct option long_options[] = {
      {"keep-going", no_argument,       NULL, 'k'},
      {"mode",       required_argument, NULL, 'm'},
      {"help",       no_argument,       NULL, 'h'},
      {NULL,         0,                 NULL,  0 }
    };

    if ((c = getopt_long(argc, argv, "km:h", long_options, NULL)) == -1)
      break;

    switch (c) {
    case 'h':
      printf("Usage: %s [OPTIONS] [FILE]\n"
	     "Parse FILE and output it's data.\n"
	     "\n"
	     "With no FILE, or when FILE is -, read standard input.\n"
	     "\n"
	     "  -k, --keep-going   keep going after the first solution\n"
	     "  -m, --mode=MODE    parse FILE as MODE; mode can be "
	     "\"text\" or \"binary\"\n"
	     "  -h, --help         display this help and exit\n", argv[0]);

      return 0;
    case 'k':
      keep_going = 1;
      break;
    case 'm':
      if (!strncasecmp(optarg, "text", strlen(optarg))) {
	mode = text;
      }
      else if (!strncasecmp(optarg, "binary", strlen(optarg))) {
	mode = binary;
      }
      else {
	fprintf(stderr, "%s: invalid mode \"%s\"\n", argv[0], optarg);
	return 1;
      }

      break;
    case '?':
      return 1;
    default:
      fprintf(stderr,
	      "%1$s: error: getopt_long(3) returned `%2$c' (ASCII %2$hhd)\n",
	      argv[0], c);

      return -1;
    }
  }

  /* Shift the args to get rid of the options. */
  argv[optind - 1] = argv[0];
  argv += optind - 1;
  argc -= optind - 1;

  if (argc > 2) {
    fprintf(stderr,
	    "%s: wrong number of positional arguments "
	    "(got %d, expected 0 or 1)\n",
	    argv[0], argc - 1);

    return 1;
  }

  if (argc == 2 && strcmp(argv[1], "-")) {
    if (!(file = fopen(argv[1], mode == text ? "r" : "rb"))) {
      fprintf(stderr, "%s: cannot open %s: %m\n", argv[0], argv[1]);
      return 2;
    }
  }
  else {
    file = stdin;
  }

  if ((mode == text) ?
      parse_human(&square, file) :
      parse_machine(&square, file)) {
    fprintf(stderr, "%s: parse error: %m\n", argv[0]);
    return 2;
  }

  fclose(file);

  /* Solve the square */
  if (square_solve(&square, NULL, NULL, keep_going, stdout)) {
    fprintf(stderr, "%s: unable to solve square: %m\n", argv[0]);
    return 1;
  }

  /* We're done with this */
  square_destroy(&square);

  return 0;
}

int squarec_main(int argc, char **argv) {
  square_t square;
  FILE *infile, *outfile;
  char *infilename, *outfilename;

  if (argc < 2 || !strcmp(argv[1], "-")) {
    infile = stdin;
    infilename = "<stdin>";
  }
  /* Open the input file */
  else {
    infilename = argv[1];

    if (!(infile = fopen(infilename, "r"))) {
      fprintf(stderr, "%s: cannot open %s: %m\n", argv[0], infilename);
      return 2;
    }
  }

  if (argc < 3 || !strcmp(argv[2], "-")) {
    outfile = stdout;
    outfilename = "<stdout>";
  }
  /* Open the output file */
  else {
    outfilename = argv[2];

    if (!(outfile = fopen(outfilename, "wb"))) {
      fprintf(stderr, "%s: cannot open %s: %m\n", argv[0], outfilename);
      return 2;
    }
  }

  /* Now parse the square */
  if (parse_human(&square, infile)) {
    fprintf(stderr, "%s: could not parse %s: %m\n", argv[0], infilename);
    return 3;
  }

  fclose(infile);

  /* And write the output file */
  if (!write_machine(&square, outfile)) {
    fprintf(stderr, "%s: could not write %s: %m\n", argv[0], outfilename);

    return 3;
  }

  fclose(outfile);

  square_destroy(&square);
  return 0;
}
