/************************************************************************\
 * Magic Square solves magic squares.                                   *
 * Copyright (C) 2019  Asher Gordon <AsDaGo@posteo.net>                 *
 *                                                                      *
 * This file is part of Magic Square.                                   *
 *                                                                      *
 * Magic Square is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * Magic Square is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with Magic Square.  If not, see                                *
 * <https://www.gnu.org/licenses/>.                                     *
\************************************************************************/

/* square.c -- functions for operating on magic squares */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <assert.h>
#ifdef HAVE_PTHREAD
# include <pthread.h>
#endif

#include "square.h"
#include "write.h"

#ifdef HAVE_PTHREAD
/* A structure containing all the arguments to pass to
   square_solve_recursive(). */
struct solve_args {
  square_t *square;
  cellval_t sum;
  square_list_t **solutions;
  size_t *solutions_size;
  char keep_going;
  char skip_trivial;
  FILE *file;
};

/* A thread-independent errno */
static int global_errno = 0;
/* Mutexes for square_solve_recursive() */
static pthread_mutex_t solutions_mutex,
  threads_left_mutex, global_errno_mutex;
/* Number of threads left available */
static unsigned long threads_left;
#endif /* HAVE_PTHREAD */

/* Static helper functions */
static cellval_t add_cellval(cellval_t, cellval_t);
static int square_solve_recursive(square_t *, cellval_t, square_list_t **,
				  size_t *, char, char, FILE *);
#ifdef HAVE_PTHREAD
static int square_solve_new_thread(square_t *, cellval_t, square_list_t **,
				   size_t *, char, char, FILE *);
static void * square_solve_unpack_args(void *);
#endif
static int square_valid(square_t *, cellval_t);
static int square_trivially_equiv(const square_t *, const square_t *);

/* This needs to be global, since square_solve_recursive() will use it
   to determine whether it has already solved a square which is
   trivially equivalent to the current one. Note that this is
   perfectly thread-safe because not only will threads only access it
   when `solutions_mutex' is locked, but they won't ever write to it
   anyway (only read from it)! */
static square_list_t *beginning = NULL;

/* Solve a magic square and return 0 on success and put all of the
   solutions in `*solutions' (which will be allocated) if `solutions'
   is not NULL. Store the size in `*solutions_size' if
   `solutions_size' is not NULL. If `keep_going' is nonzero, keep
   finding solutions even after the first match. If `file' is not
   NULL, print first the description of the square (if any) followed
   by each of the solutions. Returns nonzero on error (in which case
   `*solutions' should NOT be free()'d). `*solutions' also should NOT
   be free()'d in the event that no solutions are found. Note that all
   the elements of `*solutions' should be free()'d also with
   square_destroy(). */
int square_solve(square_t *square, square_t **solutions,
		 size_t *solutions_size, char keep_going,
		 char skip_trivial, FILE *file
#ifdef HAVE_PTHREAD
		 , unsigned long max_threads
#endif
		 ) {
  square_list_t *solutions_list = NULL;
  cellval_t sum; /* The sum that all the rows, columns, and center
		    diagonals add up to */
  size_t solutions_size_local, *solutions_size_local_ptr;
  int ret
#if defined HAVE_PTHREAD && !defined NDEBUG
    , ret1
#endif
    ;
#ifdef HAVE_PTHREAD
  struct timespec sleeptime;

  threads_left = max_threads;
#endif

  solutions_size_local_ptr = solutions_size ?
    solutions_size : &solutions_size_local;

  *solutions_size_local_ptr = 0;

  /* Get the sum */
  sum.type = INT;
  sum.i = 0;

  /* Add all of the immutable squares */
  for (size_t x = 0; x < square->size; x++) {
    for (size_t y = 0; y < square->size; y++) {
      if (!square->cells[x][y].mutable) {
	sum = add_cellval(sum, square->cells[x][y].val);

	if (!cellval_valid(sum)) {
	  /* That means an error occurred */
	  return 1;
	}
      }
    }
  }

  /* Add the other numbers too */
  for (size_t i = 0; i < square->nums_size; i++) {
    sum = add_cellval(sum, square->nums[i]);

    if (!cellval_valid(sum)) {
      /* That means an error occurred */
      return 1;
    }
  }

  /* Divide the sum of all the numbers by the size of the square to
     get the magic sum. */
  if (sum.type == INT) {
    char is_neg = (sum.i < 0);

    if (is_neg) {
      /* Negative integer division and modulo arithmetic does not work
	 properly for some reason when using values computed at
	 run-time. This looks like a compiler bug, but we should work
	 around it for now. */
      sum.i = -sum.i;
    }

    if (sum.i % square->size) {
      /* If the size doesn't divide evenly into the sum of all the
	 numbers, the square is impossible to solve. Think about it:
	 If there are only integers, how do you expect to get a
	 non-integer sum by adding them together? */
      return 0;
    }

    sum.i = sum.i / square->size;

    if (is_neg)
      sum.i = -sum.i;
  }
  else {
    sum.f = sum.f / square->size;
  }

  /* If `solutions' is NULL, we aren't saving any solutions. BUT if we
     are skipping trivially equivalent solutions, we need to save
     solutions (but we won't return them). HOWEVER, if we're only
     looking for one solution, we needn't save solutions even if we
     are skipping trival equivalence. */
  if (solutions || (skip_trivial && keep_going)) {
    /* Allocate the solutions list */
    solutions_list = malloc(sizeof(*solutions_list));

    /* Remember that this is the end */
    solutions_list->next = NULL;

    /* Remember where the beginning is */
    beginning = solutions_list;
  }

  /* Print the description, if any and `file' isn't NULL. */
  if (square->description && file) {
    if (fputs(square->description, file) == EOF ||
	putc('\n', file) == EOF)
      return 1;

    /* We seemed to have successfully written to the file, but let's
       fflush() it just to make sure. This way, we can detect errors
       early (since fputs() and putc() may have appeared to have
       successfully written to the file, while in reality they only
       wrote to the buffer). */
    if (fflush(file))
      return 1;
  }

#ifdef HAVE_PTHREAD
  /* Initialize the mutexes for square_solve_recursive() */
  pthread_mutex_init(&solutions_mutex, NULL);
  pthread_mutex_init(&threads_left_mutex, NULL);
  pthread_mutex_init(&global_errno_mutex, NULL);
#endif

  /* Now solve the square! */
  ret = square_solve_recursive(square, sum, solutions_list ?
			       &solutions_list : NULL,
			       solutions_size_local_ptr,
			       keep_going, skip_trivial, file);

#ifdef HAVE_PTHREAD
  /* Wait for all the threads to finish */
  sleeptime.tv_sec = 0;
  sleeptime.tv_nsec = 1000000;

  pthread_mutex_lock(&threads_left_mutex);
  while (threads_left < max_threads) {
    pthread_mutex_unlock(&threads_left_mutex);

    /* Sleep one millisecond */
    nanosleep(&sleeptime, NULL);

    pthread_mutex_lock(&threads_left_mutex);
  }
  pthread_mutex_unlock(&threads_left_mutex);

  /* Sleep just a bit more to wait for the last thread to actually
     exit. */
  sleeptime.tv_nsec = 1000;
  nanosleep(&sleeptime, NULL);

  /* Destroy the mutexes */
# ifdef NDEBUG
#  define STMT_SEPARATOR ;
# else /* !NDEBUG */
#  define STMT_SEPARATOR ||

  /* Assertions are enabled, so we should save the return value */
  ret1 =
# endif /* !NDEBUG */
    pthread_mutex_destroy(&solutions_mutex) STMT_SEPARATOR
    pthread_mutex_destroy(&threads_left_mutex) STMT_SEPARATOR
    pthread_mutex_destroy(&global_errno_mutex);

# undef STMT_SEPARATOR

  /* If we failed to destroy any of the mutexes, it means we forgot to
     unlock them first which indicates a bug. */
  assert(!ret1);
#endif /* HAVE_PTHREAD */

  if (ret) {
    /* An error occured */
    if (solutions_list) {
      while (beginning != solutions_list) {
	square_list_t *old = beginning;

	beginning = beginning->next;

	square_destroy(&(old->square));
	free(old);
      }

      free(solutions_list);
    }

#ifdef HAVE_PTHREAD
    /* `errno' is most likely not set properly since it is
       thread-local. We have to set it from `global_errno'. */
    errno = global_errno;
#endif

    return 1;
  }

  if (solutions_list) {
    if (*solutions_size_local_ptr) {
      /* Convert the list to an array if we are expected to, otherwise
	 just free() the array. */

      if (solutions)
	*solutions = malloc(sizeof(**solutions) * *solutions_size_local_ptr);

      for (size_t i = 0; i < *solutions_size_local_ptr; i++) {
	square_list_t *old = beginning;

	if (solutions)
	  (*solutions)[i] = beginning->square;
	else
	  square_destroy(&(beginning->square));

	beginning = beginning->next;

	free(old);
      }
    }

    assert(beginning == solutions_list);

    free(solutions_list);
  }

  return 0;
}

/* Duplicate a square including the description */
square_t square_dup(square_t square) {
  square_t new_square = square_dup_nodesc(square);

  if (square.description)
    new_square.description = strdup(square.description);

  return new_square;
}

/* Duplicate a square excluding the description */
square_t square_dup_nodesc(square_t square) {
  square_t new_square;

  new_square.size		= square.size;
  new_square.nums_size		= square.nums_size;
  /* Don't duplicate it; that's for square_dup() */
  new_square.description	= NULL;

  new_square.cells	= malloc(sizeof(*(new_square.cells)) *
				 new_square.size);
  if (new_square.nums_size)
    new_square.nums	= malloc(sizeof(*(new_square.nums)) *
				 new_square.nums_size);

  for (size_t x = 0; x < new_square.size; x++) {
    new_square.cells[x] = malloc(sizeof(*(new_square.cells[x])) *
				  new_square.size);

    for (size_t y = 0; y < new_square.size; y++) {
      new_square.cells[x][y] = square.cells[x][y];
    }
  }

  for (size_t i = 0; i < new_square.nums_size; i++) {
    new_square.nums[i] = square.nums[i];
  }

  return new_square;
}

/* Destroy a square */
void square_destroy(square_t *square) {
  if (square->description)
    free(square->description);

  /* Free each of the columns */
  for (size_t i = 0; i < square->size; i++)
    free(square->cells[i]);

  /* Now free `square->cells' itself */
  free(square->cells);

  /* Free the other numbers if there are any */
  if (square->nums_size)
    free(square->nums);
}

/***************************\
|* Static helper functions *|
\***************************/

/* Add two `cellval_t's and return the result. */
static cellval_t add_cellval(cellval_t a, cellval_t b) {
  /* Make sure the types are valid */
  if (!cellval_valid(a)) {
    errno = EINVAL;
    return a;
  }

  if (!cellval_valid(b)) {
    errno = EINVAL;
    return b;
  }

  switch (a.type) {
  case INT:
    switch (b.type) {
    case INT:
      a.i += b.i;
      break;
    case FLOAT:
      a.type = FLOAT;
      a.f = a.i + b.f;
      break;
    }

    break;
  case FLOAT:
    a.f += (b.type == INT) ? b.i : b.f;
    break;
  }

  return a;
}

/* Solve a square adding the solutions (if found) to
   `solutions'. Returns 0 on success or nonzero on error. */
static int square_solve_recursive(square_t *square, cellval_t sum,
				  square_list_t **solutions,
				  size_t *solutions_size, char keep_going,
				  char skip_trivial, FILE *file) {
  char all_immutable = 1; /* Whether all of the cells are immutable */
  int valid;
  cellval_t nums[square->nums_size];

#ifdef HAVE_PTHREAD
  pthread_mutex_lock(&global_errno_mutex);

  if (global_errno) {
    /* Something bad happened */
    pthread_mutex_unlock(&global_errno_mutex);

    return 1;
  }

  pthread_mutex_unlock(&global_errno_mutex);

  pthread_mutex_lock(&solutions_mutex);
#endif /* HAVE_PTHREAD */

  if (!keep_going && *solutions_size > 0) {
    /* We've already found a solution and we weren't asked to find
       more. */
#ifdef HAVE_PTHREAD
    pthread_mutex_unlock(&solutions_mutex);
#endif

    return 0;
  }

#ifdef HAVE_PTHREAD
  pthread_mutex_unlock(&solutions_mutex);
#endif

  valid = square_valid(square, sum);

  if (valid < 0) {
    /* Error */

#ifdef HAVE_PTHREAD
    pthread_mutex_lock(&global_errno_mutex);
    global_errno = errno;
    pthread_mutex_unlock(&global_errno_mutex);
#endif

    return 1;
  }

  if (!valid) {
    /* We needn't check this square; it's already invalid */
    return 0;
  }

  /* Make a copy of `square->nums' so we can delete certain numbers */
  for (size_t i = 0; i < square->nums_size; i++) {
    nums[i] = square->nums[i];
  }

  for (size_t x = 0; x < square->size && all_immutable; x++) {
    for (size_t y = 0; y < square->size && all_immutable; y++) {
      if (square->cells[x][y].mutable) {
	if (!square->nums_size) {
	  /* That's an error */
	  errno = EINVAL;

#ifdef HAVE_PTHREAD
	  pthread_mutex_lock(&global_errno_mutex);
	  global_errno = errno;
	  pthread_mutex_unlock(&global_errno_mutex);
#endif

	  return 1;
	}

	all_immutable = 0;

	/* Temporarily make the cell immutable */
	square->cells[x][y].mutable = 0;

	/* Decrement `nums_size' by one since we will delete one
	   number from `square->nums' */
	square->nums_size--;

	/* Try to solve the square with this cell replaced with each
	   of the other numbers (remember, `square->nums_size + 1' is
	   the actual size of `nums') */
	for (size_t i = 0; i < square->nums_size + 1; i++) {
	  square->cells[x][y].val = nums[i];

	  /* Delete the number from `square->nums' */
	  square->nums[i] = square->nums[square->nums_size];

	  /* Try to solve it */

#ifdef HAVE_PTHREAD
	  pthread_mutex_lock(&threads_left_mutex);

	  if (threads_left) {
	    int ret = square_solve_new_thread(square, sum, solutions,
					      solutions_size, keep_going,
					      skip_trivial, file);

	    pthread_mutex_unlock(&threads_left_mutex);

	    if (ret == EAGAIN) {
	      /* We've reached the limit; don't start a new thread
		 after all. */
	      ret = square_solve_recursive(square, sum, solutions,
					   solutions_size, keep_going,
					   skip_trivial, file);
	    }

	    if (ret) {
	      /* We couldn't create a new thread for some other reason
		 than EAGAIN. */
	      square->nums[i] = nums[i];
	      square->nums_size++;
	      square->cells[x][y].mutable = 1;

	      pthread_mutex_lock(&global_errno_mutex);
	      global_errno = errno;
	      pthread_mutex_unlock(&global_errno_mutex);

	      return 1;
	    }
	  }
	  else {
	    pthread_mutex_unlock(&threads_left_mutex);
#endif /* HAVE_PTHREAD */

	    if (square_solve_recursive(square, sum, solutions,
				       solutions_size, keep_going,
				       skip_trivial, file)) {
	      /* Error */
	      square->nums[i] = nums[i];
	      square->nums_size++;
	      square->cells[x][y].mutable = 1;

#ifdef PTHREAD
	      /* No need to set `global_errno' since the recursive
		 call to square_solve_recursive() already set it. */
#endif

	      return 1;
	    }
#ifdef HAVE_PTHREAD
	  }
#endif

	  /* Put the number back */
	  square->nums[i] = nums[i];
	}

	/* And don't forget to reset `square->nums_size'! */
	square->nums_size++;

	/* Reset the cell to mutable */
	square->cells[x][y].mutable = 1;
      }
    }
  }

  if (all_immutable) {
    /* Hooray! We've found a solution! */

#ifdef HAVE_PTHREAD
    pthread_mutex_lock(&solutions_mutex);

    /* Check if we've already found a solution and we're not supposed
       to keep going again, since another thread might have found a
       solution when we weren't looking. */
    if (!keep_going && *solutions_size > 0) {
      pthread_mutex_unlock(&solutions_mutex);

      return 0;
    }
#endif

    if (solutions) {
      if (skip_trivial) {
	square_list_t *cur_solution = beginning;

	/* Make sure we don't already have a square which is trivially
	   equivalent (it is rotated/reflected or equivalent). */
	while (cur_solution->next) {
	  if (square_trivially_equiv(square, &(cur_solution->square))) {
#ifdef HAVE_PTHREAD
	    pthread_mutex_unlock(&solutions_mutex);
#endif

	    return 0;
	  }

	  cur_solution = cur_solution->next;
	}

	assert(cur_solution == *solutions);
      }

      /* Record the solution */
      (*solutions)->square = square_dup_nodesc(*square);
      (*solutions)->next = malloc(sizeof(*((*solutions)->next)));
      (*solutions) = (*solutions)->next;
      (*solutions)->next = NULL; /* Remember that this is the end */
    }

    if (*solutions_size == SIZE_MAX) {
      /* Overflow */
      errno = EOVERFLOW;

#ifdef HAVE_PTHREAD
      pthread_mutex_lock(&global_errno_mutex);
      global_errno = errno;
      pthread_mutex_unlock(&global_errno_mutex);

      pthread_mutex_unlock(&solutions_mutex);
#endif

      return 1;
    }

    (*solutions_size)++;

    if (file) {
      char *description = square->description;

      /* Write the solution to `file' */

      if (*solutions_size > 1 || description) {
	/* Separate this solution from the last solution or the
	   description */
	if (putc('\n', file) == EOF) {
#ifdef HAVE_PTHREAD
	  pthread_mutex_lock(&global_errno_mutex);
	  global_errno = errno;
	  pthread_mutex_unlock(&global_errno_mutex);

	  pthread_mutex_unlock(&solutions_mutex);
#endif

	  return 1;
	}
      }

      if (keep_going) {
	if (fprintf(file, "Solution %zu:\n\n", *solutions_size) < 0) {
#ifdef HAVE_PTHREAD
	  pthread_mutex_lock(&global_errno_mutex);
	  global_errno = errno;
	  pthread_mutex_unlock(&global_errno_mutex);

	  pthread_mutex_unlock(&solutions_mutex);
#endif

	  return 1;
	}
      }

      /* Temporarily get rid of the description */
      square->description = NULL;

      if (!write_human(square, file) ||
	  /* Flush the stream after each solution is found */
	  fflush(file) == EOF) {
	/* Error! */
	square->description = description;

#ifdef HAVE_PTHREAD
	pthread_mutex_lock(&global_errno_mutex);
	global_errno = errno;
	pthread_mutex_unlock(&global_errno_mutex);

	pthread_mutex_unlock(&solutions_mutex);
#endif

	return 1;
      }

      /* Replace the description */
      square->description = description;
    }

#ifdef HAVE_PTHREAD
    pthread_mutex_unlock(&solutions_mutex);
#endif
  }

  return 0;
}

#ifdef HAVE_PTHREAD
/* Create a new thread in which to solve a square. Returns zero on
   success, `errno' on error. The calling function should lock the
   mutex `threads_left_mutex' before calling and unlock it after
   completion. */
static int square_solve_new_thread(square_t *square, cellval_t sum,
				   square_list_t **solutions,
				   size_t *solutions_size, char keep_going,
				   char skip_trivial, FILE *file) {
  square_t *square_copy;
  struct solve_args *args;
  int ret, ret1;
  pthread_t thread;
  pthread_attr_t attr;

  if (!threads_left) {
    errno = EAGAIN;
    return errno;
  }

  /* Initialize the thread attributes structure. */
  ret = pthread_attr_init(&attr);

  if (ret) {
    errno = ret;
    return ret;
  }

  /* Make the thread detached so that we don't have to call
     pthread_join() on it. */
  ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

  if (ret) {
    errno = ret;

    ret = pthread_attr_destroy(&attr);

    if (ret)
      errno = ret;

    return errno;
  }

  /* Make a copy of the square since we will be creating a new
     thread. We also need to malloc() it because when we die, our
     stack will be destroyed. */
  square_copy = malloc(sizeof(*square_copy));
  *square_copy = square_dup_nodesc(*square);

  /* No need to copy the description since we won't be writing to
     it. */
  square_copy->description = square->description;

  /* We have to allocate this with malloc because our stack will be
     destroyed. */
  args = malloc(sizeof(*args));

  /* Pack up the arguments */
  args->square = square_copy;
  args->sum = sum;
  args->solutions = solutions;
  args->solutions_size = solutions_size;
  args->keep_going = keep_going;
  args->skip_trivial = skip_trivial;
  args->file = file;

  /* Now start the thread */
  ret = pthread_create(&thread, &attr, square_solve_unpack_args, args);

  /* We're done with `attr' now */
  ret1 = pthread_attr_destroy(&attr);

  if (ret || ret1) {
    square_copy->description = NULL;
    square_destroy(square_copy);
    free(square_copy);

    errno = ret ? ret : ret1;
    return errno;
  }

  threads_left--;

  return 0;
}

/* Unpack the arguments `args' and call square_solve_recursive(). Note
   that the return value is void * to be compatible with the pthreads
   API, but it should actually be casted to a long. Returns 0 on
   success, nonzero on error. */
static void * square_solve_unpack_args(void *ptr) {
  struct solve_args args = *(struct solve_args *)ptr;
  void *ret;

  free(ptr);

  ret = (void *)(long)square_solve_recursive(args.square, args.sum,
					     args.solutions,
					     args.solutions_size,
					     args.keep_going,
					     args.skip_trivial, args.file);

  /* free() the square since we made a copy */
  args.square->description = NULL;
  square_destroy(args.square);
  free(args.square);

  pthread_mutex_lock(&threads_left_mutex);
  threads_left++;
  pthread_mutex_unlock(&threads_left_mutex);

  return ret;
}
#endif /* HAVE_PTHREAD */

/* Check if a square is valid, returning 1 if it is, 0 if it's not,
   and -1 on error. */
static int square_valid(square_t *square, cellval_t sum) {
  char mutable; /* Whether at least one cell was mutable */
  cellval_t check_sum;

  if (!cellval_valid(sum)) {
    errno = EINVAL;
    return -1;
  }

  /* Check each column */
  for (size_t x = 0; x < square->size; x++) {
    check_sum.type = INT;
    check_sum.i = 0;
    mutable = 0;

    for (size_t y = 0; y < square->size; y++) {
      if (square->cells[x][y].mutable) {
	mutable = 1;
	break;
      }

      check_sum = add_cellval(check_sum, square->cells[x][y].val);

      if (!cellval_valid(check_sum))
	return -1;
    }

    if (!mutable && !cellval_equal(check_sum, sum)) {
      /* It's invalid */
      return 0;
    }
  }

  /* Check each row */
  for (size_t y = 0; y < square->size; y++) {
    check_sum.type = INT;
    check_sum.i = 0;
    mutable = 0;

    for (size_t x = 0; x < square->size; x++) {
      if (square->cells[x][y].mutable) {
	mutable = 1;
	break;
      }

      check_sum = add_cellval(check_sum, square->cells[x][y].val);

      if (!cellval_valid(check_sum))
	return -1;
    }

    if (!mutable && !cellval_equal(check_sum, sum)) {
      /* It's invalid */
      return 0;
    }
  }

  /* Check the upper left to lower right diagonal */
  check_sum.type = INT;
  check_sum.i = 0;
  mutable = 0;

  for (size_t i = 0; i < square->size; i++) {
    if (square->cells[i][i].mutable) {
      mutable = 1;
      break;
    }

    check_sum = add_cellval(check_sum, square->cells[i][i].val);

    if (!cellval_valid(check_sum))
      return -1;
  }

  if (!mutable && !cellval_equal(check_sum, sum)) {
    /* It's invalid */
    return 0;
  }

  /* Check the upper right to lower left diagonal */
  check_sum.type = INT;
  check_sum.i = 0;
  mutable = 0;

  for (size_t i = 0; i < square->size; i++) {
    if (square->cells[square->size - 1 - i][i].mutable) {
      mutable = 1;
      break;
    }

    check_sum = add_cellval(check_sum,
			    square->cells
			    [square->size - 1 - i][i].val);

    if (!cellval_valid(check_sum))
      return -1;
  }

  if (!mutable && !cellval_equal(check_sum, sum)) {
    /* It's invalid */
    return 0;
  }

  /* If we're here, the square is valid */
  return 1;
}

/* Check whether `a' is trivially equivalent to `b'; i.e. it is a
   rotation/reflection of `b' or is equivalent to `b'. Returns
   positive if they are trivially equivalent, zero if they are
   non-trivially different, and negative if an error occured. */
static int square_trivially_equiv(const square_t *a, const square_t *b) {
  enum transform {REF_HORIZ, REF_VERT, REF_DIAG_UL_LR, REF_DIAG_UR_LL,
		  ROT_90, ROT_180, ROT_270, ROT_0, TRANS_MAX};

  /* The squares must be the same size and they must be complete (no
     `other_nums'). */
  if (a->size != b->size || a->nums_size || b->nums_size) {
    errno = EINVAL;
    return -1;
  }

  for (enum transform trans = 0; trans < TRANS_MAX; trans++) {
    char equiv = 1; /* Whether the two squares are equivalent */

    for (size_t x = 0; x < a->size && equiv; x++) {
      for (size_t y = 0; y < a->size && equiv; y++) {
	cellval_t comp_cell; /* The cell to compare against */

	/* Make sure the current cell in `b' is valid */
	if (!cellval_valid(b->cells[x][y].val)) {
	  errno = EINVAL;
	  return -1;
	}

	/* Transform `a' and get the cell to compare against. */
	switch (trans) {
	case REF_HORIZ:
	  /* Horizontal reflection */
	  comp_cell = a->cells[a->size - 1 - x][y].val;
	  break;
	case REF_VERT:
	  /* Vertical reflection */
	  comp_cell = a->cells[x][a->size - 1 - y].val;
	  break;
	case REF_DIAG_UL_LR:
	  /* Diagonal reflection (from upper left corner to lower
	     right corner) */
	  comp_cell = a->cells[a->size - 1 - y][a->size - 1 - x].val;
	  break;
	case REF_DIAG_UR_LL:
	  /* Diagonal reflection (from upper right corner to lower
	     left corner) */
	  comp_cell = a->cells[y][x].val;
	  break;
	case ROT_90:
	  /* 90 degree rotation */
	  comp_cell = a->cells[a->size - 1 - y][x].val;
	  break;
	case ROT_180:
	  /* 180 degree rotation */
	  comp_cell = a->cells[a->size - 1 - x][a->size - 1 - y].val;
	  break;
	case ROT_270:
	  /* 270 degree rotation */
	  comp_cell = a->cells[y][a->size - 1 - x].val;
	  break;
	case ROT_0:
	  /* 0 degree rotation (no transformation) */
	  comp_cell = a->cells[x][y].val;
	  break;
	default:
	  assert(0);
	}

	/* Make sure `comp_cell' is valid */
	if (!cellval_valid(comp_cell)) {
	  errno = EINVAL;
	  return -1;
	}

	/* Now check if they're equal */
	if (!cellval_equal(comp_cell, b->cells[x][y].val)) {
	  /* They're not equal which means the whole squares with this
	     transformation are not equivalent. */
	  equiv = 0;
	}
      }
    }

    if (equiv) {
      /* The squares are equivalent. We're done! */
      return 1;
    }
  }

  /* If we're here, the squares were not equivalent. */
  return 0;
}
