/*
 * mail-to-news.c  ---  Mail to news gateway.
 *
 * Copyright (C)  2003,2004,2011  Marco Parrone
 *
 * This file is part of mail-to-news.
 *
 * mail-to-news is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * mail-to-news is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with metadiary.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Commentary:
 *
 * mail-to-news is a mail to news gateway.
 *
 * Code:
 */

/* Includes. */

/* Remember to update ../configure.ac when including new files. */

#include <unistd.h>
#include <fcntl.h>
#include <malloc.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#else
# define VERSION "unknown"
# define PACKAGE_BUGREPORT "marco@marcoparrone.com"
#endif /* HAVE_CONFIG_H */

/* End of includes. */

/* Pre-processed definitions. */

#ifndef TRUE
# define TRUE 1
#endif /* ! TRUE */

#ifndef FALSE
# define FALSE 0
#endif /* ! FALSE */

/* End of pre-processed definitions. */

/* Forward declarations. */

void realloc_v (void **ptr, unsigned int howmany, unsigned int type_size);
void read_error (char *func);
void write_error (char *func);
void read_char (void);
void write_char (void);
void header_content (void);
void header_content_2 (void);
void header_name_start (void);
void header_name_start_2 (void);
void header_name (void);
void header_name_end (void);
void header_skip_0 (void);
void header_skip (void);
void header_skip_2 (void);
void header_skip_3 (void);
void header_add (void);
void body_read (void);
void body (void);
void body_write (void);
void body_double (void);
void remove_from (void);
char **splithts (char *headers_to_strip);
int to_strip_p (char *header);
void print_help (void);
void print_version (void);
int main (int argc, char *argv []);

/* End of forward declarations. */

/* Global variables. */

int infd = STDIN_FILENO; /* Input file descriptor. */
int sock; /* Socket for communicating with the news server. */
char *header_lines_to_add = NULL; /* Header lines to add. */
char **headers_to_strip = NULL; /* Headers to strip from the message. */

char buf = 0; /* single character buffer, used for most read/write. */

/*
 * Single character buffer for recording the previous value of `buf'.
 */
unsigned int prev = 0;

/*
 * This "big buffer" is used ti temporary store each header name, for
 * comparing it to the headers to strip: it is used by the following
 * functions: `header_name', `header_name_end', `header_skip_0'.
 */
char *hbuf = NULL; /* Buffer. */
unsigned int hbufi = 0; /* Index. */
unsigned int hbuf_size = 255; /* Arbitrary start value. */

/*
 * If at_eof != FALSE, then the "body" function will return without
 * doing other function calls (and so the program will radily
 * terminate).  This variable is modified by `read_char'.
 */
unsigned int at_eof = FALSE;

/* End of global variables. */

/* Functions implementations. */

void
realloc_v (void **ptr, unsigned int howmany, unsigned int type_size)
{
  *ptr = realloc (*ptr, (howmany * type_size));
  if (*ptr == NULL)
    {
      fprintf (stderr, "mail-to-news: Memory exhausted.\n");
      exit (EXIT_FAILURE);
    }
}

void read_error (char *func)
{
  fprintf (stderr, "mail-to-news: read error (%s).\n", func);
  exit (EXIT_FAILURE);
}

void write_error (char *func)
{
  fprintf (stderr, "mail-to-news: write error (%s).\n", func);
  exit (EXIT_FAILURE);
}

void read_char (void)
{
  unsigned int readretv;
  prev = buf;
  readretv = read (infd, &buf, sizeof (char));
  if (readretv == -1)
    read_error ("read_char");
  else if (readretv == 0)
    at_eof = TRUE;
}

void write_char (void)
{
  if (buf == '\n' && prev != '\r')
    {
      if (write (sock, "\r\n", sizeof (char) * 2) == -1)
	write_error ("write_char");
    }
  else
    {
      if (write (sock, &buf, sizeof (char)) == -1)
	write_error ("write_char");
    }
}

void header_content (void)
{
  read_char ();
  header_content_2 ();
}

void header_content_2 (void)
{
  if (buf == '\n')
    { 
      write_char ();
      header_name_start ();
    }
  else
    {
      write_char ();
      header_content ();
    }
}

void header_name_start (void)
{
  read_char ();
  header_name_start_2 ();
}

void header_name_start_2 (void)
{
  switch (buf)
    {
    case ' ':
    case '\t':
      write_char ();
      header_content ();
      break;
    case '\n':
      header_add ();
      break;
    default:
      header_name ();
      break;
    }
}

void header_name (void)
{
  if (hbufi == (hbuf_size - 1))
    {
      hbuf_size += 25; /* Arbitrary step value. */
      realloc_v ((void **) &hbuf, hbuf_size, sizeof (char));
    }
  hbuf [hbufi++] = buf;
  if (buf == ':')
    {
      header_name_end ();
    }
  else
    {
      read_char ();
      header_name ();
    }
}
  
void header_name_end (void)
{
  hbuf [hbufi - 1] = 0;
  if (to_strip_p (hbuf) == TRUE)
    {
      header_skip_0 ();
    }
  else
    {
      hbuf [hbufi - 1] = ':';
      if (write (sock, hbuf, sizeof (char) * hbufi) == -1)
	write_error ("header_name_end");
      else
	hbufi = 0;
      header_content ();
    }
}

void header_skip_0 (void)
{
  hbufi = 0;
  header_skip ();
}

void header_skip (void)
{
  read_char ();
  header_skip_2 ();
}

void header_skip_2 (void)
{
  if (buf == '\n')
    {
      read_char ();
      header_skip_3 ();
    }
  else
    {
      header_skip ();
    }
}

void header_skip_3 (void)
{
  switch (buf)
    {
    case ' ':
    case '\t':
      header_skip ();
      break;
    case '\n':
      header_add ();
      break;
    default:
      header_name ();
      break;
    }
}

void header_add (void)
{
  unsigned int len;
  unsigned int ccc;
  if (header_lines_to_add != NULL)
    {
      len  = strlen (header_lines_to_add);
      for (ccc = 0; ccc < len; ++ccc)
	{
	  buf = header_lines_to_add [ccc];
	  write_char ();
	}
      buf = '\n';
      write_char ();
    }
  buf = '\n';
  write_char ();
  body_read ();
}

void body_read (void)
{
  read_char ();
  body ();
}

void body (void)
{
  if (at_eof == FALSE)
    {
      if (buf == '.')
	body_double ();
      else
	body_write ();
    }
  /* If at_eof != FALSE then don't call any function ==> return to main. */
}

void body_write (void)
{
  write_char ();
  body_read ();
}

void body_double (void)
{
  if (prev == '\n')
    {
      if (write (sock, "..", sizeof (char) * 2) == -1)
	write_error ("body_double");
      body_read ();
    }
  else
    {
      body_write ();
    }
}

void remove_from (void)
{
  unsigned int readretv;
  char lbuf [5];
  readretv = read (infd, &lbuf, sizeof (char) * 5);
  if (readretv == -1)
    {
      read_error ("remove_from");
    }
  else if (readretv == 0)
    {
      read_error ("remove_from: unexpected EOF");
    }
  else if (! (lbuf [0] == 'F'
	      && lbuf [1] == 'r'
	      && lbuf [2] == 'o'
	      && lbuf [3] == 'm'
	      && (lbuf [4] == ' ' || lbuf [4] == '\t')))
    {
      read_error ("remove_from: first line does not match '^From .*'");
    }
  else
    {
      prev = ' ';
      header_skip ();
    }
}

char ** splithts (char *headers_to_strip)
{
  unsigned int count;
  unsigned int allocated = 100; /* Arbitrary start value. */
  char **items = NULL;
  realloc_v ((void **) &items, allocated, sizeof (char *));
  items [0] = strtok (headers_to_strip, " ");
  for (count = 1; items [count - 1] != NULL; count++)
    {
      if (count == (allocated - 1))
	{
	  allocated = count + 10; /* Arbitrary step value. */
	  realloc_v ((void **) &items, allocated, sizeof (char *));
	}
      items [count] = strtok (NULL, " ");
    }
  return items;
}

int to_strip_p (char *header)
{
  int i;
  if (headers_to_strip != NULL)
    for (i = 0; headers_to_strip [i] != NULL; i++)
      if (! strcmp (header, headers_to_strip [i]))
	return TRUE;
  return FALSE;
}

void print_help (void)
{
  printf ("mail-to-news %s\n"
	  "\n"
	  "Purpose:\n"
	  "  mail-to-news is a mail to news gateway.\n"
	  "\n"
	  "Usage: mail-to-news [OPTIONS]...\n"
	  "   -h               --help                       Show help and exit.\n"
	  "   -V               --version                    Show version and exit.\n"
	  "\n", VERSION);

  printf ("   -i FILENAME      --input FILENAME             Input file (defaults to\n"
	  "                                                   stdin).\n"
	  "\n"
	  "   -s HEADERS       --strip-headers HEADERS      Headers to strip.\n"
	  "   -a HEADER-LINES  --add-header-lines HEADER-LINES  Header lines to add.\n");

  printf ("\n"
	  "   -H HOST          --host HOST                  News server host (defaults\n"
	  "                                                   to localhost).\n"
	  "\n"
	  "   -p PORT          --port PORT                  News server port (defaults\n"
	  "                                                   to 119).\n"
	  "\n"
	  "Please read the info manual for more informations.\n"
	  "\n"
	  "Please send bug-reports to <%s>.\n",
	  PACKAGE_BUGREPORT);
  exit (EXIT_SUCCESS);
}

void print_version (void)
{
  printf ("mail-to-news %s\n", VERSION);
  exit (EXIT_SUCCESS);
}

int main (int argc, char *argv [])
{
  char lbuf [] = { 0, 0, 0, 0};
  char trash;
  int sock_mode;
  struct sockaddr_in servername;
  struct hostent *hostinfo;

  int arg;
  FILE *fp;
  char *hostname = "localhost";
  char *port = "119";

  /* Parse the command line. */
  for (arg = 0; arg < argc; ++arg)
    {
      if (strcmp (argv [arg], "-h") == 0
	  || strcmp (argv [arg], "--help") == 0)
	{
	  print_help ();
	}
      else if (strcmp (argv [arg], "-V") == 0
	       || strcmp (argv [arg], "--version") == 0)
	{
	  print_version ();
	}
      else if (strcmp (argv [arg], "-i") == 0
	       || strcmp (argv [arg], "--input") == 0)
	{
	  if (arg < (argc - 1))
	    {
	      fp = fopen (argv [++arg], "r");
	      if (fp == NULL)
		{
		  fprintf (stderr, "mail-to-news:"
			   " Cannot read input file `%s'.\n",
			   argv [arg]);
		  exit (EXIT_FAILURE);
		}
	      else
		{
		  infd = fileno (fp);
		}
	    }
	  else
	    {
	      fprintf (stderr, "mail-to-news: missing parameter for"
		       " `--input' (`-i') option.\n");
	      exit (EXIT_FAILURE);
	    }
	}
      else if (strcmp (argv [arg], "-s") == 0
	       || strcmp (argv [arg], "--strip-headers") == 0)
	{
	  if (arg < (argc - 1))
	    {
	      headers_to_strip = splithts (argv [++arg]);
	    }
	  else
	    {
	      fprintf (stderr, "mail-to-news: missing parameter for"
		       " `--strip-headers' (`-s') option.\n");
	      exit (EXIT_FAILURE);
	    }
	}
      else if (strcmp (argv [arg], "-a") == 0
	       || strcmp (argv [arg], "--add-header-lines") == 0)
	{
	  if (arg < (argc - 1))
	    {
	      header_lines_to_add = argv [++arg];
	    }
	  else
	    {
	      fprintf (stderr, "mail-to-news: missing parameter for"
		       " `--add-header-lines' (`-a') option.\n");
	      exit (EXIT_FAILURE);
	    }
	}
      else if (strcmp (argv [arg], "-H") == 0
	       || strcmp (argv [arg], "--host") == 0)
	{
	  if (arg < (argc - 1))
	    {
	      hostname = argv [++arg];
	    }
	  else
	    {
	      fprintf (stderr, "mail-to-news:"
		       " missing parameter for `--host' (`-H') option.\n");
	      exit (EXIT_FAILURE);
	    }
	}
      else if (strcmp (argv [arg], "-p") == 0
	       || strcmp (argv [arg], "--port") == 0)
	{
	  if (arg < (argc - 1))
	    {
	      port = argv [++arg];
	    }
	  else
	    {
	      fprintf (stderr, "mail-to-news:"
		       " missing parameter for `--port' (`-p') option.\n");
	      exit (EXIT_FAILURE);
	    }
	}
    }

  /* Connect to the news server. */
  sock = socket (PF_INET, SOCK_STREAM, 0);
  if (sock < 0)
    {
      fprintf (stderr, "mail-to-news: connection error.\n");
      exit (EXIT_FAILURE);
    }
  servername.sin_family = AF_INET;
  servername.sin_port = htons (atoi (port));
  hostinfo = gethostbyname (hostname);
  if (hostinfo == NULL)
    {
      fprintf (stderr, "mail-to-news: host unknown.\n");
      exit (EXIT_FAILURE);
    }
  servername.sin_addr = *(struct in_addr *) hostinfo->h_addr;
  if (connect (sock, (struct sockaddr *) &servername, sizeof (servername)) < 0)
    {
      fprintf (stderr, "mail-to-news: connection error.\n");
      exit (EXIT_FAILURE);
    }

  /* Setup some variables which will be needed later. */
  sock_mode = fcntl (sock, F_GETFL, 0);
  realloc_v ((void **) &hbuf, hbuf_size, sizeof (char));

  /* Read the welcome message from the news server. */
  read (sock, lbuf, sizeof (char) * 3);

  /* Ignore the rest the server reply. */
  fcntl (sock, F_SETFL, O_NONBLOCK | sock_mode);
  while (read (sock, &trash, sizeof (char)) == 1);
  fcntl (sock, F_SETFL, sock_mode);

  /* Tell the news server we want to post an article. */
  if (write (sock, "post\r\n", sizeof (char) * 6) == -1)
    write_error ("main");

  /* Read the reply of the news server. */
  if (read (sock, lbuf, sizeof (char) * 3) == -1)
    read_error ("main");

  /* If the reply isn't OK, then show it and exit. */
  if (strcmp (lbuf, "340") != 0)
    {
      fprintf (stderr, "mail-to-news: error, cannot post: ");
      fprintf (stderr, "%s", lbuf);
      fcntl (sock, F_SETFL, O_NONBLOCK | sock_mode);
      while (read (sock, &trash, sizeof (char)) == 1)
	fputc (trash, stderr);
      fcntl (sock, F_SETFL, sock_mode);
      fputc ('\n', stderr);
      exit (EXIT_FAILURE);
    }

  /* Ignore the rest the server reply. */
  fcntl (sock, F_SETFL, O_NONBLOCK | sock_mode);
  while (read (sock, &trash, sizeof (char)) == 1);
  fcntl (sock, F_SETFL, sock_mode);

  /* Remove the '^From .*' line and call the needed following functions. */
  remove_from ();

  /* Tell to the news server that the article is ended. */
  if (write (sock, "\r\n.\r\n", sizeof (char) * 5) == -1)
      write_error ("main");

  /* Read the reply from the news server. */
  if (read (sock, lbuf, sizeof (char) * 3) == -1)
      read_error ("main");

  /* If the reply isn't OK, then show it and exit. */
  if (strcmp (lbuf, "240") != 0)
    {
      fprintf (stderr, "mail-to-news: error: post failed: ");
      fprintf (stderr, "%s", lbuf);
      fcntl (sock, F_SETFL, O_NONBLOCK | sock_mode);
      while (read (sock, &trash, sizeof (char)) == 1)
	fputc (trash, stderr);
      fcntl (sock, F_SETFL, sock_mode);
      fputc ('\n', stderr);
      return EXIT_FAILURE;
    }

  /* Tell "bye bye" to the news server. */
  if (write (sock, "quit\r\n", sizeof (char) * 6) == -1)
    write_error ("main");

  /* Clean up the resources and exit. */
  close (sock);
  if (fp != NULL)
    fclose (fp);
  free (headers_to_strip);
  free (hbuf);
  exit (EXIT_SUCCESS);
}

/* End of functions implementations. */

/* mail-to-news.c ends here. */
