/*
 * Make7Libre, free (as in freedom) implementation of the
 * unfamous game 'Make 7 Hexa Puzzle'
 * Copyright 2016 (C) Felicien PILLOT <felicien.pillot@member.fsf.org>
 * 
 * This file is part of Make7Libre.
 * 
 * Make7Libre is free software: you can redistribute it and/or modify
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Make7Libre is distributed in the hope that it will be useful,
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * with Make7Libre.  If not, see <http://www.gnu.org/licenses/>.
 */
/***********
 * game.cc *
 * ~~~~~~~ *
 ***********/

#include "../include/game.h"


/*
 * /////////////////////////
 * //   Public functions  //
 * /////////////////////////
 * ----------------------------------------------+
 * Game ();                                      |
 * connect_every_button ();                      |
 * display (Glib::RefPtr<Gtk::Application> app); |
 * end_game ();                                  |
 * import_every_widget ();                       |
 * launch_game ();                               |
 * ----------------------------------------------+
 */

Game::Game ()
{
}

void
Game::connect_every_button ()
{
  /* Interface buttons */
  // Cancel action
  if (button_cancel)
    button_cancel->signal_clicked ().connect
      (sigc::mem_fun (*this, &Game::on_cancel_action_activate));
  // Close scores
  if (button_close_scores)
    button_close_scores->signal_clicked ().connect
      (sigc::mem_fun (*this, &Game::on_close_scores_activate));
  // Display credits
  if (button_credits)
    button_credits->signal_clicked ().connect
      (sigc::mem_fun (*this, &Game::on_display_credits_activate));
  // Erase hexa
  if (button_erase)
    button_erase->signal_clicked ().connect
      (sigc::mem_fun (*this, &Game::on_erase_hexa_activate));
  // Display help
  if (button_help)
    button_help->signal_clicked ().connect
      (sigc::mem_fun (*this, &Game::on_display_help_activate));
  // New game
  if (button_new)
    button_new->signal_clicked ().connect
      (sigc::mem_fun (*this, &Game::on_new_game_activate));
  // Next hexa
  if (button_next)
    button_next->signal_clicked ().connect
      (sigc::mem_fun (*this, &Game::on_next_hexa_activate));
  // Quit game
  if (button_quit)
    button_quit->signal_clicked ().connect
      (sigc::mem_fun (*this, &Game::on_quit_game_activate));
  // Rotate hexas
  if (button_rotate)
    button_rotate->signal_clicked ().connect
      (sigc::mem_fun (*this, &Game::on_rotate_hexas_activate));
  // Display scores
  if (button_scores)
    button_scores->signal_clicked ().connect
      (sigc::mem_fun (*this, &Game::on_display_scores_activate));
  
  /* Game buttons */
  /* Each button is linked to one function, but the parameter
   * changes (indicating the position)
   * [ VERY FASTIDIOUS ... ISN'T THERE A BETTER WAY ??? ]
   */
  nnw->signal_clicked ().connect
    (sigc::bind<int>
     (sigc::mem_fun (*this, &Game::game_button_selected),
      13) );
  nn->signal_clicked ().connect
    (sigc::bind<int>
     (sigc::mem_fun (*this, &Game::game_button_selected),
      14));
  nne->signal_clicked ().connect
    (sigc::bind<int>
     (sigc::mem_fun (*this, &Game::game_button_selected),
      15));
  nww->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	22));
  nw->signal_clicked ().connect
    (sigc::bind<int>
     (sigc::mem_fun (*this, &Game::game_button_selected),
      23));
  ne->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	24));
  nee->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	25));
  ww->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	31));
  w->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	32));
  m->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	33));
  e->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	34));
  ee->signal_clicked ().connect
    (sigc::bind<int>
     (sigc::mem_fun (*this, &Game::game_button_selected),
      35));
  sww->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	41));
  sw->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	42));
  se->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	43));
  see->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	44));
  ssw->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	51));
  ss->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	52));
  sse->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	53));
  /*  
  mainNw->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	-10));
  mainNe->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	-9));
  mainW->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	-1));
  mainM->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	0));
  mainE->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	1));
  mainSw->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	9));
  mainSe->signal_clicked ().connect
      (sigc::bind<int>
       (sigc::mem_fun (*this, &Game::game_button_selected),
	10));
  */
}

void
Game::display (Glib::RefPtr<Gtk::Application> app)
{
  Glib::RefPtr<Gdk::Pixbuf> icon =
    Gdk::Pixbuf::create_from_file (ICON_DIR "/32x32/apps/make7libre.png");
  make7window->set_icon (icon);
  make7window->override_background_color (Gdk::RGBA("#E0D7CA"));
  app->run (*make7window);
}

void
Game::end_game ()
{
  delete make7window;
}

void
Game::import_every_widget ()
{
  Glib::RefPtr<Gtk::Builder> builder = Gtk::Builder::create_from_file
    (UI_DIR "/basic.glade");
  
  // Get every Make 7 windows
  builder->get_widget ("make7window", make7window);
  builder->get_widget ("make7scores", make7scores);
  builder->get_widget ("make7message", make7message);
  builder->get_widget ("make7credits", make7credits);
  builder->get_widget ("make7error_board", make7error_board);
  // Get every interface buttons
  builder->get_widget ("button_cancel", button_cancel);
  builder->get_widget ("button_close_scores", button_close_scores);
  builder->get_widget ("button_credits", button_credits);
  builder->get_widget ("button_erase", button_erase);
  builder->get_widget ("button_help", button_help);
  builder->get_widget ("button_new", button_new);
  builder->get_widget ("button_next", button_next);
  builder->get_widget ("button_quit", button_quit);
  builder->get_widget ("button_rotate", button_rotate);
  builder->get_widget ("button_scores", button_scores);
  // Get score label
  builder->get_widget ("current_score", current_score);  
  // Get every game buttons
  // (board)
  builder->get_widget ("nnw", nnw);
  builder->get_widget ("nn", nn);
  builder->get_widget ("nne", nne);
  builder->get_widget ("nww", nww);
  builder->get_widget ("nw", nw);
  builder->get_widget ("ne", ne);
  builder->get_widget ("nee", nee);
  builder->get_widget ("ww", ww);
  builder->get_widget ("w", w);
  builder->get_widget ("m", m);
  builder->get_widget ("e", e);
  builder->get_widget ("ee", ee);
  builder->get_widget ("sww", sww);
  builder->get_widget ("sw", sw);
  builder->get_widget ("se", se);
  builder->get_widget ("see", see);
  builder->get_widget ("ssw", ssw);
  builder->get_widget ("ss", ss);
  builder->get_widget ("sse", sse);
  // (selection area)
  builder->get_widget ("mainNw", mainNw);
  builder->get_widget ("mainNe", mainNe);
  builder->get_widget ("mainW", mainW);
  builder->get_widget ("mainM", mainM);
  builder->get_widget ("mainE", mainE);
  builder->get_widget ("mainSw", mainSw);
  builder->get_widget ("mainSe", mainSe);
  Board = { {13, nnw}, {14, nn}, {15, nne}, {22, nww},
	    {23, nw}, {24, ne}, {25, nee}, {31, ww},
	    {32, w}, {33, m}, {34, e}, {35, ee}, {41, sww},
	    {42, sw}, {43, se}, {44, see}, {51, ssw},
	    {52, ss}, {53, sse} };
  SelectionArea = { {-10, mainNw}, {-9, mainNe},   
		    {-1, mainW}, {0, mainM}, {1, mainE},
		    {9, mainSw}, {10, mainSe} };
}

void
Game::launch_game ()
{
  // Define keys
  BoardKeys = { 13, 14, 15, 22, 23, 24, 25,
		31, 32, 33, 34, 35, 41, 42, 43,
		44, 51, 52, 53 };
  SelectionAreaKeys = { -10, -9, -1, 0,	1, 9, 10 };
  // Reset the score
  current_score->set_label ("0 pts");
  // Clean board
  reset_board ();
  // Prepare the selection area
  get_random_hexas ();
  refresh_selection_area ();
}


/*
 * /////////////////////////
 * //  Private functions  //
 * /////////////////////////
 * ----------------------------------------+
 * display_error (int i);                  |
 * game_button_selected (int ref);         |
 * get_main_hexa ();                       |
 * get_neighbour (int rel, int ref);       |
 * get_pos2 ();                            |
 * get_random_hexas ();                    |
 * get_second_hexa ();                     |
 * on_cancel_action_activate ();           |
 * on_close_help_activate ();              |
 * on_close_scores_activate ();            |
 * on_display_credits_activate ();         |
 * on_display_help_activate ();            |
 * on_display_scores_activate ();          |
 * on_erase_hexa_activate ();              |
 * on_new_game_activate ();                |
 * on_next_hexa_activate ();               |
 * on_quit_game_activate ();               |
 * on_rotate_hexas_activate ();            |
 * next_turn (int a, int b);               |
 * refresh_score (int bonus);              |
 * refresh_selection_area ();              |
 * reset_board ();                         |
 * reset_selection_area ();                |
 * scan_neighbours (int ref, int oldRef);  |
 * set_main_hexa (std::string Snum);       |
 * set_pos2 (int givenPos2);               |
 * set_second_hexa (std::string Snum);     |
 * ----------------------------------------+
 */

void
Game::display_error (int i)
{
  std::cout << "display_error () : " << i << std::endl;
  make7error_board->show ();
}

void
Game::game_button_selected (int ref)
{
  std::cout << "game_button_selected () : " << ref
	    << " -> " << Board[ref]->get_label () << std::endl;
  if (Board[ref]->get_label () != "")
    display_error (1);
  else
    {
      if (get_second_hexa () != "nosecondhexa")
	{
	  int nb = get_neighbour (pos2, ref);
	  // Out of the board or over an existing value
	  if (nb == -42)
	    display_error (2);
	  else if (Board[nb]->get_label () != "")
	    display_error (3);
	  else
	    {
	      Board[ref]->set_label (get_main_hexa ());
	      Board[nb]->set_label (get_second_hexa ());
	      next_turn (ref, nb);
	    }
	}
      else
	{
	  Board[ref]->set_label (get_main_hexa ());
	  next_turn (ref, -42);
	}
    }
}

Glib::ustring
Game::get_main_hexa ()
{
  return mainM->get_label ();
}

int
Game::get_neighbour (int rel, int ref)
{
  std::cout << "get_neighbour ()" << std::endl;
  if ((rel == -10 && (ref == 15 || ref == 14 || ref == 13 ||
		      ref == 22 || ref == 31))
      ||
      (rel == -9 && (ref == 13 || ref == 14 || ref == 15 ||
		     ref == 25 || ref == 35))
      ||
      (rel == -1 && (ref == 13 || ref == 22 || ref == 31 ||
		     ref == 41 || ref == 51))
      ||
      (rel == 1 && (ref == 15 || ref == 25 || ref == 35 ||
		    ref == 44 || ref == 53))
      ||
      (rel == 9 && (ref == 31 || ref == 41 || ref == 51 ||
		    ref == 52 || ref == 53))
      ||
      (rel == 10 && (ref == 51 || ref == 52 || ref == 53 ||
		     ref == 44 || ref == 35)))
    // Forbidden move
    return -42;
  else
    return rel + ref;
}

int
Game::get_pos2 ()
{
  return pos2;
}

void
Game::get_random_hexas ()
{
  std::cout << "get_random_hexas ()" << std::endl;
  
  int nHexa = rand () % 2;
  if (nHexa == 1)
    {
      valHexa1 = rand () % 3 + 1;
      set_pos2 (-42);
    }
  if (nHexa == 0)
    {
      valHexa1 = rand () % 3 + 1;
      valHexa2 = rand () % 3 + 1;
      set_pos2 (9);
    }
}

Glib::ustring
Game::get_second_hexa ()
{
  if (pos2 != -42)
    return SelectionArea[pos2]->get_label ();
  return "nosecondhexa";
}

void
Game::on_cancel_action_activate ()
{
  std::cout << "on_cancel_action_activate ()" << std::endl;
}

void
Game::on_close_help_activate ()
{
  std::cout << "on_close_help_activate ()" << std::endl;
  make7message->hide ();
}

void
Game::on_close_scores_activate ()
{
  std::cout << "on_close_scores_activate ()" << std::endl;
  make7scores->hide ();
}

void
Game::on_display_credits_activate ()
{
  std::cout << "on_display_credits_activate ()" << std::endl;
  make7credits->show ();
}

void
Game::on_display_help_activate ()
{
  std::cout << "on_display_help_activate ()" << std::endl;
  make7message->show ();
}

void
Game::on_display_scores_activate ()
{
  std::cout << "on_display_scores_activate ()" << std::endl;
  make7scores->show ();
}

void
Game::on_erase_hexa_activate ()
{
  std::cout << "on_erase_hexa_activate ()" << std::endl;
}

void
Game::on_new_game_activate ()
{
  std::cout << "on_new_game_activate ()" << std::endl;
  launch_game ();
}

void
Game::on_next_hexa_activate ()
{
  std::cout << "on_next_hexa_activate ()" << std::endl;
}

void
Game::on_quit_game_activate ()
{
  std::cout << "on_quit_game_activate ()" << std::endl;
  if (make7window)
    make7window->hide ();
}

void
Game::on_rotate_hexas_activate ()
{
  std::cout << "on_rotate_hexas_activate ()" << std::endl;
  int actualPos2 = get_pos2 ();
  if (actualPos2 == -10)
    set_pos2 (-9);
  if (actualPos2 == -9)
    set_pos2 (1);
  if (actualPos2 == 1)
    set_pos2 (10);
  if (actualPos2 == 10)
    set_pos2 (9);
  if (actualPos2 == 9)
    set_pos2 (-1);
  if (actualPos2 == -1)
    set_pos2 (-10);
  reset_selection_area ();
  refresh_selection_area ();
}

void
Game::next_turn (int a, int b)
{
  int oldA, oldB;
  std::list<int> listA, listB;
  get_random_hexas ();
  refresh_selection_area ();
  nMatch = 0;
  
  // Main hexa
  listA = {a};
  scan_neighbours (a, listA);
  if (listA.size () > 2)
    {
      // Merge values for a
      oldA = std::stoi (Board[a]->get_label ());
      Board[a]->set_label (std::to_string (++oldA));
    }
  // Second hexa
  if (b != -42 && Board[b]->get_label () != "")
    {
      listB = {b};
      scan_neighbours (b, listB);
      if (listB.size () > 2)
	{
	  // Merge values for b
	  oldB = std::stoi (Board[b]->get_label ());
	  Board[b]->set_label (std::to_string (++oldB));
	}
    }
  
  // Consequences
  refresh_board ();
  refresh_score ();
}

void
Game::refresh_board ()
{
  std::cout << "refresh_board ()" << std::endl;
  // ListI contains the already checked hexas
  // begginning with 13, the first hexa of the board.
  std::list<int> listI;
  int oldI;
  for (int i : BoardKeys)
    {
      listI = {i};
      scan_neighbours (i, listI);
      if (listI.size () > 2)
	{
	  oldI = std::stoi (Board[i]->get_label ());
	  Board[i]->set_label (std::to_string (++oldI));
	}
    }
}

void
Game::refresh_score ()
{
  std::cout << "refresh_score ()" << std::endl;
  int oldScore = std::stoi (current_score->get_label ());
  std::string score = std::to_string (oldScore + nMatch) + " pts";
  current_score->set_label (score);
}

void
Game::refresh_selection_area ()
{
  std::cout << "refresh_selection_area ()" << std::endl;
  reset_selection_area ();
  set_main_hexa (std::to_string (valHexa1));
  set_second_hexa (std::to_string (valHexa2));
}

void
Game::reset_board ()
{
  std::cout << "reset_board ()" << std::endl;
  std::list<int>::iterator key;
  for (key = BoardKeys.begin ();
       key != BoardKeys.end (); key ++)
    Board[*key]->set_label ("");
}

void
Game::reset_selection_area ()
{
  std::cout << "reset_selection_area ()" << std::endl;
  std::list<int>::iterator key;
  for (key = SelectionAreaKeys.begin ();
       key != SelectionAreaKeys.end (); key ++)
    {
      SelectionArea[*key]->set_label ("");
      Glib::PropertyProxy<Gtk::ReliefStyle> prop =
	SelectionArea[*key]->property_relief ();
      prop.set_value (Gtk::RELIEF_NONE);
    }
}

void
Game::scan_neighbours (int ref, std::list<int> &oldRefs)
{
  std::cout << "scan_neighbours (" << ref << ")" << std::endl;
  // Scan every neighbours around ref hexa
  for (std::list<int>::iterator key = SelectionAreaKeys.begin ();
       key != SelectionAreaKeys.end (); key ++)
    {
      // Skip the central position
      if (*key == 0)
	continue;
      int Nb = get_neighbour (*key, ref);
      // Check if we are not out of the board
      if (Nb == -42)
	continue;
      // Check if we don't match the previous hexa
      bool skip = false;
      for (int i : oldRefs)
	if (Nb == i)
	  skip = true;
      if (skip)
	continue;
      // If this matching hexa has the same value as the current one
      if (Board[ref]->get_label () != "" &&
	  Board[Nb]->get_label () == Board[ref]->get_label ())
	{
	  nMatch++;
	  oldRefs.push_back (Nb);
	  scan_neighbours (Nb, oldRefs);
	  if (oldRefs.size () > 2)
	    {
	      std::cout << "Erasing " << Nb << std::endl;
	      Board[Nb]->set_label ("");
	    }
	}
    }
}

void
Game::set_main_hexa (std::string Snum)
{
  std::cout << "set_main_hexa ()" << std::endl;
  mainM->set_label (Snum);
  Glib::PropertyProxy<Gtk::ReliefStyle> prop =
    mainM->property_relief ();
  prop.set_value (Gtk::RELIEF_NORMAL);
}

void
Game::set_pos2 (int givenPos2)
{
  pos2 = givenPos2;
}

void
Game::set_second_hexa (std::string Snum)
{
  std::cout << "set_second_hexa ()" << std::endl;
  if (pos2 != -42)
    {
      SelectionArea[pos2]->set_label (Snum);
      Glib::PropertyProxy<Gtk::ReliefStyle> prop =
	SelectionArea[pos2]->property_relief ();
      prop.set_value (Gtk::RELIEF_NORMAL);
    }
}
