/*
 * Make7Libre, free (as in freedom) implementation of the
 * unfamous game 'Make 7 Hexa Puzzle'
 * Copyright 2016-2017 (C) Felicien PILLOT <felicien.pillot@member.fsf.org>
 * 
 * This file is part of Make7Libre.
 * 
 * Make7Libre is free software: you can redistribute it and/or modify
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * Make7Libre is distributed in the hope that it will be useful,
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * with Make7Libre.  If not, see <http://www.gnu.org/licenses/>.
 */
/***********
 * game.h  *
 * ~~~~~~~ *
 ***********/

#ifndef MAKE7LIBRE_GAME_H__
#define MAKE7LIBRE_GAME_H__

#if HAVE_CONFIG_H
#  include <config.h>
#endif /* HAVE_CONFIG_H */
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif /* HAVE_UNISTD_H */

#include <iostream>
#include <gtkmm.h>
#include "gettext.h"

/** Main class
 * contains the graphical interface (set by Glade and imported with Gtk::Builder)
 * and the main functions of the game.
 */
class Game
{
public:
	/* Public functions */
	Game (); 											/**< Constructor */
	void connect_every_button ();						/**< Function containing a lot of lines */
	void display (Glib::RefPtr<Gtk::Application> app);  /**< Here is actually displayed the window */
	void end_game ();									/**< Just a function that save the score */
	void import_every_widget ();						/**< Function containing a lot of lines too */
	void launch_game ();								/**< All initializations needed at the beginning of a new game */

private:
	/* Private functions */
	void display_error (int i);							/**< Spawns a Gtk::MessageDialog containing an appropriated message */
	void define_hexa (int ref, std::string valref);		/**< Define the new hexa then check neighbors while there are */
	void erase_hexa (int ref);							/**< Erase value and color of the given hexa */
	void game_button_selected (int ref);				/**< Called when any Board button is clicked */
	Glib::ustring get_main_hexa ();						/**< Returns the value of the main hexa button label */
	int get_neighbour (int rel, int givenRef);			/**< Calculates the position of a neighbor button */
	int get_pos2 ();									/**< Returns pos2 (not very useful) */
	void get_random_hexas ();							/**< Generate random numbers and fill valHexa1 and valHexa2 with them */
	Glib::ustring get_second_hexa ();					/**< Same as get_main_hexa () but with the second */
	void get_snapshot ();								/**< Backup the whole Board and SelectionArea in a std::map */
	void merge_hexas (int newHexa, std::list<int> oldHexas);	/**< Erase values and colors of matching neighbors */
	void on_cancel_action_activate ();					/**< Signal function activated by button_cancel */
	void on_close_scores_activate ();					/**< Signal function activated by button_close_scores */
	void on_display_credits_activate ();				/**< Signal function activated by button_credits */
	void on_display_help_activate ();					/**< Signal function activated by button_help */
	void on_display_scores_activate ();					/**< Signal function activated by button_scores */
	void on_erase_hexa_activate ();						/**< Signal function activated by button_erase */
	void on_new_game_activate ();						/**< Signal function activated by button_new */
	void on_next_hexa_activate ();						/**< Signal function activated by button_next */
	void on_quit_game_activate ();						/**< Signal function activated by button_quit */
	void on_rotate_hexas_activate ();					/**< Signal function activated by button_rotate */
	int refresh_score ();								/**< Update the amount of points in the score label */
	void refresh_selection_area ();						/**< Re-display the content of the SelectionArea */
	void reset_board ();								/**< Erase values and colors of all hexas on the Board */
	void reset_selection_area ();						/**< Erase values and colors of all hexas on the SelectionArea */
	void reset_snapshot ();								/**< Re-display an old configuration of Board and SelectionArea, using the std::map created by get_snapshot () */
	bool scan_neighbours (int ref, std::list<int> &oldRefs);	/**< Check if the neighbors of a given hexa have the same value as it */
	void set_main_hexa (std::string Snum);				/**< Put the string to the label of the main hexa in SelectionArea */
	void set_pos2 (int givenPos2);						/**< Set pos2 to the given value (not very useful) */
	void set_second_hexa (std::string Snum);			/**< Same as set_main_hexa (), but with the second */
	bool triggers (int ref);							/**< Deal with a new hexa on the board */
	void update_hexa_color (int hexa);					/**< Change the color of a given hexa */

	/* Windows*/
	Gtk::Window *make7window, *make7scores;
	Gtk::MessageDialog *make7message, *make7error_board;
	Gtk::AboutDialog *make7credits;

	/* Buttons pointers */
	// For the interface
	Gtk::Button	*button_cancel, *button_close_scores, *button_credits,
	*button_help, *button_new, *button_next,
	*button_quit, *button_rotate, *button_scores;
	Gtk::ToggleButton *button_erase;
	//  For the Game Board
	Gtk::Button	*nnw, *nn, *nne, *nww, *nw, *ne, *nee, *ww,
	*w,	*m, *e, *ee, *sww,*sw, *se, *see, *ssw, *ss, *sse,
	//  For the Game Selection Area
	*mainNw, *mainNe, *mainW, *mainM, *mainE, *mainSw, *mainSe;

	/* Game score and box */
	Gtk::Label *current_score;
	Gtk::Box *box_score;

	/* Some shared global variables */
	std::map<int,Gtk::Button*> Board, SelectionArea;
	std::map<int,std::string> BoardSnapshotMap, Colors;
	std::list<int> BoardKeys, SelectionAreaKeys, ColorsKeys;
	int pos2, max, valHexa1, valHexa2, score, nMatch;
	std::string oldValHexa1, oldValHexa2;
	Glib::RefPtr<Gtk::CssProvider> CssProvider;
	bool transition;
};

#endif // MAKE7LIBRE_GAME_H__
