/*
 * marsh:  a short program to read and save data
 *         from a MAS-345 multimeter to a file.
 *
 * Copyright (C) 2005 Marcello Carla'
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  Address of the author:
 *  Department of Physics, University of Florence
 *  Via G. Sansone 1
 *  Sesto F.no (Firenze) I 50019 - Italy
 *  e-mail: carla@fi.infn.it
 *
 */

/*
 *  Compile with "gcc marsh.c -o marsh" or "make marsh" and run "./marsh".
 *  Use "./marsh -h" to see available options.
 *
 */

#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sys/time.h>
#include <time.h>
#include <stdlib.h>
#include <ctype.h>

#define  DATA_STRING 14   /* length of data string from MAS-345 */
#define  MAX_PORTS 4      /* can be increased up to 32 */
#define  MAX_TIME_OUT 7   /* calculation disabled after these many failures */
#define  MAX_RESULT 40    /* max length for result string */

#define  DTR 2
#define  RTS 4

  struct port_area {
    char *port;                 /* device name */
    int  fd;                    /* file descriptor */
    struct termios oldtio;      /* port setting as found at start */
    char inbuf[DATA_STRING+1];  /* data area for input string */
    int  j;                     /* character count in inbuf */
    int  mask;                  /* flag position in mask */
  };

struct termios newtio;
int  rts = RTS, dtr = DTR;
char *default_port = "/dev/ttyS0";
char *default_out = "mas345.dat";
int D = 0, DD = 0, DDD = 0;                 /* verbose debug flag */

/*
 * restore_port_settings - to be called on exit for clean operation
 */

void restore_port_settings (int exit_value, struct port_area *p) {

  if (D) printf ("D - restoring port settings at fd %d with %o %o %o %o\n",
		 p->fd,
		 p->oldtio.c_iflag, p->oldtio.c_cflag,
		 p->oldtio.c_oflag, p->oldtio.c_lflag);

  ioctl (p->fd, TIOCMBIC, &dtr);                /* clear the DTR line */

  tcflush(p->fd, TCIFLUSH);
  if (tcsetattr(p->fd, TCSANOW, &p->oldtio))
    perror ("### 'restore port settings' failed");
  close (p->fd);
}

/*
 * catch_the_signal(signal) - termination signals are to be
 *                            catched for clean exit - SIGALRM has
 *                            to be catched to return from pause()
 */

void catch_the_signal (signal) {

  if (signal == SIGALRM || signal == SIGCHLD) {
    if (DD) printf ("DD - received signal %d\n", signal);
    return;
  }
  if (D) printf ("D - received signal %d\n", signal);
  exit (0);
}

/*
 * close the auxiliary calculator
 */

void close_calculator (int exit_value, int fd_elab[]) {

  write (fd_elab[1], "quit\n", 5);
  close (fd_elab[1]);
  close (fd_elab[2]);
}

/*
 * time operations: difference and sum
 */

void time_diff (struct timeval *a, struct timeval *b, struct timeval *d) {
  d->tv_sec = a->tv_sec - b->tv_sec;
  d->tv_usec = a->tv_usec - b->tv_usec;
  if (d->tv_usec < 0) {
    d->tv_usec += 1000000;
    d->tv_sec--;
  }
}

void time_add (struct timeval *a, struct timeval *b, struct timeval *s) {
  s->tv_sec = a->tv_sec + b->tv_sec;
  s->tv_usec = a->tv_usec + b->tv_usec;
  if (s->tv_usec > 999999) {
    s->tv_usec -= 1000000;
    s->tv_sec++;
  }
}

/*
 * time_stamp - convert timeval to a 24 chars string with format:
 *
 *      yyyy-mm-dd  hh:mm:ss.dd
 *                1    1    2
 *      0    5    0    5    0
 *
 * and return the string length
 *
 */

int time_stamp(char *buf, struct timeval *tv) {
  struct tm now;

  localtime_r(&tv->tv_sec, &now);

  sprintf (buf,"%4.4d-%2.2d-%2.2d  %2.2d:%2.2d:%2.2d.%2.2d ",
           now.tm_year+1900, now.tm_mon+1, now.tm_mday,
           now.tm_hour, now.tm_min, now.tm_sec, (int) (tv->tv_usec/1.e4));
  buf[23] = '\0';
  return (23);
}


/*
 * open and setup the port
 */

void open_setup_line (struct port_area *p) {

  if (D) printf ("D - entering line setup for port <%s>\n", p->port);

  if ((p->fd = open (p->port, O_RDWR | O_NOCTTY | O_NONBLOCK)) == -1) {
    fprintf (stderr, "### Unable to open port %s: %s\n",
	     p->port, strerror(errno));
    exit (-1);
  }

  if (D) printf ("D - port <%s> opened with file descriptor %d\n",
		 p->port, p->fd);

  /* save old port settings and set up new attributes */
  /* ref: man 3 termios */

  if (tcgetattr (p->fd, &p->oldtio)) {
    fprintf (stderr, "### Failed reading setting of port <%s>: %s\n",
	     p->port, strerror(errno));
    exit (-1);
  }
  if (D) printf("D - found port settings %o %o %o %o on port <%s>\n",
		p->oldtio.c_iflag, p->oldtio.c_cflag,
		p->oldtio.c_oflag, p->oldtio.c_lflag, p->port);

  /* register to restore old port settings on exit */

  if (on_exit((void*) restore_port_settings, p)) {
    perror ("### Cannot register for clean exit");
    exit (-1);
  }

  /* now activate new settings and flush buffers */

  if (tcsetattr(p->fd, TCSANOW, &newtio)) {
    fprintf (stderr, "### Cannot set port attributes on port <%s>: %s\n",
	     p->port, strerror(errno));
    exit (-1);
  }    

  if (D) printf("D - going to clear RTS and set DTR\n");
  ioctl (p->fd, TIOCMBIC, &rts);      /* clear RTS line for - supply */
  ioctl (p->fd, TIOCMBIS, &dtr);      /* set DTR line for + supply */

  usleep(1000);
  tcflush(p->fd, TCIOFLUSH);

  if (D) printf("D - done with line setup\n");
  return;
}

/*
 * start the calculator
 */

void start_calc (char **elab, char * elab_p, int fd_elab[]) {

  int pid_comp;

  if (D) printf ("D - starting external calculator\n");

  if (pipe(fd_elab) || pipe(fd_elab+2)) {
    perror ("### Failed creating pipes for calculator. Calculation disabled");
    *elab = 0;
    return;
  }
  if (D) printf ("D - created pipes (%d %d),(%d %d)\n",
		 fd_elab[0], fd_elab[1], fd_elab[2], fd_elab[3]);

  if ((pid_comp = fork()) == -1) {
    perror ("### 'fork' failed - calculation disabled");
    close (fd_elab[0]); close (fd_elab[1]);
    close (fd_elab[2]); close (fd_elab[3]);
    *elab = 0;
    return;
  }

  if (D) printf ("D - forked. Here pid %d\n", pid_comp);

  if (pid_comp == 0) {

    /* this is child - 'exec bc' */

    close (fd_elab[1]);
    close (fd_elab[2]);

    if (D) printf ("D - child - starting calculator 'bc' with i/o %d %d\n",
		   fd_elab[0], fd_elab[3]);

    if (dup2(fd_elab[0],0) == -1 || dup2(fd_elab[3],1) == -1 ||
	execlp ("bc", "bc", 0) == -1) {
      perror ("### Couldn't start external calculator.");
      close (fd_elab[0]); close (fd_elab[3]);
      exit (-1);
    }
  }

   /* this is parent */

  close (fd_elab[0]);
  close (fd_elab[3]);

  if (on_exit((void *) close_calculator, fd_elab)) {
    perror ("### Failed registering exit routine for calculator program");
    fprintf (stderr, "### Maybe you shall kill it yourself.\n");
  }

  if (D) printf ("D - parent - sending preset command to calculator: <%s>\n",
		 elab_p);

  if (elab_p) {                    /* send preset to calculator, if any */
    write (fd_elab[1], elab_p, strlen(elab_p));
    write (fd_elab[1], "\n", 1);
  }

  return;
}

/*
 * on line elaboration of data
 */

int local_elab (char **elab,
		struct port_area *p,
		int last_port,
		char * result,
		int fd_elab[]) {
  int j, nread, sel;
  char buf[20];
  fd_set pipe_fd;
  struct timeval t_calc;
  static int timeouts = MAX_TIME_OUT;

  /* send data */

  for (j = 0 ; j <= last_port ; j++) {
    sprintf (buf, "m%d=%6.6s;", j, (p+j)->inbuf+3);
    if (DDD) printf ("DDD - sending to child: <%s>\n", buf);
    write (fd_elab[1], buf, strlen(buf));
  }

  /* send command */

  write (fd_elab[1], *elab, strlen(*elab));
  write (fd_elab[1], "\n", 1);

  if (DD) printf ("DD - going to read from child ...\n");

  /* wait for results (up to 0.1 sec) */

  FD_ZERO (&pipe_fd);
  FD_SET (fd_elab[2], &pipe_fd);
  t_calc.tv_sec = 0;
  t_calc.tv_usec = 100000;
  usleep(10000);

  sel = select(fd_elab[2]+1, &pipe_fd, NULL, NULL, &t_calc);

  if (sel > 0) {

    nread = read (fd_elab[2], result, 40);

    if (nread == MAX_RESULT) {
      fprintf (stderr, "### %s\n",
	       "Result string is too long. Calculations disabled!");
      nread = 0;
      *elab = 0;
    }

    if (nread > 0) {
      for (j = 0; j < nread ; j++) {      /* sanity check for result string */
	if (! isprint(result[j])) {
	  if (result[j] != '\n')
	    fprintf (stderr, "### Bad character from calculator: %o at %d\n",
		     result[j], j);
	  result[j] = ' ';
	}
      }
    } else if (nread == -1) {
      perror ("### Failed reading results from calculator");
      nread = 0;
    }

  } else if (sel == 0) {

    fprintf (stderr, "### Timeout while reading from calculator\n");
    nread = 0;
    if (timeouts-- <= 0) {
      fprintf (stderr, "### Too many timeouts. Calculation disabled!\n");
      *elab = 0;
    }

  } else {
    perror ("### Failed waiting calculator results. Calculation disabled!");
    *elab = 0;
    nread = 0;
  }

  result[nread] = '\0';
  if (DD) printf ("DD - received: %d <%s>\n", nread, result);

  return nread;
}

/*
 * write_to_file
 */

void write_to_file (int fd, char * string) {
  write (fd, "# ", 2);
  write (fd, string, strlen(string));
  write (fd, "\n", 1);
}

/*
 *   "the main"
 */

int main (int argc, char *argv[]) {

  struct port_area p[MAX_PORTS];
  int last_port = 0, mask, mask_prototype = 0;
  struct sigaction act;
  int j, l, stamp_s=0, stamp_t=0, file_out=0, file_only=0, sel;
  int out_fd;
  int fd_elab[4]={-1,-1,-1,-1};     /* descriptors for pipe to calculator */
  fd_set rfds;
  char c, *out_file=default_out, *heading = 0;
  char *elab = 0, *elab_p = 0;
  struct timeval request, tb, ts;         /* data structures for timing */
  struct timeval interval, begin, future, elapsed, now;
  char outbuf[40 + MAX_RESULT + MAX_PORTS * (DATA_STRING + 1)];
  float delay = 1.;              /* delay before reading */
  int repeat = -1, nc;           /* repetition; -1 = endless */
  struct itimerval timer;

  /*  printf ("### marsh - Copyright (C) 2005 Marcello Carla' ###\n"); */

  p[0].port = default_port;

  /* parse command line options */

#define display \
"\nmarsh - get readings from a MAS345 multimeter via RS232 line\n"\
"\n"\
"Usage: marsh [OPTION]\n"\
"\n"\
" -l   (string) line to be used (default: /dev/ttyS0)\n"\
" -m   (string) one more line for one more multimeter\n"\
" -i   (float) time interval between readings (sec.)\n"\
" -r   (int) measure repetition count (default: endless)\n"\
" -s   add time stamp\n"\
" -t   add elapsed time (sec.)\n"\
" -o   (filename) write output to file\n"\
" -f   (filename) write output **only** to file\n"\
" -c   (string) insert comment line into file head\n"\
" -e   (string) on line calculations with 'bc'; readings are m0, m1, ... \n"\
" -p   (string) preset string for external calculator\n"\
" -d   activate verbose debug (more d's more debug) \n"\
" -h   print this help and exit\n"

  opterr = 0;
  while ((c = getopt (argc, argv, "di:l:m:r:hsto::f::c:e:p:")) != -1)
    switch (c)
      {
      case 'd':
	if (D && DD) DDD = 1;
	if (D) DD = 1;
	D = 1;
	if (!DD) printf ("Debug now active!\n");
	break;
      case 'i':
	if (sscanf(optarg, "%f" , &delay) != 1) {
	  fprintf (stderr, "### Bad argument for time delay: <%s>\n", optarg);
	  exit (-1);
	}
	break;
      case 'l':
	p[0].port = optarg;
        if (D) printf ("D - selected port <%s>\n", p[0].port);
	break;
      case 'm':
	if (D) printf ("D - adding one more multimeter on line <%s>\n",
		       optarg);
	if (++last_port == MAX_PORTS) {
	  fprintf (stderr, "%s%d%s\n", "### Only ", MAX_PORTS,
		   " ports allowed. Modify MAX_PORTS and recompile.");
	  exit (-1);
	}
	p[last_port].port = optarg;
	break;
      case 'r':
	repeat = atoi(optarg);
	if (repeat < 0) {
	  fprintf (stderr, "%s%d%s\n",
		  "### Warning - negative repetition count <", repeat,
		   "> taken as 'endless loop'");
	  repeat = -1;
	}
	if (D) printf("D - repetition count: %d\n", repeat);
	break;
      case 's':
	stamp_s = 1;
	break;
      case 't':
	stamp_t = 1;
	break;
      case 'f':
	file_only = 1;
      case 'o':
	file_out = 1;
	if (optarg) out_file = optarg;
	if (D) printf ("D - data output to file <%s>\n", out_file);
	break;
      case 'c':
	heading = optarg;
	break;
      case 'e':
	elab = optarg;
	if (D) printf ("D - process data with: <%s>\n", elab);
	break;
      case 'p':
	elab_p = optarg;
	break;
      case 'h':
      case '?':
      default:
	if (optopt != '?') {
	  fprintf (stderr, "### Unknown option: -%c.\n", optopt);
	  fprintf (stderr, display); exit (-1);
	}
	fprintf (stderr, display);
	exit (0);
      }
  if (optind != argc) {
    for (j = optind; j < argc; j++) {
      fprintf (stderr, "### Unknown argument: <%s>\n", argv[j]);
    }
    fprintf (stderr, display);
    exit (-1);
  }

  /* register to catch stop signals and force clean exit; 
     register SIGALRM to return after pause() */

  act.sa_handler = catch_the_signal;
  sigemptyset(&act.sa_mask);    /* exclude all signals from block mask */
  act.sa_flags = 0;             /* behaviour flag: none */

  if (sigaction(SIGINT, &act, 0) || sigaction(SIGQUIT, &act, 0) ||
      sigaction(SIGTERM, &act, 0) || sigaction(SIGALRM, &act, 0) ||
      sigaction(SIGHUP, &act, 0) || sigaction(SIGCHLD, &act, 0)) {
    perror ("### Error while registering signal handler");
    exit (-1);
  }

  if (D) printf ("D - registered signal handlers - starting calculator\n");

  /* start the calculator, if requested */

  if (elab)
    start_calc (&elab, elab_p, fd_elab);
  if (D) printf ("D - started calculator with i/o %d %d\n",
		 fd_elab[1], fd_elab[2]);

  /* build the 'interval' structure from 'delay' */

  interval.tv_sec = delay;
  interval.tv_usec = (delay - interval.tv_sec) * 1e6;
  if (D) printf("D - time delay: %f = %ld sec + %ld usec\n",
		delay, interval.tv_sec, interval.tv_usec);

  /* open and setup the lines */

  bzero (&newtio, sizeof(newtio));          /* setup termios structure */

  newtio.c_cflag = CS7 | CSTOPB | CLOCAL | CREAD;
  newtio.c_cc[VMIN] = 14;

  cfsetospeed (&newtio, B600);
  cfsetispeed (&newtio, B600);


  for (l = 0 ; l <= last_port ; l++ ) {     /* open lines and set param's */
    open_setup_line (p+l);
    mask_prototype |= p[l].mask = 1 << l;
  }

  /* open output file (if any) */

  if (file_out) {
    if ((out_fd =
	 open (out_file, O_WRONLY | O_CREAT | O_EXCL, 0644)) == -1) {
      fprintf (stderr, "### Unable to open file <%s>: %s\n",
	      out_file, strerror(errno));
      exit (-1);
    }
    gettimeofday (&begin,NULL);

    nc = time_stamp (outbuf, &begin);

    write_to_file (out_fd, outbuf);
    if (heading) write_to_file (out_fd, heading);
    
    for (l = 0 ; l <= last_port ; l++ ) write_to_file (out_fd, p[l].port);

    if (elab) write_to_file (out_fd, elab);
    if (elab_p) write_to_file (out_fd, elab_p);
  }

  /* start the time count */

  timer.it_interval.tv_sec = 0;
  timer.it_interval.tv_usec = 0;

  gettimeofday (&begin,NULL);
  future.tv_sec = begin.tv_sec;
  future.tv_usec = begin.tv_usec;

  /* main loop - here readings are taken */

  while (repeat>0 ? repeat-- : repeat) {

    /* send a solicit character to each active line and clear the buffers */

    mask = mask_prototype;
    gettimeofday (&request, NULL);
    for (l = 0 ; l <= last_port ; l++ ) {
      write (p[l].fd, "\n", 1);
      p[l].inbuf[0] = '\0';
      p[l].j = 0;
    }
    if (DD) printf ("DD - count %d   mask = %o\n", repeat, mask);

    while (mask) {

      FD_ZERO (&rfds);             /* prepare the select() data structure */
      for (l = 0 ; l <= last_port ; l++ ) FD_SET (p[l].fd, &rfds);
      ts.tv_sec = 1;
      ts.tv_usec = 0;

      if ((sel = select (p[last_port].fd+1, &rfds, NULL, NULL, &ts)) > 0) {

	for (l = 0 ; l <= last_port ; l++ ) {         /* for each port ... */

	  if (FD_ISSET(p[l].fd, &rfds)) {             /* ... if ready ...  */

                                   /* ... check for unwanted characters ...*/
	    if (p[l].mask & ~mask) {
	      tcflush (p[l].fd, TCIOFLUSH);
	      fprintf (stderr,
		       "### Spurious characters on line %d - flushed!\n", l);
	    }

	    p[l].j += read (p[l].fd, p[l].inbuf + p[l].j,
			    DATA_STRING - p[l].j);        /*  ... and read */
	    if (DDD) printf ("DDD - port %d byte %d: %o\n",
			     l, p[l].j, *(p[l].inbuf + p[l].j - 1));

	    if (p[l].j == DATA_STRING) {      /* if string is complete ... */

                                          /* ... and properly terminated ... */
	      if (*(p[l].inbuf + p[l].j - 1) == '\r') {
		mask &= ~p[l].mask;                        /* ... close it */
		*(p[l].inbuf + p[l].j - 1) = ' ';

	      } else {            /* otherwise complain and flush the line */

		fprintf (stderr,
			 "### Error on input %d: <%s>\n", l, p[l].inbuf);
		tcflush (p[l].fd, TCIOFLUSH);
	      }
	    }
	  }
	}

      } else if (sel == 0) {
	fprintf (stderr, "### Time out on input on:\n");
	for (l = 0 ; l <= last_port ; l++ ) {
	  if (mask & p[l].mask) {
	    *(p[l].inbuf + p[l].j) = '\0';
	    fprintf (stderr, "   port %d after %d bytes\n", l, p[l].j);
	    tcflush (p[l].fd, TCIOFLUSH);
	  }
	}
	break;
      } else {
	perror ("### Unable to select()");
	exit (-1);
      }
    }

    for (l = 0 ; l <= last_port ; l++ ) {

      for (j = 0; j < p[l].j ; j++) {      /* sanity check for result string */
	if (! isprint(p[l].inbuf[j])) {
	  fprintf (stderr, "### Bad character from calculator: %o at %d\n",
		   p[l].inbuf[j], j);
	  p[l].inbuf[j] = ' ';
	  mask |= ~p[l].mask;
	}
      }
      if (!strncmp(p[l].inbuf+5, ".OL", 3) ||
	  !strncmp(p[l].inbuf+5, "O.L", 3) ||
	  !strncmp(p[l].inbuf+5, "OL.", 3)) {
	fprintf (stderr, "### Measurement overflow!\n");
	mask |= ~p[l].mask;
      }
      if (DD) {
	printf ("DD - line: %d   mode:<%2.2s>  value:<%6.6s>  unit:<%4.4s>\n", 
		l, p[l].inbuf, p[l].inbuf+3, p[l].inbuf+9);
      }
    }

    /* write time section to output bufer */

    gettimeofday (&tb,NULL);

    if (mask) sprintf (outbuf, "#");     /* there is some error */
    else sprintf (outbuf, " ");
    nc = 1;

    if (stamp_s) nc += time_stamp(outbuf + nc, &request);

    if (stamp_t) {
      time_diff(&request, &begin, &elapsed);
      nc += sprintf (outbuf+nc, "%10.1f  ",
			 elapsed.tv_sec+elapsed.tv_usec/1.e6);
    }

    if (DD) printf ("DD - stamp:<%d> <%s>\n", nc, outbuf);

    /* write data section */

    for (l = 0 ; l <= last_port ; l++ ) {

      p[l].inbuf[p[l].j] = '\0';
      if (DD) printf("DD - line %d: <%s>\n", l, p[l].inbuf);
      if ((mask & p[l].mask) == 0) {
	nc += sprintf (outbuf+nc, "   %s", p[l].inbuf);
      } else
	nc += sprintf (outbuf+nc, "                 ");
    }

    /* execute on line data elaboration */

    if (DD) printf ("DD - starting optional elaboration\n");
    if (elab && !mask) {
      outbuf[nc++] = ' '; outbuf[nc++] = ' ';
      nc += l = local_elab (&elab, p, last_port, outbuf+nc, fd_elab);
      if (l == 0 && elab) outbuf[0] = '#';    /* mark for bad record */
    }
    nc += sprintf (outbuf+nc, "\n");

    /* write output buffer to console & file */

    if (!file_only) printf ("%s", outbuf);
    if (file_out) write (out_fd, outbuf, nc);

    /* schedule next operation */

    time_add (&future, &interval, &future);       /* when again? */
    gettimeofday (&now, NULL);
    time_diff (&future, &now, &timer.it_value);   /* time to wait */
    if (DD) printf ("going to wait for %ld sec + %ld usec\n",
		   timer.it_value.tv_sec, timer.it_value.tv_usec);

    if (timer.it_value.tv_sec < 0) {

#define  ErrSync "Error: synchronization lost - trying to resynchronize"

      if (file_out) write (out_fd, "# "ErrSync"\n", sizeof(ErrSync)+3);

      while (timer.it_value.tv_sec < 0) {
	if (file_only)
	  fprintf (stderr, "%s\n", ErrSync);
	else
	  printf ("# %s\n", ErrSync);

	time_add (&future, &interval, &future);
	time_diff (&future, &now, &timer.it_value);
      }
    }

    if (setitimer(ITIMER_REAL, &timer, NULL)) {
      perror ("### Couldn't start timer");
      exit (-1);
    }
    pause();                          /* wait for SIGALRM signal */
  }
  exit (0);
}
