/* 
   The class mtv::undefined<T> provides a non-intrusive way to mark an object
   of type T as undefined, preventing the access to a logically uninitialized
   or invalid object, even if that object is still existing and has been
   correctly constructed.

   Copyright (C) 2005,2006,2007,2008,2009,2010 Daniele de Rigo

   This file is part of Mastrave/C++.

   Mastrave/C++ is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.



   version: 0.1.7

*/

#ifndef MASTRAVE_UTIL_UNDEFINED_H
#define MASTRAVE_UTIL_UNDEFINED_H

#include <algorithm>  // std::swap
#include <sstream>    // std::stringstream
#include <string>

#include <stdexcept>  // std::invalid_argument


namespace mtv{



   const char undefined_version_[] = "0.1.7";



   template <typename T>
   class undefined{



   public:

      typedef std::invalid_argument exception_type ;
      typedef T                     value_type     ;



   protected:

      bool         is_defined_  ;
      T            value_       ;
      std::string  description_ ;


/*
      void* that ( ) const
      {
         return (void*)this;
      }
*/


      void error ( const char          * file ,
                   const unsigned long   line ) const
      {
         std::stringstream s;
         s << "error: attempting to access an undefined variable ("
           << description_ << ")"
           << "\n( throw by '" << file << "' : " << line << " )";
         throw exception_type( s.str() );
      }



      void swap ( undefined<T> & other )
      {
         std::swap( is_defined_  , other.is_defined_  );
         std::swap( value_       , other.value_       );
         std::swap( description_ , other.description_ );
      }



      void set_definition_status ( bool status )
      {
         is_defined_ = status;
      }



   public:

      undefined ( const std::string & descr = "unknown" ) :
         is_defined_(  false ),
         description_( descr )
      { }


   /*
      // removed because it is contradictory with the semantic of an
      // undefined object...
      undefined( const T           & v                  ,
                 const std::string & descr( "unknown" ) ) :
         is_defined_(  true  ),
         value_(       v     ),
         description_( descr )
      {}
   */



      bool is_defined ( ) const
      {
         return is_defined_;
      }



      T get ( ) const
      {
         if( this->is_defined() ) {
            return value_;
         }
	 else {
            error( __FILE__ , __LINE__ );
         }
      }



      operator T ( ) const
      {
         return this->get();
      }



      T& operator = ( const T & v )
      {
         this->set( v );
         return value_;
      }



      undefined<T>& operator = ( const undefined<T> & other )
      {
         undefined<T> temp( other );
         this->swap( temp );
         return *this;
      }



      void set ( const T & v )
      {
         value_ = v;
         this->set_definition_status( true );
      }



      void set ( const T           & v     ,
                 const std::string & descr )
      {
         this->set( v );
         this->set_description( descr );
      }



      void set_description( const std::string & descr )
      {
         description_ = descr;
      }



      std::string get_description ( ) const
      {
         return description_;
      }



      void undefine ( )
      {
         this->set_definition_status( false );
      }


   };


}

#endif

