function elements = elem( matrix, rows, cols )
% Copyright (C) 2005,2006,2007,2008,2009,2010 Daniele de Rigo 
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% elements = elem( matrix, rows, cols )
% 
% Utility function for enabling on the fly element extraction, even from a
% temporary variable which is unnamed -- and therefore cannot be explicitly
% invoked without assigning it to another variable or passing it to a
% function -- and whose lifetime ends after the evaluation of the command
% which created it.
%
%
%
% Input arguments:
%
% <matrix>         ::matrix::
%                  scalar, vector or matrix of objects
%
% <rows>           ::index::
%                  scalar, vector or matrix of row indexes
%
% <cols>           ::index,string::
%                  scalar, vector or matrix of column indexes, or ascii flag
%                  indicating the dimesnion of the indexing to be considered:
%                    'r'  : rows 
%                    'c'  : columns 
% 
%
%
% A brief example should explain it better than a formal argument descriprion.
% Given a matrix:
% 
%    A = rand( 10, 7 )
% 
% then the left statement gives the same result as the right one:     
% 
%    elem( A, 2:3, 1:6 )  ==  A( 2:3, 1:6 )
%    elem( A, 2:3 )       ==  A( 2:3 )
%    elem( A )            ==  A
% 
% This may be useful to provide Matlab the capability to manage on-the-fly
% temporary variables emulating the native feature of GNU Octave:
% 
%    A      = rand(5);
%    thirdA = sort(sort(A)(3,:))(3);             % GNU Octave: ok   Matlab: error
% 
% Adding auxiliary variables can avoid this problem:
% 
%    sortA            = sort(A);
%    sort_third_row_A = sort(sortA(3,:));        % GNU Octave: ok   Matlab: ok
%    thirdA           = sort_third_row_A(3);     % GNU Octave: ok   Matlab: ok
%     
% otherwise, it is possible to use:
% 
%    thirdA = elem(sort(elem(sort(A),3,1:5)),3); % GNU Octave: ok   Matlab: ok
% 
%
% 
% -------- Special indexing --------
%
% Given a matrix of indexes which row (or column) size is the same as the 
% row (or column) size of A:
%  
%    [rA,cA] = size(A);   n = ceil(rand(1)*(min(rA,cA)-2))+1
%    row_idx = ceil( rand( rA , n )* cA ) 
%    col_idx = ceil( rand( n , cA )* rA )
% 
% then the left statements give the same result as the right ones:     
% 
%    elem( A, col_idx , 'c' )  ==  A( col_idx + repmat([0:(cA-1)]*rA,n,1)  )
%    elem( A, row_idx , 'r' )  ==  A( repmat([1:rA]',1,n) + (row_idx-1)*rA  ) 
% 
%
% Example 1:
%      
%    A   =     [  0.68    0.97    0.23    0.26    0.12
%                 0.47    0.96    0.67    0.91    0.31
%                 0.17    0.36    0.58    0.58    0.25
%                 0.97    0.50    0.83   [0.82]-+ 0.91
%                 0.52    0.66    0.19    0.67  | 0.94  ]
%                                               |
%    col_idx = [    1       4       3      (4)--+   5
%                   3       1       4       2   |   1   ]
%                                               |
%    elem( A, col_idx ,  'c'  )                 |
%                                               |
%    ans =        0.68    0.50    0.58   [0.82]-+ 0.94
%                 0.17    0.97    0.83    0.91    0.12
%      
%    elem( A, col_idx ,  'c'  ) == A( col_idx + [ 0 5 10 15 20
%                                                 0 5 10 15 20 ] )
%      
% Example 2: given a matrix of data with some invalid data, compute the median
% for each row and column.
% 
%    Y=rand(10,7);  Y(rand(size(Y))>.5)=NaN  % some invalid data
%    ok=isfinite(Y)                          % true if valid data
% 
% Classical resolution (with interpreted 'for' loop ):
% 
%    median_col_Y=zeros(1,7); 
%    for i=1:7 
%       median_col_Y(i)=median(  Y( find(ok(:,i)) , i )  );
%    end 
%    median_row_Y=zeros(10,1); 
%    for i=1:10 
%       median_row_Y(i)=median(  Y( i , find(ok(i,:)) )  );
%    end 
% 
% Alternative solution:
% 
%    c_idx = (sum(ok ) +1)/2;    r_idx = (sum(ok')'+1)/2;
%    median_col_Y2 = mean(elem(sort(Y ) ,[floor(c_idx) ; ceil(c_idx)], 'c' ) )
%    median_row_Y2 = mean(elem(sort(Y')',[floor(r_idx)   ceil(r_idx)], 'r' )')'
% 
% The results are identical:
% 
%    median_col_Y2 == median_col_Y
%    median_row_Y2 == median_row_Y
% 
%
%
% version: 0.4.5

where = sprintf(  '(in function %s)'  , mfilename );


switch(nargin)
   case 1
      check_is( matrix ,  'matrix'  ,                                     ...
         '%s the first argument must be a matrix'  ,                      ...
         where                                                            ...
      );
      elements = matrix;
      return
   case 2
      check_is( matrix ,  'matrix'  ,                                     ...
         '%s the first argument must be a matrix'  ,                      ...
         where                                                            ...
      );
      check_is( rows ,  'index'  ,                                        ...
         '%s the second argument must be a matrix of valid indexes'  ,    ...
         where                                                            ...
     );
      check_is( max(rows) <= numel(matrix) ,  'true'  ,                   ...
         '%s an index of the second argument exceeds the <matrix> size' , ...
         where                                                            ...
      );
      elements = matrix(rows);
      return
   case 3 
      check_is( matrix ,  'matrix'  ,                                     ...
         '%s the first argument must be a matrix'  ,                      ...
         where                                                            ...
      );
      check_is( rows ,  'index'  ,                                        ...
         '%s the second argument must be a matrix of valid indexes'  ,    ...
         where                                                            ...
     );
      if( ischar(cols) )
         if(    size(rows,1)==size(matrix,1) && strcmp( cols,  'r'  ) )
            elements = matrix(                                            ...
                repmat( [1:size(matrix,1)]', 1, size( rows ,2)  ) +       ...
                   (rows-1) * size(matrix,1)                              ...
                );
            return
         elseif(size(rows,2)==size(matrix,2) && strcmp( cols,  'c'  ) )
            elements = matrix(                                            ...
                rows + repmat(                                            ...
                   [1:size(matrix,1):prod(size(matrix))]-1,               ...
                   size(rows,1) , 1)                                      ...
                );
            return        
         else
            elements = matrix(rows, cols);
            return
         end
      else   
         check_is( cols ,  'index'  ,                                     ...
            '%s the third argument must be a matrix of valid indexes'  ,  ...
            where                                                         ...
         );
         elements = matrix( rows, cols );
         return
      end
   otherwise
      fprintf( 2,  'Usage: elem( matrix, rows, cols )\n'  );
      error(  ' '  );
end


% Local Variables:
% mode:mastrave
% End:


