function [ file_path , file_name , bytes , attr ] = fileinfo( file_pattern )
% Copyright (C) 2005,2006,2007,2008 Daniele de Rigo 
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% [ file_path , file_name , bytes , attr ] = fileinfo( file_pattern )
% 
% Given a string <file_pattern> (that may contain wildcard characters), returns
% - the <file_path> cell-array of strings each of them representing the
%   full path of the corresponding matched file;
% - the <file_name> cell-array of strings each of them representing the
%   name of the corresponding matched file;
% - the vector of integers <bytes> showing the amount of bytes allocated
%   to each file;
% - the structure <attr> whose fields consist of vectors of logicals being
%   true when the corresponding attribute belong to a matched file, false
%   otherwise; the checked attributes are the followings:
%      - directory: for each element of <file_path> the corresponding row of
%        the field <is_dir> is true when the element is a directory, false
%        otherwise;
%      - readability: the true values of the field <is_writ> indicate the
%        readabale elements of <file_path>;
%      - writability: the true values of the field <is_writ> indicate the
%        writabale elements of <file_path>;
%      - executability: the true values of the field <is_exec> indicate the
%        executable elements of <file_path>;
%
%
% Input arguments:
% 
% <file_pattern>     ::string::
%                    string constraining the path and the name of the files to
%                    match: it may contain wildcard characters
%
%
% Examples of usage:
%
%    filelist = fileinfo( '*file*.m' )
%
%    [ file_path , file_name , bytes , attr ] = fileinfo( '../{,*/}*file*' );
%    format   = sprintf( '%%10d   %%d   %%d   %%d   %%d   %%-%ds  %%s\n' , ...
%                             max( cellfun( 'prodofsize' , file_name ) ) )
%    fprintf( 1 , '     bytes  dir  r   w   x\n' );
%    for i=1:numel(bytes)
%       fprintf(                                              ...
%          1              , format         , bytes(i)       , ...
%          attr.is_dir(i) , attr.is_read(i), attr.is_writ(i), ...
%          attr.is_exec(i), file_name{i}   , file_path{i}     ...
%       );
%    end
%
%
% version: 0.6.4

where = sprintf(  '(in function %s)'  , mfilename );

% mastrave-kernel: this function is used by function_decompose that is used by
% check_nargin, therefore the test for the allowed number of input arguments
% is performed without invoking check_nargin

usage_msg = sprintf( [                                    ...
   'Usage: [ file_path , file_name, is_dir , bytes ] = '  ...
   'fileinfo( file_pattern )\n'                           ...
] );

if nargin<1
   fprintf( 2,  'Error: not enough input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end
if nargin>1
   fprintf( 2,  'Error: too many input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end

check_is(                                                      ...
   file_pattern ,  'string'                                  , ...
   '%s the first argument <file_pattern> must be a string.'  , ...
   where                                                       ...
);


file_regexp  = file_pattern;
file_regexp  = strrep( file_regexp , '.' , '\.'    );
file_regexp  = strrep( file_regexp , '$' , '\$'    );
file_regexp  = strrep( file_regexp , '^' , '\^'    );
file_regexp  = strrep( file_regexp , ',' , '\|'    );
file_regexp  = strrep( file_regexp , '{' , '\('    );
file_regexp  = strrep( file_regexp , '}' , '\)'    );
file_regexp  = strrep( file_regexp , '?' , '[^/]'  );
file_regexp  = strrep( file_regexp , '*' , '[^/]*' );

file_basedir = regexprep( file_pattern , '^([./]*).*$' , '$1' );
% ls equivalent options:
%    -1     list one file per line
%    -p     append / indicator to directories
%    -d     list directory entries instead of their contents, and
%           do not dereferende symbolic links
base_command = sprintf(                        ...
   'find %s -regex "^%s$" '                  , ...
   file_basedir,                               ...
   file_regexp                                 ...
);
[ status, result ] = unix(                   ...
   sprintf(                                  ...
      '%s 2> /dev/null'                    , ...
      base_command                           ...
   )                                         ...
);
file_path = result;

if numel( file_path ) %  if <file_pattern> matches some existing file
   endline   = sprintf( '\n' );
   file_name = regexprep(                                         ...
      file_path ,                                                 ...
      [ '(^|' endline ')[^' endline ']*[/]([^/' endline ']+)' ] , ...
      '$1$2'                                                      ...
   );
   file_name = line2cell( file_name );
   file_path = line2cell( file_path );
   [ status, result ] = unix(                                ...
      sprintf(                                               ...
         ...%             uid  gid  bytes
         ...% permis__      |   |   |     __ file type
         ...%         \     |   |   |    /
         ...%          %m  %U  %G  %s  %y
         '%s -printf "%%m %%U %%G %%s %%y\n" 2> /dev/null' , ...
         base_command                                        ...
      )                                                      ...
   );
   %                                      uid  gid  bytes
   %                   permissions __        |  |  |     __ file type
   %                                 \       |  |  |    /
   %                                 %c%c%c %d %d %d %c
   %                                  1 2 3  4  5  6  7
   result       = sscanf(  result , '%c%c%c %d %d %d %c\n' );
   result       = reshape( result , 7 , [] )';
   permission   = result( : , 1:3 ) - '0';
   uid          = result( : , 4 );
   gid          = result( : , 5 );
   bytes        = result( : , 6 );
   attr.is_dir  = result( : , 7 ) == 'd'-0;

   [ status, result ] = unix( 'id -u' );
   myuid        = sscanf( result , '%d' );
   [ status, result ] = unix( 'id -G' );
   mygid        = sscanf( result , '%d' );

   same_uid     = find(  uid == myuid );
   same_gid     = mfind( gid ,  mygid );

   mypermission = permission( : , 3 );
   mypermission(    same_gid ) = bitor( ...
      mypermission( same_gid )        , ...
      permission(   same_gid , 2 )      ...
   );
   mypermission(    same_uid ) = bitor( ...
      mypermission( same_uid )        , ...
      permission(   same_uid , 1 )      ...
   );,
   %                               4 <-- readability mask
   attr.is_read = logical( bitand( 4, mypermission ) );
   %                               2 <-- writability mask
   attr.is_writ = logical( bitand( 2, mypermission ) );
   %                               1 <-- executability mask
   attr.is_exec = logical( bitand( 1, mypermission ) );
else
   file_path    = {};
   file_name    = {};
   bytes        = [];
   attr.is_dir  = [];
   attr.is_read = [];
   attr.is_writ = [];
   attr.is_exec = [];
end


function cells = line2cell( txt )
   endline = sprintf( '\n' );
   if txt( end )  ~= endline
      txt( end+1 ) = endline;
   end
   lid     = ( txt == endline );
   id      = find( lid );
   txt     = txt( ~lid );
   cells   = mat2cell( txt, 1, diff([0 id])-1 );


% Local Variables:
% mode:mastrave
% End:


