function [interval_idx, value_idx] = find_disjoint_interval( disjoint_intervals, values, inclusion )
% Copyright (C) 2006,2007,2008,2009,2010 Daniele de Rigo 
% 
% This file is part of Mastrave.
%
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% [interval_idx, value_idx] = find_disjoint_interval( disjoint_intervals ,
%                                                     values             ,
%                                                     inclusion          )
%
% Given the vectors <a> and <b> with the same size and satisfying the 
% following conditions:
%      all( a            <       b        )
%      all( a            == sort(a)       )
%      all( b            == sort(b)       )
%      all( b(1:end-1)   <=      a(2:end) )
%      disjoint_intervals = [ a b ]
% find_disjoint_interval finds the index <i> of <a> and <b> and the
% index <j> of another vector <values> such that 
% 
%      a(i) <= values(j) <  b(i)
% 
% if <inclusion> == '[)' (or if it's omitted), otherwise
% 
%      a(i) <  values(j) <= b(i)
% 
% if <inclusion> == '(]' .
% 
% The sets of indexes <i> and <j> are returned respectively in the
% vectors <interval_idx> and <value_idx>.     
% 
%
%
% Input arguments:
%
% <disjoint_intervals>    ::matrix,real::
%                         2-columns matrix of non-overlapping real
%                         intervals sorted in ascending order
%
% <values>                ::real::
%                         vector or matrix of values for which determine
%                         to which interval each of them belong 
%
% <inclusion>             ::string::
%                         set which interval border check has to be
%                         performed.  If omitted, default option is '[)'.
%                         Valid options are:
%
%                               option    |      meaning
%                            -------------+--------------------------------
%                              '[)'       | a(i) <= values(j) <  b(i)
%                            -------------+--------------------------------
%                              '(]'       | a(i) <  values(j) <= b(i)
%
%
%
% Example of usage:
%
%    intvs = [ 0.3 0.6;
%              0.6 0.7;
%              0.7 0.8;
%              0.8 1.2;
%              2.1 2.7 ]
%    vals = [ 0.2 0.5 0.7 0.8 2.0 ]'
%    
%    [ intv_idx, val_idx ] = find_disjoint_interval( intvs, vals );
%    [ intvs(intv_idx,1) vals(val_idx) intvs(intv_idx,2) ]
%    
%    [ intv_idx, val_idx ] = find_disjoint_interval( intvs, vals , '(]' );
%    [ intvs(intv_idx,1) vals(val_idx) intvs(intv_idx,2) ]
%
%
%
% Correctness test:
%
% case 1 (medium size)
% 
%    N      = 2000;
%    interv = sort(randn(2*N,1)); interv = reshape(interv,2,N)';
%    n      = ceil( abs(randn(1))*N )
%    vals   = randn(n,1); 
%    tic
%    [r,c]  = find( repmat(vals,1,N) >= repmat(interv(:,1)',n,1) & ...
%                   repmat(vals,1,N) <  repmat(interv(:,2)',n,1)  );
%    old_method = toc
%                   
%    tic
%    [iidx, vidx] = find_disjoint_interval( interv, vals );
%    new_method = toc
%    
%    isequal( sort(iidx) ,      c  )
%    isequal(      vidx  , sort(r) )
%      
%    fprintf( 1 , [ '<find_disjoint_interval> with option [) ' ...
%       'is %g times faster\n' ] , old_method/new_method );
%    
%    
%    interv = sort(randn(2*N,1)); interv = reshape(interv,2,N)';
%    n      = ceil( abs(randn(1))*N )
%    vals   = randn(n,1); 
%    tic
%    [r,c]  = find( repmat(vals,1,N) >  repmat(interv(:,1)',n,1) & ...
%                   repmat(vals,1,N) <= repmat(interv(:,2)',n,1)  );
%    old_method = toc
%                     
%    tic
%    [iidx, vidx] = find_disjoint_interval( interv, vals , '(]');
%    new_method = toc
%    
%    isequal( sort(iidx) ,      c  )
%    isequal(      vidx  , sort(r) )
%     
%    fprintf( 1 , [ '<find_disjoint_interval> with option (] ' ...
%       'is %g times faster\n' ] , old_method/new_method );
%
%
% case 2 (huge size: repmat cannot be used, due to the risk of memory
% exhaustion)
%
%    N      = 20000;
%    interv = sort(randn(2*N,1)); interv = reshape(interv,2,N)';
%    n      = ceil( abs(randn(1))*N )
%    vals   = randn(n,1); 
%    tic
%    r = []; c = [];
%    for i = 1:N
%       idx = find( vals >= interv(i,1) & vals < interv(i,2) );
%       r   = [r ; idx];
%       if( length(idx) )
%          c = [c ; zeros(size(idx))+i];
%       end
%    end
%    old_method = toc
%                     
%    tic
%    [iidx, vidx] = find_disjoint_interval( interv, vals );
%    new_method = toc
%      
%    isequal( sort(iidx) ,      c  )
%    isequal(      vidx  , sort(r) )
%     
%    fprintf( 1 , [ '<find_disjoint_interval> is %g ' ...
%       'times faster\n' ] , old_method/new_method );
% 
%
% case 3 (tiny size: performance plot)
% 
%    Ni   = 1:50;
%    olds = zeros(size(Ni));
%    news = zeros(size(Ni));
%    vals = {};
%    for i=1:length(Ni)
%       N = Ni(i);
%       interv = sort(randn(2*N,1)); interv = reshape(interv,2,N)'; 
%       n      = ceil( abs(randn(1000,1))*N ); 
%       for  j = 1:length(n) vals{j} = randn(n(j),1); end;
%         
%       tic;
%       for  j = 1:length(n)
%          [r,c]  = find( ...
%             repmat(vals{j},1,N) >= repmat(interv(:,1)',n(j),1) & ...
%             repmat(vals{j},1,N) <  repmat(interv(:,2)',n(j),1)   ...
%          );
%       end
%       olds(i) = toc;
%         
%       tic;
%       for  j = 1:length(n)
%          [iidx, vidx] = find_disjoint_interval( interv, vals{j} );
%       end
%       news(i) = toc;
%    end
%      
%    loglog( Ni, olds, 'r' , Ni, news, 'g' )
%
%
%
% version: 0.6.3

where        = sprintf(  '(in function %s)'  , mfilename );

interval_idx = []; 
value_idx    = []; 

usage_msg = sprintf( [                                                          ...
   'Usage:\n'                                                                   ...
   '[interval_idx, value_idx] = find_disjoint_interval( disjoint_intervals ,\n' ...
   '                                                    values             ,\n' ...
   '                                                    inclusion          )\n' ...
] );

if nargin<2
   fprintf( 2,  'Error: not enough input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end

if nargin < 3
   inclusion =  '[)'  ;
end

check_is( disjoint_intervals ,  'matrix'                             , ...
   [                                                                   ...
      '%s the first argument <disjoint_intervals> must be '            ...
      'a matrix.'                                                      ...
   ],                                                                  ...
   where                                                               ...
);

check_is( disjoint_intervals ,  'real'                               , ...
   [                                                                   ...
      '%s the first argument <disjoint_intervals> must be '            ...
      'of real numbers.'                                               ...
   ],                                                                  ...
   where                                                               ...
);

check_is( size(disjoint_intervals,2)==2 ,  'true'                    , ...
   [                                                                   ...
      '%s the first argument <disjoint_intervals> must be '            ...
      'a two-columns matrix.'                                          ...
   ],                                                                  ...
   where                                                               ...
);

check_is(                                                              ...
   all( diff( reshape(disjoint_intervals', [], 1) ) >=0 ) ,  'true'  , ...
   [                                                                   ...
      '%s the first argument <disjoint_intervals> must be a sorted '   ...
      'sequence of non-overlapping intervals'                          ...
   ],                                                                  ...
   where                                                               ...
);

check_is( values ,  'real'                                           , ...
   '%s the second argument <values> must be of reals'                , ...
   where                                                               ...
);

check_is( inclusion ,  'string'                                      , ...
   '%s the third argument <inclusion> must be a string'              , ...
   where                                                               ...
);


[interval_idx, value_idx] = find_disjoint_interval_( ...
   disjoint_intervals,                               ...
   values,                                           ...
   inclusion                                         ...
);

% Local Variables:
% mode:mastrave
% End:

