function [interval_idx, value_idx] = find_disjoint_interval_( disjoint_intervals, values, inclusion )
% Copyright (C) 2006,2007,2008,2009,2010 Daniele de Rigo 
% 
% This file is part of Mastrave.
%
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% [interval_idx, value_idx] = find_disjoint_interval_( disjoint_intervals ,
%                                                      values             ,
%                                                      inclusion          )
%
%    Warning: find_disjoint_interval_ should not be used in end-user code
%    because it deliberately skips input and output arguments checks and
%    other pre- and post-condition testing.
%    End-user code should use instead the find_disjoint_interval function
%    (without the ending underscore).
%
% Given the vectors <a> and <b> with the same size and satisfying the 
% following conditions:
%      all( a            <       b        )
%      all( a            == sort(a)       )
%      all( b            == sort(b)       )
%      all( b(1:end-1)   <=      a(2:end) )
%      disjoint_intervals = [ a b ]
% find_disjoint_interval_ finds the index <i> of <a> and <b> and the
% index <j> of another vector <values> such that 
% 
%      a(i) <= values(j) <  b(i)
% 
% if <inclusion> == '[)', otherwise
% 
%      a(i) <  values(j) <= b(i)
% 
% if <inclusion> == '(]' .
% 
% The sets of indexes <i> and <j> are returned respectively in the
% vectors <interval_idx> and <value_idx>.     
% 
%
%
% Input arguments:
%
% <disjoint_intervals>    ::matrix,real::
%                         2-columns matrix of non-overlapping real
%                         intervals sorted in ascending order
%
% <values>                ::real::
%                         vector or matrix of values for which determine
%                         to which interval each of them belong 
%
% <inclusion>             ::string::
%                         set which interval border check has to be
%                         performed.
%                         Valid options are:
%
%                               option    |      meaning
%                            -------------+--------------------------------
%                              '[)'       | a(i) <= values(j) <  b(i)
%                            -------------+--------------------------------
%                              '(]'       | a(i) <  values(j) <= b(i)
%
%
%
% Example of usage:
%
%    intvs = [ 0.3 0.6;
%              0.6 0.7;
%              0.7 0.8;
%              0.8 1.2;
%              2.1 2.7 ]
%    vals = [ 0.2 0.5 0.7 0.8 2.0 ]'
%    
%    [ intv_idx, val_idx ] = find_disjoint_interval_( intvs, vals , '[)' );
%    [ intvs(intv_idx,1) vals(val_idx) intvs(intv_idx,2) ]
%    
%    [ intv_idx, val_idx ] = find_disjoint_interval_( intvs, vals , '(]' );
%    [ intvs(intv_idx,1) vals(val_idx) intvs(intv_idx,2) ]
%
%
%
% Correctness test:
%
% case 1 (medium size)
% 
%    N      = 2000;
%    interv = sort(randn(2*N,1)); interv = reshape(interv,2,N)';
%    n      = ceil( abs(randn(1))*N )
%    vals   = randn(n,1); 
%    tic
%    [r,c]  = find( repmat(vals,1,N) >= repmat(interv(:,1)',n,1) & ...
%                   repmat(vals,1,N) <  repmat(interv(:,2)',n,1)  );
%    old_method = toc
%                   
%    tic
%    [iidx, vidx] = find_disjoint_interval_( interv, vals, '[)' );
%    new_method = toc
%    
%    isequal( sort(iidx) ,      c  )
%    isequal(      vidx  , sort(r) )
%    
%    fprintf( 1 , [ '<find_disjoint_interval_> with option [) ' ...
%       'is %g times faster\n' ] , old_method/new_method );
%    
%    
%    interv = sort(randn(2*N,1)); interv = reshape(interv,2,N)';
%    n      = ceil( abs(randn(1))*N )
%    vals   = randn(n,1); 
%    tic
%    [r,c]  = find( repmat(vals,1,N) >  repmat(interv(:,1)',n,1) & ...
%                   repmat(vals,1,N) <= repmat(interv(:,2)',n,1)  );
%    old_method = toc
%                   
%    tic
%    [iidx, vidx] = find_disjoint_interval_( interv, vals , '(]');
%    new_method = toc
%    
%    isequal( sort(iidx) ,      c  )
%    isequal(      vidx  , sort(r) )
%    
%    fprintf( 1 , [ '<find_disjoint_interval_> with option (] ' ...
%       'is %g times faster\n' ] , old_method/new_method );
%
%
% case 2 (huge size: repmat cannot be used, due to the risk of memory
% exhaustion)
%
%    N      = 20000;
%    interv = sort(randn(2*N,1)); interv = reshape(interv,2,N)';
%    n      = ceil( abs(randn(1))*N )
%    vals   = randn(n,1); 
%    tic
%    r = []; c = [];
%    for i = 1:N
%       idx = find( vals >= interv(i,1) & vals < interv(i,2) );
%       r   = [r ; idx];
%       if( length(idx) )
%          c = [c ; zeros(size(idx))+i];
%       end
%    end
%    old_method = toc
%                   
%    tic
%    [iidx, vidx] = find_disjoint_interval_( interv, vals, '[)' );
%    new_method = toc
%    
%    isequal( sort(iidx) ,      c  )
%    isequal(      vidx  , sort(r) )
%    
%    fprintf( 1 , [ '<find_disjoint_interval_> is %g ' ...
%       'times faster\n' ] , old_method/new_method );
% 
%
% case 3 (tiny size: performance plot)
% 
%    Ni   = 1:50;
%    olds = zeros(size(Ni));
%    news = zeros(size(Ni));
%    vals = {};
%    for i=1:length(Ni)
%       N = Ni(i);
%       interv = sort(randn(2*N,1)); interv = reshape(interv,2,N)'; 
%       n      = ceil( abs(randn(1000,1))*N ); 
%       for  j = 1:length(n) vals{j} = randn(n(j),1); end;
%       
%       tic;
%       for  j = 1:length(n)
%          [r,c]  = find( ...
%             repmat(vals{j},1,N) >= repmat(interv(:,1)',n(j),1) & ...
%             repmat(vals{j},1,N) <  repmat(interv(:,2)',n(j),1)   ...
%          );
%       end
%       olds(i) = toc;
%       
%       tic;
%       for  j = 1:length(n)
%          [iidx, vidx] = find_disjoint_interval_( interv, vals{j}, '[)' );
%       end
%       news(i) = toc;
%    end
%    
%    loglog( Ni, olds, 'r' , Ni, news, 'g' )
%
%
%
% version: 0.6.3

interval_idx = []; 
value_idx    = []; 

switch( inclusion )
   case  '[)'  
   case  '(]'  
      [ interval_idx, value_idx ] = find_disjoint_interval_(  ...
         -disjoint_intervals(end:-1:1,end:-1:1) ,             ...
         -values ,                                            ...
         '[)'                                                 ...
      );
      interval_idx = size(disjoint_intervals,1)+1-interval_idx;
      %value_idx   = value_idx(end:-1:1);
      return
   otherwise
   % default: '[)'
end


[values, s_idx]    = sort(values(:)); %u_idx(s_idx)=[1:length(values)]';
n_int              = size( disjoint_intervals, 1 );
n_val              = length( values );

from               = [ disjoint_intervals(:,1); values ];
from_pos           = [[1:n_int] -[1:n_val]]';
to                 = [ disjoint_intervals(:,2); values ];
to_pos             = [[1:n_int] -[1:n_val]]';
[ tmp, from_idx ]  = sort( from ); 
[ tmp, to_idx   ]  = sort( to   ); 

sorted_from_pos    = from_pos( from_idx ); 
sorted_to_pos      =   to_pos(   to_idx ); 

sx                 = [[sorted_from_pos;0] [0;sorted_to_pos]];

%for i=1:size(sx,1)
%   if sx(i,1)>0
%      fprintf(1,  ' %3d  %4.4g '  ,sx(i,1), disjoint_intervals(sx(i,1),1));
%   elseif sx(i,1)==0     
%      fprintf(1,  ' %3d    -  '  ,sx(i,1) );
%   else
%      fprintf(1,  ' %3d [%4.4g]'  ,sx(i,1), values(-sx(i,1)));
%   end
%   if sx(i,2)>0
%      fprintf(1,  ' %3d  %4.4g '  ,sx(i,2), disjoint_intervals(sx(i,2),2));
%   elseif sx(i,2)==0     
%      fprintf(1,  ' %3d    -  '  ,sx(i,2) );
%   else
%      fprintf(1,  ' %3d [%4.4g]'  ,sx(i,2), values(-sx(i,2)));
%   end
%   fprintf(1,  '\n'  );
%end;

sx_idx             = find( (~diff(sx')') & (sx(:,1)<0) );

value_idx          = s_idx(-sorted_from_pos( sx_idx ));
uv_idx             = zeros(max(value_idx),1); 
uv_idx(value_idx)  = 1:length(value_idx); 
uv_idx             = uv_idx( uv_idx ~= 0 );
value_idx          = value_idx( uv_idx );

from2intervals_idx = zeros(size(sorted_from_pos)); 
from2intervals_idx(sorted_from_pos>0) = 1;  
from2intervals_idx = cumsum(from2intervals_idx); 

%[sorted_from_pos from2intervals_idx]; 

interval_idx       = from2intervals_idx(sx_idx);
interval_idx       = interval_idx( uv_idx );

% Local Variables:
% mode:mastrave
% End:

