function txt=int2ordinal( num )
% Copyright (C) 2005,2006,2007 Daniele de Rigo 
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% txt = int2ordinal( num )
%
% Converts an integer into a string that represent its corresponding
% ordinal number.
% If the argument passed is a vector or a matrix of integers, returns a
% cell array of strings, with the same size.
%
% Input arguments:
%
% <num>              ::integer::
%                    matrix of integers
%
% Examples of usage:
%
%    int2ordinal( round(  randn)      )
%    int2ordinal( round(5*randn)      )
%    int2ordinal( round(5*randn(3,4)) )
%    int2ordinal(  -3:21   )
%    int2ordinal( [-3:21]' )
%
%
% version: 0.3.4

where = sprintf(  '(in function %s)'  , mfilename );

% mastrave-kernel: this function is used by check_is, therefore all the 
% precondition checks are performed without invoking check_is or check_nargin

if(nargin~=1)
   fprintf( 2,  'Error: expected 1 argument instead of %d\n'  , nargin );
   fprintf( 2,  'Usage: int2ordinal( num )\n'  );
   error(  ' '  );
end;


if( ~isnumeric(num) )
   fprintf( 2,[  'error %s: the argument must be numeric '  ...
                 'instead of type "%s"\n'                ] ,...
               where, class(num)  );
   error(  ' '  );
end;
if( any( floor(num(:)) ~= num(:) ) )
   fprintf( 2,[  'error %s: the argument must be an '  ...
                 'integer scalar or matrix\n'       ]  , where );
   error(  ' '  );
end;

%---begin of the old scalar version---
%
% if( length(num)~=1 )
%    disp(  'error: the argument must be scalar'  );
%    return;
% end;
% 
%---end of the old scalar version---


ordinal = {  'zeroth'   , ...
             'first'    , ...
             'second'   , ...
             'third'    , ...
             'fourth'   , ...
             'fifth'    , ...
             'sixth'    , ...
             'seventh'  , ...
             'eighth'   , ...
             'ninth'    };

txt  = cell(size(num));
num  = num(:);

idx  = num >=0 & num < 10;  % integers in [0,9]
% The deal (and deal_) function in octave fails with logical indexing of
% output cell-array.
% Suppose that:
%    n_tot  = numel( cellarray )
%    n_tot == numel( logical_indexes )
%    n_true = numel( find(logical_indexes) ) % n_true <= n_tot
% Use:
%    [ cellarray{ find(logical_indexes) } ] = deal_( ... ) % nargout: n_true
% instead of
%    [ cellarray{      logical_indexes  } ] = deal_( ... ) % nargout: n_tot
[txt{ find(idx) }] = deal_( ordinal{ num(idx)+1 } ); 

idx  = ~idx;                % integers out of [0,9]

idx1 = idx & mod(abs(num),10)==1;
idx2 = idx & mod(abs(num),10)==2;
idx3 = idx & mod(abs(num),10)==3;
idxN = idx & ~(idx1 | idx2 | idx3);

idx_i    = { idx1 idx2 idx3 idxN };
suffix_i = { 'st' 'nd' 'rd' 'th' };

% floor(log10(num))+1 is the number of digits: an additional char is
% required by negative numbers
    dig  = floor(log10(abs(num)+eps)) + 1 + (sign(num)<0);
max_dig  = max(dig);

%sprintf('%dst%*.0d|',[num(idx1) mdig-dig(idx1) zeros(sum(idx1),1)]')

for i=1:numel(idx_i)
   if any( idx_i{i} )
      txt_i    = repmat( ' ' , max_dig+2, sum( idx_i{i} ) );
      txt_i(:) = sprintf(  [  '%d'  suffix_i{i}  '%*.0d'  ] ,  ...
                           [ num(           idx_i{i} )      ,  ...
                             max_dig - dig( idx_i{i} )      ,  ...
                             zeros(    sum( idx_i{i} ) ,1 ) ]' ...
                        );
      txt_i    = cellstr(txt_i');
      [txt{ find( idx_i{i} ) }] = deal_( txt_i{:} );
   end
end

if numel( num )==1
   txt = txt{1};
end

%---begin of the old scalar version---
%
% txt = num2str(num);
%
% if num >=0 & num < 10
%    txt = ordinal{num+1};
% else
%    switch( mod(abs(num),10) )
%    case 1
%       txt = [txt  'st'  ];
%    case 2
%       txt = [txt  'nd'  ];
%    case 3
%       txt = [txt  'rd'  ];
%    otherwise
%       txt = [txt  'th'  ];
%    end
% end
% 
%---end of the old scalar version---


function varargout = deal_( varargin )
   varargout = varargin;


% Local Variables:
% mode:mastrave
% End:


