function multi_array = mat2multi( matrix, dim, siz )
% Copyright (C) 2005,2006,2007,2008,2009 Daniele de Rigo
%
% This file is part of Mastrave.
%
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
%
% ---------------------------------------------------------------------------
%
% multi_array = mat2multi( matrix, dim, siz )
%
% Some functions are defined for matrices but not for multidimensional arrays
% (md-arrays).
% mat2multi allows to create md-arrays from matrices which can be processed
% even with ill-expandable functions operating column by column.
% Column-wise operations by default apply to the columns of a matrix as if the
% matrix were a vector of column vectors.  This class of functions includes
% reductions (in the APL sense) such as sum(.), prod(.); and scans (in the APL
% sense) such as cumsum(.), cumprod(.).
% The aforementioned basic functions extend to md-arrays, however some
% user-defined functions may not allow the same.   mat2multi converts a
% matrix of size:.
%    [ n<dim>  prod([n1 .. n<dim>-1 n<dim>+1 nk]) ]
% into a k-dimensional array <multi_array> of size <siz>:
%    [ n1 ... n<dim> ... nk ]
% as if <multi_array> were a collection of vectors having n<dim> elements each
% and being stored in the nd-array along its <dim> dimension.
%
% The function multi2mat can produce from a given md-array a matrix suitable
% to be reversed by mat2multi.
%
%
%
% Input arguments:
%
% <matrix>           ::matrix,!sparse::
%                    numeric, charachter or cell-array matrix (whose number
%                    of dimensions cannot exceed 2)
%
% <dim>              ::scalar_natural_nonzero::
%                    dimension of <multi_array> to which each <matrix>
%                    column vector must belong
%
% <siz>              ::numel,row_vector::
%                    vector whose elements are the desired cardinality of
%                    each dimension of <multi_array>
%
%
%
%
% Examples of usage:
%
%    mda     = zeros( 3, 4, 2);
%    mda(:)  = 1:numel( mda )
%
%    dim     = 1  % 4x2 vectors of 3 elements each
%    multi2mat( mda , dim )
%
%    dim     = 2  % 3x2 vectors of 4 elements each
%    multi2mat( mda , dim )
%
%    dim     = 3  % 3x4 vectors of 2 elements each
%    multi2mat( mda , dim )
%
%    % reverse the conversion:
%    [ mat , siz ] = multi2mat( mda , dim );
%    mat2multi( mat , dim , siz )
%
%    % scan example using cumsum
%    scan1   = cumsum( mda , dim )
%    scan2   = mat2multi( cumsum( mat , 1 ) , dim , siz )
%    scan1  == scan2
%
%    % reduction example using sum
%    red1    = sum( mda , dim )
%    red_siz = siz; red_siz( dim ) = 1
%    red2    = mat2multi( sum( mat , 1 ) , dim , red_siz )
%    red1   == red2
%
%
%
% version: 0.2.3

where = sprintf(  '(in function %s)'  , mfilename );

usage_msg = sprintf( [                   ...
   'Usage:  multi_array = mat2multi( matrix, dim, siz )\n'  ...
] );

if nargin<3
   fprintf( 2,  'Error: not enough input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end

check_is( matrix ,  'matrix'                                 , ...
         '%s the first argument <matrix> must be a matrix.'  , ...
         where );

check_is( issparse(matrix) ,  'false'                        , ...
        [  '%s the first argument <matrix> cannot be a '       ...
           'sparse matrix.'                                 ], ...
          where );

check_is( dim ,  'scalar_natural_nonzero'          ,  ...
        [  '%s the second argument <dim> must be '    ...
           'a positive integer.\n'                 ], ...
          where );

check_is( siz ,  'row_vector'  , ...
        [ '%s the third argument <siz> must be a '    ...
          'row vector\n'                           ], ...
          where );

check_is( siz ,  'numel'  , ...
        [ '%s the third argument <siz> must be of '   ...
          'non negative values\n'                  ], ...
          where );

check_is( numel(matrix) == prod(siz),  'true'  ,              ...
        [ '%s the first argument <matrix> has %d elements '   ...
          'while the expected number of elements is the '     ...
          'product of the sizes <siz> of the <multi_array> '  ...
          'to return, which is %d.\n'                       ], ...
         where, numel(matrix), prod(siz) );

multi_array = mat2multi_( matrix, dim, siz );

% Local Variables:
% mode:mastrave
% End:

