function [pos, varargout] = mfind( matrix, array )
% Copyright (C) 2005,2006,2007 Daniele de Rigo
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% [pos, ...] = mfind( matrix, array )
% 
% Find indices and values of those elements of <matrix> that are equal
% to any element of <array>.
%
% Input arguments:
% 
% <matrix>                 ::numstring::
%                          matrix of numbers or of characters
%
% <array>                  ::numstring,vector::
%                          array  of numbers or of characters
% 
% This function extends the find(.) function.
% If more than one output are assigned, the behaviour is analogous to that
% of the find(.) function.
% 
% 
% Examples of usage:
% 
%    n = 300; N = n*n;
%    A = reshape( 1:N , n , n );
%    p = primes( N );
% 
%    tic; [ r1 , c1 ] = mfind( A , p ); toc
% 
%    tic; 
%    r2 = zeros(numel(p),1);  c2 = r2;  
%    for i=1:numel(p) 
%       [ r2(i), c2(i) ] = find( A==p(i) ); 
%    end
%    toc
% 
%    all( r1 == r2 )
%    all( c1 == c2 )
%    hist( c1 , n )
% 
%    t                 =  'mfind( matrix, array )'  ;
%    indexes           =   mfind( t ,  '(,)'  )
%    [rows, cols]      =   mfind( t ,  '(,)'  )
%    [rows, cols, val] =   mfind( t ,  '(,)'  )
%
%    t(t==' ') = [];  % erase spaces
%    [rows, cols, val] =   mfind( t ,  '(,)'  )
%    t(cols)   = [];  % erase separators
%    len       = diff([0 cols])-1; % token length  
%    tokens    = mat2cell( t , 1 , len )
% 
% 
% version: 0.5.2

where = sprintf(  '(in function %s)'  , mfilename );

% mastrave-kernel: this function is used by check_nargin, therefore the
% test for the allowed number of input arguments is performed without
% invoking check_nargin

if(nargin~=2)
   fprintf( 2,  'Error: expected 2 arguments instead of %d.\n'  , nargin );
   fprintf( 2,  'Usage: mfind( matrix, array )\n'  );
   error(  ' '  );
end;

check_is( matrix ,  'numstring'                      ,  ...
          [  '%s the first argument <matrix> must be '  ...
             'numeric or a string'  ], where );
check_is( array ,  'numstring'                       ,  ...
          [  '%s the second argument <array> must be '  ...
             'numeric or a string'  ], where );
check_is( array ,  'vector'                          ,  ...
          [  '%s the second argument <array> must be '  ...
             'scalar or a vector'   ], where );

m = zeros( size(matrix) );

if numel(array) ~= 1
   [ idx , dim  ] = real2index( { matrix, array } );
   [ m1 , array ] = deal( idx{:} );
   
   s     = sparse( array-0, ones(size(array)), ones(size(array)), dim, 1);
   
   % <matrix> and <m1> may be numeric or strings.
   % The following assignment from sparse to a numeric object is 
   % widely portable
   m(:) = s( m1(:)-0 ); 
else  % use find(.)
   m(:) = matrix == array;
end 

if nargout > 1
   [pos, varargout{1:nargout-1}] = find( m );
else % ensure assignment to <ans> if <nargout> == 0
   pos                           = find( m );
end
if nargout == 3
   varargout{2} = matrix( sub2ind( size(matrix), pos, varargout{1} ) );
end





% Local Variables:
% mode:mastrave
% End:

