function [answer] = mreg_replace( string , pattern , replacement , options )
% Copyright (C) 2006,2007,2008,2009,2010 Daniele de Rigo 
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% [answer] = mreg_replace( string , pattern , replacement , options )
%
% Perform a perl-like extended regular expression search and replace on
% a given <string>.
%
% Input arguments:
% 
% <string>            ::cellstring::
%                     string or cell-array of strings on which to apply the
%                     regular expression search and replace 
%
% <pattern>           ::string::
%                     perl-like extendend regular expression search
%                     Valid operators are:
%
%                         operator      |         meaning
%                     ------------------+------------------------------------
%                         <expr>|<expr> | boolean or: both <expr>s can match
%                                       | (one at a time)
%                     ------------------+------------------------------------
%                         (<expr>)      | subexpression: group the content
%                                       | <expr> as if it were a single
%                                       | token; <expr> is marked and can be
%                                       | referred to in the subsequent part
%                                       | of <pattern> or returned in
%                                       | <replacement> using the operator
%                                       | \<n> (see it) for the <n>th marked
%                                       | subexpression
%                     ------------------+------------------------------------
%                         \<n>          | match the <n>th marked 
%                                       | subexpression
%                     ------------------+------------------------------------
%                         (?:<expr>)    | non-marked subexpression: it
%                                       | cannot be referred to nor returned
%                                       | but still consumes the characters 
%                                       | it matches, as in the basic
%                                       | (<expr>) operator
%                     ------------------+------------------------------------
%                         (?=<expr>)    | forward asserion: non-marked
%                                       | subexpression which is a test on
%                                       | the characters following the
%                                       | current matching point performed
%                                       | without actually consuming any
%                                       | character
%                     ------------------+------------------------------------
%                         (?<=<expr>)   | backward asserion: non-marked
%                                       | subexpression which is a test on
%                                       | the characters preceding the
%                                       | current matching point performed
%                                       | without actually consuming any
%                                       | character
%                     ------------------+------------------------------------
%                         (?!<expr>)    | forward negative asserion: test on
%                                       | the characters following the 
%                                       | matching point to ensure they do
%                                       | not match <expr>
%                     ------------------+------------------------------------
%                         (?<!<expr>)   | backward negative asserion: test on
%                                       | the characters preceding the 
%                                       | matching point to ensure they do
%                                       | not match <expr>
%                     ------------------+------------------------------------
%                         .             | any single character except newline
%                     ------------------+------------------------------------
%                         [<expr>]      | any single character contained
%                                       | in <expr>; if <expr> contains a
%                                       | range of characters in the form
%                                       | <c1>-<c2> (e.g a-z or 0-9), then
%                                       | all characters from <c1> to <c2>
%                                       | will match
%                     ------------------+------------------------------------
%                         [^<expr>]     | any single character not included
%                                       | in the set <expr> 
%                     ------------------+------------------------------------
%                         \w            | [a-zA-Z0-9_] words
%                     ------------------+------------------------------------
%                         \W            | [^\w] non-words
%                     ------------------+------------------------------------
%                         \d            | [0-9] digits
%                     ------------------+------------------------------------
%                         \D            | [~\d] non-digits
%                     ------------------+------------------------------------
%                         \s            | [ \t\r\n\v\f] spaces
%                     ------------------+------------------------------------
%                         \S            | [^\s] non-spaces
%                     ------------------+------------------------------------
%                         \n            | newline character
%                     ------------------+------------------------------------
%                         \t            | tab character
%                     ------------------+------------------------------------
%                        <tok>{<n>,<m>} | match the token <tok> if it's
%                                       | repeated from a minimum of <n>
%                                       | times to a maximum of <m> times
%                                       | (greedy quantifier: it matches as
%                                       | much as possible -- up to the
%                                       | maximum number of permitted times) 
%                     ------------------+------------------------------------
%                        <tok>{<n>}     | <tok>{<n>,<n>} (greedy quantifier) 
%                     ------------------+------------------------------------
%                        <tok>{<n>,}    | <tok>{<n>,65535}
%                                       | (greedy quantifier)
%                     ------------------+------------------------------------
%                        <tok>*         | <tok>{0,}  (greedy quantifier) 
%                     ------------------+------------------------------------
%                        <tok>?         | <tok>{0,1} (greedy quantifier)
%                     ------------------+------------------------------------
%                        <tok>+         | <tok>{1,}  (greedy quantifier)
%                     ------------------+------------------------------------
%                        <tok>{<n>,<m>}?| lazy quantifier: it matches the 
%                                       | minimum number of permitted times 
%                     ------------------+------------------------------------
%                        <tok>{<n>,}?   | <tok>{<n>,65535} (lazy quantifier)
%                     ------------------+------------------------------------
%                        <tok>*?        | <tok>{0,}  (lazy quantifier) 
%                     ------------------+------------------------------------
%                        <tok>??        | <tok>{0,1} (lazy quantifier)
%                     ------------------+------------------------------------
%                        <tok>+?        | <tok>{1,}  (lazy quantifier)
%                     ------------------+------------------------------------
%                        ^              | begin of the line
%                     ------------------+------------------------------------
%                        $              | end of the line (newline excluded) 
%
%
%
% <replacement>       ::string::
%                     replacement string
%                     Valid operators are:
%
%                         operator      |         meaning
%                     ------------------+------------------------------------
%                         \<n>          | return the <n>th subexpression
%                                       | marked in <pattern>
%                     ------------------+------------------------------------
%                         \n            | newline character
%                     ------------------+------------------------------------
%                         \t            | tab character
%
%
%
% <options>           ::string::
%                     modifier codes (default: 'g')
%                     Valid modifiers are:
%
%                         operator      |         meaning
%                     ------------------+------------------------------------
%                         i             | non case-sensitive matching
%                     ------------------+------------------------------------
%                         s             | if passed, add newline in the
%                                       | . operator set of characters
%                     ------------------+------------------------------------
%                         g             | "global" : replace all matches,
%                                       | and not just the first one. 
%
%
%
% Examples of usage
%
%    mreg_replace(                                                    ...
%       '/  1 /* 2 **/ 3 /**4*/ 5 ' , '/\*.*?\*/' , '[]', 'g'         ...
%    )
%    mreg_replace(                                                    ...
%       'Foo bar baz foO BaR  bAz' , '([a-z ]{2,4})' , '[\1]' , 'g'   ...
%    )
%    mreg_replace(                                                    ...
%       'Foo bar baz foO BaR  bAz' , '([a-z ]{2,4})' , '[\1]' , 'i'   ...
%    )
%    mreg_replace(                                                    ...
%       'Foo bar baz foO BaR  bAz' , '([a-z ]{2,4})' , '[\1]' , 'gi'  ...
%    )
%    mreg_replace(                                                    ...
%       'Foo bar baz foO BaR  bAz' , '([a-z ]{2,4}?)' , '[\1]' , 'gi' ...
%    )
%    mreg_replace(                                                    ...
%       { 'Foo bar baz' , 'foO BaR  bAz' } ,                          ...
%       '([a-z ]{2,4}?)' , '[\1]' , 'gi'                              ...
%    )
%
%
% 
% Shell requirements: cat, perl
%
%
%
% version: 0.2.14

where     = sprintf(  '(in function %s)'  , mfilename );

usage_msg = sprintf( [                                    ...
   'Usage: [answer] = mreg_replace(               ... \n' ...
   '   string , pattern , replacement , options   ... \n' ...
   ')\n'  ...
] );

if nargin<3
   fprintf( 2,  'Error: not enough input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end

if nargin<4
   options = 'g';
end

check_is( string ,  'cellstring'                           , ...
   [  '%s the first argument <string> must be a string, '    ...
      'or a cell array of strings.'                       ], ...
   where                                                     ...
);
is_cell = logical(1);
if ischar( string )
   is_cell = logical(0);
   string  = { string };
end 

check_is( pattern ,  'string'                              , ...
   '%s the second argument <pattern> must be a string.'    , ...
   where                                                     ...
);

check_is( replacement ,  'string'                          , ...
   '%s the third argument <replacement> must be a string.' , ...
   where                                                     ...
);

check_is( options ,  'string'                              , ...
   '%s the fourth argument <options> must be a string.'    , ...
   where                                                     ...
);

valid_opts  = 'isg' ;
valid_id    = mfind( options , valid_opts );
check_is( numel( valid_id ) == numel( options ) , 'true'   , ...
   [                                                         ...
      '%s unknown option passed in the fourth argument '     ...
      '<option> (valid options are %s).'                     ...
   ]                                                       , ...
   where                                                   , ...
   sprintf( '''%c'' ' , valid_opts )                         ...
);
   

pattern     = strrep( pattern     , '!'  , '\!' );
pattern     = strrep( pattern     , '\n' , sprintf( '\\\n' ) );
pattern     = strrep( pattern     , '\t' , sprintf( '\t' ) );
pattern     = strrep( pattern     , '\f' , sprintf( '\f' ) );
pattern     = strrep( pattern     , '\r' , sprintf( '\r' ) );
pattern     = strrep( pattern     , '\v' , sprintf( '\v' ) );
replacement = strrep( replacement , '\n' , sprintf( '\\\n' ) );
replacement = strrep( replacement , '\t' , sprintf( '\t' ) );
replacement = strrep( replacement , '\f' , sprintf( '\f' ) );
replacement = strrep( replacement , '\r' , sprintf( '\r' ) );
replacement = strrep( replacement , '\v' , sprintf( '\v' ) );

% check regexp validity
command     = sprintf(                            ...
   'perl -c -e ''s!%s!%s!%s'' 2>&1'           ,   ...
    pattern                                   ,   ...
    replacement                               ,   ...
    options                                       ...
);
[ status , answer ] = unix( command );
check_is( status == 0 , 'true'                             , ...
   '%s the passed regexp is not well-formed:\n"%s"'        , ...
   where                                                   , ...
   answer                                                    ...
);


string_fn   = tempname;

answer      = cell( size(string) );
for i = 1:numel(string)
   fid      = fopen( string_fn , 'wt' );
   fprintf( fid , '%s' , string{i} );
   fclose( fid );

   command  = sprintf(                              ...
      'cat ''%s'' | perl -p -e ''s!%s!%s!%s'' ' ,   ...
      string_fn                                 ,   ...
      pattern                                   ,   ...
      replacement                               ,   ...
      options                                       ...
   );
 
   [ status , answer{i} ] = unix( command );
end
if ~is_cell
   answer   = answer{1};
end

delete( string_fn );



% Local Variables:
% mode:mastrave
% End:


