function [ matrix, siz ] = multi2mat_( multi_array, dim )
% Copyright (C) 2005,2006,2007,2008,2009 Daniele de Rigo 
% 
% This file is part of Mastrave.
% 
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
% 
% ---------------------------------------------------------------------------
% 
% [ matrix, siz ] = multi2mat_( multi_array, dim )
%
%    Warning: multi2mat_ should not be used in end-user code because it
%    deliberately skips input and output arguments checks and other pre- and 
%    post-condition testing.
%    End-user code should use instead the multi2mat function (without the
%    ending underscore).
% 
% Some functions are defined for matrices but not for multidimensional arrays
% (md-arrays).
% multi2mat_ allows to deal with md-arrays even with those ill-expandable
% functions which operate column by column.
% Column-wise operations by default apply to the columns of a matrix as if the
% matrix were a vector of column vectors.  This class of functions includes
% reductions (in the APL sense) such as sum(.), prod(.); and scans (in the APL
% sense) such as cumsum(.), cumprod(.).
% The aforementioned basic functions extend to md-arrays, however some
% user-defined functions may not allow the same.   multi2mat_ converts a
% k-dimensional array <multi_array> of size <siz>:
%    [ n1 ... n<dim> ... nk ]
% into a matrix of size:
%    [ n<dim>  prod([n1 .. n<dim>-1 n<dim>+1 nk]) ]
% as if <multi_array> were a collection of vectors having n<dim> elements each
% and being stored in the nd-array along its <dim> dimension.
% 
% The function mat2multi_ can reverse the conversion.
%
%
%
% Example of usage:
%
%    mda     = zeros( 3, 4, 2);
%    mda(:)  = 1:numel( mda )
%
%    dim     = 1  % 4x2 vectors of 3 elements each
%    multi2mat_( mda , dim )  
%
%    dim     = 2  % 3x2 vectors of 4 elements each
%    multi2mat_( mda , dim )  
%
%    dim     = 3  % 3x4 vectors of 2 elements each
%    multi2mat_( mda , dim )
%    
%    % reverse the conversion:
%    [ mat , siz ] = multi2mat_( mda , dim );
%    mat2multi_( mat , dim , siz ) 
%
%    % scan example using cumsum
%    scan1   = cumsum( mda , dim )
%    scan2   = mat2multi_( cumsum( mat , 1 ) , dim , siz )
%    scan1  == scan2
%
%    % reduction example using sum
%    red1    = sum( mda , dim )
%    red_siz = siz; red_siz( dim ) = 1
%    red2    = mat2multi_( sum( mat , 1 ) , dim , red_siz )
%    red1   == red2
%
%
%
% version: 0.2.4

siz           = size( multi_array );

dim           = mod( dim-1 , numel( siz ) ) + 1;
% <dims> is the needed permutation among the <multi_matrix> dimensions
dims          = [dim 1:dim-1 dim+1:numel( siz )]; 

if numel( multi_array ) == 0
   matrix     = zeros( siz(dim) , prod( siz(dims(2:end)) ) );
   return
end

matrix        = reshape( permute(multi_array,dims), siz(dim), [] );



% Local Variables:
% mode:mastrave
% End:


