function [ indexes, cardinality, glossary ] = real2index( reals )
% Copyright (C) 2005,2006,2007,2008,2009 Daniele de Rigo
%
% This file is part of Mastrave.
%
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
%
% ---------------------------------------------------------------------------
%
% [ indexes, cardinality, glossary ] = real2index( reals )
%
% Floating point values cannot be used as indexes.
% real2index(...) maps floating point numbers into positive integers
% preserving their order.
% In case of sparse matrices, the mapping replaces only nonzeros.
% If <reals> is a cell-array of matrices then <indexes> is returned as a
% cell-array whose elements are the corresponding matrices of indices.
% Each index of <indexes> represents a value of (the elements of) <reals> which
% may occur several times in <reals>. To each occurrence of the same value
% corresponds the same index.
% The set of these correspondences is returned in the vector <glossary>, whose
% length is the <cardinality> of the indices set.
%
%
% Input arguments:
%
% <reals>                  ::cellnumstring::
%                          matrix of numbers or of characters
%                          or cell array whose elements are matrices of
%                          numbers/characters
%
%
% Example of usage:
%
%    m1            =  '[ indexes, cardinality, glossary ] = real2index( reals )'
%    m2            =  randn(1,20)
%    m3            =  m2( ceil(rand(1,20)*20) )
%    m4            =  zeros(3,2);
%    m4(:)         =  m2(1:numel(m4))
%    m5            =  zeros(5);
%    m5(ceil(rand(1,numel(m4))*25)) = m4(:)
%    m5            = sparse( m5 )
%    idx           = real2index( m1 )
%    [idx, len]    = real2index( {m1 , m2} )
%    [idx, len]    = real2index( {m1 , m2 , m3} )
%    [idx, len]    = real2index( {m1 , m2 , m4} )
%    [idx, len]    = real2index( {m1 , m2 , m5} )
%    [idx, len]    = real2index( {m4 , m5} )
%    [idx, len]    = real2index( m5 )
%    [idx, len, g] = real2index( m2 )
%    m2_rep        = zeros(size(m2));
%    m2_rep(:)     = g(idx)
%    [idx, len, g] = real2index( m1 )
%    m1_rep        = char( zeros(size(m1)) );
%    m1_rep(:)     = char( g(idx) )
%
%
% version: 0.6.7

where = sprintf(  '(in function %s)'  , mfilename );

% mastrave-kernel: this function is used by check_nargin, therefore the
% test for the allowed number of input arguments is performed without
% invoking check_nargin

if(nargin~=1)
   fprintf( 2,  'Error: expected one argument instead of %d.\n'  , nargin );
   fprintf( 2,  'Usage: real2index( reals )\n'  );
   error(  ' '  );
end;
check_is( reals ,  'cellnumstring'                      ,  ...
          [  '%s the argument <reals> must be numeric '    ...
             'or a string or a cell array with '           ...
             'numeric or string elements'              ], where );

if iscell( reals )
%   len  = cellfun(  'prodofsize'  , reals ); 
   [len , sp_lid]  = get_numel( reals );
   clen = [ 0 ; cumsum( len(:) ) ];
   n    = zeros( clen(end), 1 );
   pos  = cell(1,numel(reals));
   for i=1:numel(reals)
      if sp_lid(i)
         [r,c,nz] = find(reals{i});
         pos{i}   = {r,c};
         n( clen(i) + [1:len(i)] ) = nz-0;
      else
         n( clen(i) + [1:len(i)] ) = reals{i}(:)-0;
      end
   end
else
   if issparse( reals )
      [r,c,nz] = find(reals);
      pos      = {r,c};
      n        = nz-0;
   else
      n        = reals(:)-0;
   end
end

% <n> may be numeric or a string
[revid, sn, sid] = score(n);
unique_lid       = [1; diff(sn-0)] ~= 0;
glossary         = sn(unique_lid);

% old
in               = zeros(size(sn));
in(unique_lid)   = 1;          % <sn> may be numeric or a string
                               % (sn-0 is numeric)
in               = cumsum(in); % <in> is a vector of positive integers
% may be replaced by:
%    in          = cumsum(unique_lid);

cardinality      = in(end);    % amount of distinct elements
in               = in(revid);  % un-sort <in>

% replace <reals> values with their corresponding indexes
% (in case of sparse matrices, replace only nonzeros)
indexes = reals;
if iscell( reals )
   for i=1:numel(reals)
      if sp_lid(i)
         siz           = size(indexes{i});
         indexes{i}    = sparse( pos{i}{:},                  ...
                                 in( clen(i) + [1:len(i)] ), ...
                                 siz(1)   , siz(2)               );
      else
         indexes{i}    = indexes{i}-0;
         indexes{i}(:) = in( clen(i) + [1:len(i)] );
      end
   end
else
   if issparse( reals )
      siz        = size(indexes);
      indexes    = sparse( pos{:},    ...
                           in,        ...
                           siz(1), siz(2) );
   else
      indexes    = indexes-0;
      indexes(:) = in;
   end
end


function [len, sp_lid] = get_numel( reals )
   len    = cellfun(  'prodofsize'  , reals ); 
   sp_lid = cellfun( @issparse      , reals );
   if iscell(sp_lid)  sp_lid = logical([sp_lid{:}]); end
   % [octave 2.9.9, 3.0.2, 3.0.3] cellfun( @nnz , {} ) doesn't return args!
   if any(sp_lid)
      nz = cellfun( @nnz            , reals(sp_lid) );
      if iscell(nz)
         if numel(nz)
            nz = [nz{:}] ; 
         else
            nz = [];
         end
      end
      len( sp_lid ) = nz;
   end




% Local Variables:
% mode:mastrave
% End:

