function [score_idx, sorted_X, sorted_idx] = score( X , dim , mode )
% Copyright (C) 2005,2006,2007,2008,2009 Daniele de Rigo
%
% This file is part of Mastrave.
%
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
%
% ---------------------------------------------------------------------------
%
% [score_idx, sorted_X, sorted_idx] = score( X , dim , mode )
%
% similar to
%
%   [sorted_X, sorted_idx] = sort( X , dim , mode )
%
% score(X) returns as first element <score_idx> a vector representing
% the score of each <X> element. If <X> is a vector <vec> the relation
% between <score_idx> and <sorted_idx> is:
%
%    [score_idx, sorted_vec, sorted_idx] = score( vec );
%           vec( sorted_idx ) == sorted_vec
%    sorted_vec( score_idx  ) ==        vec
%
% If <X> is a matrix <M> the relation between <score_idx> and <sorted_idx> is:
%
%    siz    = size(M);
%    [r, c] = ind2sub( siz, 1:numel(M) );
%    [score_idx, sorted_M, sorted_idx] = score(M);
%           M( sub2ind( siz, sorted_idx, c ) ) == sorted_M
%    sorted_M( sub2ind( siz,  score_idx, c ) ) ==        M
%
% If <X> is a sparse matrix <sM>, <score_idx> and <sorted_idx> are related to
% the nonzeros elements of <sM>;
%    [score_idx, sorted_sM, sorted_idx] = score(sM);
%    [r1, c1,  nz] = find(sM);
%    [r2, c2, snz] = find(sorted_sM);
%    sid           = nonzeros(sorted_idx);
%    scr           = nonzeros(score_idx );
%    sorted_sM == sparse( r2, c2, nz(sid) )
%
%
% Input arguments:
%
% <X>                ::sortable::
%                    numeric vector o matrix, or string, or cell-array
%                    of strings
%
% <dim>              ::scalar_natural_nonzero::
%                    dimension along which to sort <X>
%
% <mode>             ::string::
%                    direction in which <X> will be sorted. Valid modes are:
%
%                         mode    |      meaning
%                    -------------+----------------------------
%                      'ascend'   | sort in  ascending order
%                    -------------+----------------------------
%                      'descend'  | sort in descending order
%
%
% Examples of usage:
%
%    A = zeros(5,1); A(:) = randperm(numel(A))*10
%    [revid, sA, sid]   = score( A )
%    A2    =  A;
%    A2(:) = sA( revid );
%    all( A(:) == A2(:) )
%
%    A = zeros(1,5); A(:) = randperm(numel(A))*10
%    [revid, sA, sid]   = score( A )
%    A2    =  A;
%    A2(:) = sA( revid );
%    all( A(:) == A2(:) )
%
%    A = zeros(3,4); A(:) = randperm(numel(A))*10
%
%    [revid, sA, sid]     = score( A )
%    [r, c]               = ind2sub( size(A), 1:numel(A) );
%    A2    =  A;
%    A2(:) = sA( sub2ind( size(A), revid(:), c(:) ) );
%    all( A(:) == A2(:) )
%
%    [revid, sA, sid]     = score( A , 2 )
%    [r, c]               = ind2sub( size(A), 1:numel(A) );
%    A2    =  A;
%    A2(:) = sA( sub2ind( size(A), r(:), revid(:) ) );
%    all( A(:) == A2(:) )
%
%    A = zeros(3,4,2); A(:) = ceil(rand(1,numel(A))*numel(A))*10
%    [revid, sA, sid]       = score( A , 2 )
%    [rr{1:3}]              = ind2sub( size(A), 1:numel(A) );
%    A2    =  A;
%    A2(:) = sA( sub2ind( size(A), rr{1}(:), revid(:), rr{3}(:) ) );
%    all( A(:) == A2(:) )
%
%    % multidim. cell array
%    cA = mat2cell( sprintf('%03d',A(:)), 1, ones(1,numel(A))*3 );
%    cA = reshape( cA, size(A) )
%    [revcid, scA, scid]    = score( cA , 2 )
%    all( revcid(:) == revid(:) )
%    all(   scid(:) ==   sid(:) )
%    disp('  A   cA  |  sA  scA');
%    for i=1:numel(A)
%       fprintf( 1, '%3d ''%s'' | %3d ''%s''\n', A(i), cA{i}, sA(i), scA{i} );
%    end
%
%
% version: 0.8.7

where = sprintf(  '(in function %s)'  , mfilename );

% mastrave-kernel: this function is used by mfind that is used by check_nargin,
% therefore the test for the allowed number of input arguments is performed
% without invoking check_nargin

usage_msg = sprintf( [                                                     ...
   'Usage: [score_idx, sorted_X, sorted_idx] = score( X , dim , mode )\n'  ...
   '       [score_idx, sorted_X, sorted_idx] = score( X , [dim|mode] )\n'  ...
   '       [score_idx, sorted_X, sorted_idx] = score( X )\n'            ]  );

if nargin<1
   fprintf( 2,  'Error: not enough input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end
if nargin>3
   fprintf( 2,  'Error: too many input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end

check_is( X ,  'sortable'                               , ...
         '%s the first argument <X> must be sortable.'  , ...
         where );
switch nargin
case 1
   if min(size(X)) == 1  &  numel(X) > size(X,1)
      dim = 2;
   else
      dim = 1;
   end
   mode =  'ascend'  ;
case 2
   check_is( dim ,  'numstring'                   , ...
             [  '%s the second argument must be a string '  ...
                'or a positive integer.\n'               ], ...
             where );
   if ischar( dim )
      mode = dim;
      if ~any( strcmp( mode , {  'ascend'  ,  'descend'  } ) )
         check_is( 0 ,  'true'                               , ...
                  [  '%s the second argument <mode> must be '  ...
                     'either "ascend" or "descend"\n'          ...
                     '(passed unknown "%s" mode).\n'        ], ...
                  where, mode );
      end
      if min(size(X)) == 1  &  numel(X) > size(X,1)
         dim = 2;
      else
         dim = 1;
      end
   else
      check_is( dim ,  'scalar_natural_nonzero'  , ...
               [  '%s the second argument <dim> must be '  ...
                  'a positive integer.\n'               ], ...
               where );

      check_is( dim <= ndims(X),  'true'  , ...
               [ '%s the second argument <dim> must be a '   ...
                 'valid dimension\n'                         ...
                 '(passed %d while ndims(<X>) is %d).\n'  ], ...
               where , dim , ndims(X) );
      mode =  'ascend'  ;
   end
case 3
   check_is( dim ,  'scalar_natural_nonzero'  , ...
            [  '%s the second argument <dim> must be '  ...
               'a positive integer.\n'               ], ...
            where );

   check_is( dim <= ndims(X),  'true'  , ...
            [ '%s the second argument <dim> must be a '    ...
               'valid dimension\n'                         ...
               '(passed %d while ndims(<X>) is %d).\n'  ], ...
            where , dim , ndims(X) );

   check_is( mode ,  'string'                                    , ...
             '%s the third argument <mode> must be a string.\n'  , ...
             where );

   if ~any( strcmp( mode , {  'ascend'  ,  'descend'  } ) )
      check_is( 0 ,  'true'                              , ...
               [  '%s the third argument <mode> must be '  ...
                  'either "ascend" or "descend"\n'         ...
                  '(passed unknown "%s" mode).\n'       ], ...
               where, mode );
   end
end

if ~numel( X )
   [ score_idx, sorted_X, sorted_idx ] = deal( [], X , [] );
   return
end

if issparse( X )
   lens            = full(sum(spones(X),dim));
   lens            = lens(:);
   old_siz         = size(X);
   [rows, cols, X] = find(X);
   if dim == 2
      X = X';
   end
end

% matlab 7.1.0:
% "using ==> cell.sort DIM argument not supported for cell arrays"
if ~exist( 'OCTAVE_VERSION' )
   if iscell(X)
      [sorted_X, sorted_idx] = strcell_sort( X, dim, mode );
      % warning('using ==> cell.sort DIM argument not supported for cell arrays');
   else
      [sorted_X, sorted_idx] = sort(X,dim,mode);
   end
else
   [sorted_X, sorted_idx] = sort(X,dim,mode);
end
siz                    = size(X);
score_idx              = zeros(siz);
[subs{1:ndims(X)}]     = ind2sub( siz, 1:numel(X) );
order                  = subs{dim};
subs{dim}(:)           = sorted_idx(:);
X_id                   = sub2ind( siz, subs{:} );
% X_id                   = sorted_idx +...
%                          repmat(0:numel(X)/siz(1)-1, siz(1),1)*...
%                          siz(1);
score_idx(X_id)        = order; %repmat([1:siz(1)]',1,numel(X)/siz(1));
% return
%
% n=length(X);
% X=reshape(X,1,n);
% [sorted_X sorted_idx]=sort(X);
% score_idx=[1:n]*(repmat(sorted_idx',1,n)==repmat([1:n],n,1));

if exist(  'cols'  ) % if <X> was a sparse matrix
   if dim == 1
      bias = cols;
   else % <dim> == 2
      bias = rows;
   end
   score_idx = score_idx(:);
   if strcmp( mode ,  'ascend'  )
      [ans, pos] = sort( score_idx + bias*numel(score_idx) ,   'ascend'  );
   else
      [ans, pos] = sort( score_idx - bias*numel(score_idx) ,  'descend'  );
   end
   sorted_X   = X( pos );
   sorted_idx =    pos  ;
   score_idx  = score_idx(  pos );
   bias       = bias(       pos );
   ge_id      = sorted_X >= 0;

   [sbias, bid] = sort(bias);
   seq        = ones( size(bias) );
   nz_lens    = nonzeros(lens);
   seq(cumsum(nz_lens(1:end-1))+1)= 1-nz_lens(1:end-1);
   seq        = cumsum(seq);
   seq(bid)   = seq;
   seq(ge_id) = seq(ge_id) + old_siz(dim)-lens(bias(ge_id));
   if strcmp( mode ,  'descend'  )
      seq = old_siz(dim)-seq+1;
   end
   if dim == 1
      rows = seq;
      cols = bias;
   else % <dim> == 2
      rows = bias;
      cols = seq;
   end
   sorted_X   = sparse( rows , cols , sorted_X   , old_siz(1), old_siz(2) );
   sorted_idx = sparse( rows , cols , sorted_idx , old_siz(1), old_siz(2) );
   score_idx  = sparse( rows , cols , score_idx  , old_siz(1), old_siz(2) );
end



function [sorted_cX, sorted_idx] = strcell_sort( cX, dim, mode )
   [ sorted_cX, sorted_idx ] = deal( {} , [] );
   if ~numel( cX ) return; end
   [ scX , sid ]             = sort( cX );

   % strings to integer indexes mapping
   first_lid    = strcmp( scX(1:end-1), scX(2:end) );
   first_lid    = [ logical(1) ; ~first_lid(:) ];
   ustring      = scX( first_lid );
   uindex       = cumsum( first_lid );
   % for i=1:numel(uindex)                        %debug
   %    fprintf('%3d "%s"\n', uindex(i), scX{i}); %debug
   % end                                          %debug

   % nX has to be the integer indexes mapping of cX
   nX           = zeros( size(cX));
   nX( sid )    = uindex;
   [ sorted_nX, sorted_idx ] = sort( nX, dim, mode );
   % back from indexes to strings
   sorted_cX    = ustring( sorted_nX );


% Local Variables:
% mode:mastrave
% End:


