function shifted_series_cell = series_shift_domain( series_cell )
% Copyright (C) 2006,2007,2008,2009 Daniele de Rigo
%
% This file is part of Mastrave.
%
% Mastrave is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Mastrave is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Mastrave.  If not, see <http://www.gnu.org/licenses/>.
%
% ---------------------------------------------------------------------------
%
% shifted_series_cell = series_shift_domain( series_cell )
%
% Rearranges a set of data series whose domain has two dimensions and whose
% codomain is <n>-dimensional.  This utility may help to visualize complex data
% using the standard 2-D plot facilities.  See the example for an intuitive
% introduction.  The input set of data must be passed as a cell array of
% <n_param> matrices each of them representing an <n>-dimensional data series.
% All data series must have the same number of <n_elem> vectors of data (i.e
% each matrix must be composed by <n_elem> rows, each row being a row-vector
% of <n> columns).  Each series of data is supposed to be ordered with respect
% to a parameter <p1> which is the first dimension of the data domain.
% Matrices are supposed to be ordered with respect to a parameter <p2> which
% is the second dimension of the data domain.  That is, vectors in the same
% row of different matrices should represent <n>-dimensional data points
% measured for different values of the parameter <p2> while the parameter <p1>
% remains constant.  Vectors of the same matrix should represent measures for
% different values of the parameter <p1> while the parameter <p2> remains
% constant.
% In case of multi-array, all dimensions beyond the second one are splitted
% and considered as data matrices corresponding to different values of <p2>.
%
% The output set of data is a cell array of matrices in which the role of the
% domain dimensions (i.e. of the parameters <p1> and <p2>) is inverted.
%
%
%
% Input arguments:
%
% <series_cell>      ::cellnumeric-1::
%                    cell array of matrices all having the same size.
%
%
%
% Example of usage:
%
%    n_elem = 50;   n_param = 50;   n0 = n_elem+2*n_param;
%    t = [ 1:n0 ]';
%    y = [ sin(t/n_elem*pi) cos(t/n_elem*pi)+2 ] + randn(numel(t),2)/5;
%    s = {y}; id=1:n0;   Id = (n0-n_elem)/2+[1:n_elem];
%    tt= t(Id);          pp = 2+[1:n_param]'-1;          S  = {s{end}(Id,:)};
%
%    % compute moving averages with windows of 2*i-1 elements
%    for i=1:n_param-1
%       p        = 2*i;
%       id       = id(1:end-2);
%       s{end+1} = ( y(id,:) + s{end}(1+id,:)*(p-1) + y(p+id,:) ) / (p+1);
%       Id       = Id-1;
%       S{end+1} = s{end}(Id,:);
%    end
%
%
%    %----------------- with series_cell2mat(..) ----------------
%    title( 'Central moving average with growing moving window length' )
%    figure(1);
%    [ x , y ] = series_cell2mat( tt , S , 'true' );
%    plot( x , y );
%    xlabel( 'time' ); ylabel( 'values' )
%
%    S = series_shift_domain( S );
%    figure(2);
%    [ x , y ] = series_cell2mat( pp , S , 'true' );
%    plot( x , y );
%    xlabel( 'moving window length' ); ylabel( 'values' )
%
%
%    %----------------- without series_cell2mat(..) ----------------
%    title( 'Central moving average with growing moving window length' )
%    legend('off');  hold('off');  figure(1);
%    for i=1:numel(S)
%       plot( tt , S{i}(:,1) , 'r' , tt , S{i}(:,2) , 'b' );   hold on;
%    end
%    xlabel( 'time' ); ylabel( 'values' )
%
%    S = series_shift_domain( S );
%    figure(2);
%    for i=1:numel(S)
%       plot( pp , S{i}(:,1) , 'r' , pp , S{i}(:,2) , 'b' );   hold on;
%    end
%    xlabel( 'moving window length' ); ylabel( 'values' )
%
%
%
% version: 0.2.8

where = sprintf(  '(in function %s)'  , mfilename );

usage_msg = sprintf( [                   ...
   'Usage:  shifted_series_cell = series_shift_domain( series_cell )\n'  ...
] );

if nargin < 1
   fprintf( 2,  'Error: not enough input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end

if nargin > 1
   fprintf( 2,  'Error: too many input arguments\n'  );
   fprintf( 2,  usage_msg  );
   error(  ' '  );
end

check_is( series_cell ,  'cellnumeric-1'                      , ...
         [ '%s the first argument <series_cell> must be a '     ...
           'cell array of numeric elements.'                ] , ...
         where );

if numel( series_cell ) > 1
   check_is( series_cell ,  'same_size'                       , ...
         [ '%s the first argument <series_cell> must be'        ...
           'composed by matrices all having the same size.' ] , ...
         where );
end

shifted_series_cell = series_shift_domain_( series_cell );


% Local Variables:
% mode:mastrave
% End:

