/***************************************************************
 *                   Material Object Library                   *
 *     Class Eshelby : declaration for the Eshelby tensor      *
 *                    simula+@metz.ensam.fr                    *
 *		     GNU/linux version 0.1.8                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2005,2006 COLLARD Christophe
 * copyright  2005,2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  2005,2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class Eshelby
    \brief Numerical computation of Eshelby tensor for isotropic spheroidal inclusions

    \htmlonly 
    <FONT color="#838383">

    Eshelby belongs to Material Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    We note \f$ C \f$ the elasticity tensor (see \ref isotropic_tensor), \f$ T^{II} \f$ the interaction tensor (see \ref TII) and V the inclusion volume. Then, Eshelby tensor reads \n
\f$ \displaystyle S^{II}_{ijkl} = T^{II}_{ijpq} C_{pqkl} \f$.

    \author copyright \htmlonly &#169; \endhtmlonly 2005, 2006 Christophe COLLARD \n
            copyright \htmlonly &#169; \endhtmlonly 2005, 2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	    copyright \htmlonly &#169; \endhtmlonly 2005, 2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.1.8
    \date 2005-2006
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type Eshelby
#endif

#if !defined _Eshelby_h
#define _Eshelby_h


#if !defined(__MATHS_H)
#include "../MOL++/maths.h"
#endif

#if !defined(__VECTORS_H)
#include "../MOL++/vectors.h"
#endif

#if !defined(__TENSOR4_H)
#include "../MOL++/tensors4.h"
#endif

#if !defined(__ISOTROPIC_ELASTICITY_TENSORS_H)
#include "isotropic elasticity tensors.h"
#endif

#if !defined(__TII_H)
#include "TII.h"
#endif

using namespace std;


//===========
class Eshelby
//===========
{
  public :
    Eshelby () {}   // default constructor
    ~Eshelby () {}  // destructor

    template <class T, template<class T> class tensor_type> tensor4<T> ellipsoid (const tensor_type<T>&, const vector<T>&, int, int);
    template <class T> tensor4<T> sphere (const isotropic_elasticity_tensor<T>&);
    template <class T> tensor4<T> penny_shape (const isotropic_elasticity_tensor<T>&, const vector<T>&);
    // computes interaction TII = T_2323 = T_2332 for spheroid oblates
    template <class T> T oblate_spheroid_2332 (const isotropic_elasticity_tensor<T>&, const vector<T>&);
};

//=====Private methods for Eshelby==================================================


//=====Public methods for Eshelby===================================================


/*!
  \brief Computes Eshelby tensor in a general case

  First we compute \f$ T^{II} \f$. Then we compute Eshelby tensor. \n
  (see detailed description and \ref TII)

  \param L elasticity tensor
  \param geometry inclusion geometry
  \param nItg1 number of points for the integration on \f$ \theta \f$
  \param nItg2 number of points for the integration on \f$ \varphi \f$
  \return Eshelby tensor
*/

//------------------------------------------------------------------------------------------------------
template <class T, template<class T> class tensor_type>
tensor4<T> Eshelby::ellipsoid (const tensor_type<T>& L, const vector<T>& geometry, int nItg1, int nItg2)
//------------------------------------------------------------------------------------------------------
{
  assert (geometry.dim() == 3);

  TII<T> tii(nItg1,nItg2);

  return (tii.ellipsoid(L,geometry) || L);
}


/*!
  \brief Computes Eshelby tensor for sphere

  \f$ \begin{aligned}
  & S^{II}_{1111} = S^{II}_{2222} = S^{II}_{3333} = \frac{7 - 5\nu}{15(1-\nu)}, \\
  & S^{II}_{1122} = S^{II}_{2233} = S^{II}_{3311} = S^{II}_{1133} = S^{II}_{2211} = S^{II}_{3322} = \frac{5\nu - 1}{15(1-\nu)}, \\
  & S^{II}_{1212} = S^{II}_{1221} = S^{II}_{2112} = S^{II}_{2121} = S^{II}_{2323} = S^{II}_{2332} = S^{II}_{3223} = S^{II}_{3232} = S^{II}_{1313} = S^{II}_{1331} = S^{II}_{3113} = S^{II}_{3131} = \frac{4 - 5\nu}{15(1-\nu)}.
  \end{aligned} \f$ \n
  (see Toshio Mura, Micromechanics of Defects in Solid, Kluwer Academic Publishers, ISBN 90-247-3256-5)

  \param C elasticity tensor
  \return Eshelby tensor
*/

//-------------------------------------------------------------------------------------
template <class T> tensor4<T> Eshelby::sphere (const isotropic_elasticity_tensor<T>& C)
//-------------------------------------------------------------------------------------
{
  T nu = C.Poisson();
  T coef = 1/(15 * (1-nu));

  tensor4<T> SII(3);
  SII(1,1,1,1) = SII(2,2,2,2) = SII(3,3,3,3) = (7 - 5*nu) * coef;
  SII(1,1,2,2) = SII(2,2,3,3) = SII(3,3,1,1) = SII(1,1,3,3) = SII(2,2,1,1) = SII(3,3,2,2) = (5*nu - 1) * coef;
  SII(1,2,1,2) = SII(1,2,2,1) = SII(2,1,1,2) = SII(2,1,2,1) = SII(2,3,2,3) = SII(2,3,3,2) = SII(3,2,2,3) = SII(3,2,3,2) = SII(1,3,1,3) = SII(1,3,3,1) = SII(3,1,1,3) = SII(3,1,3,1) = (4 - 5*nu) * coef;

  return SII;
}


/*!
  \brief Computes Eshelby tensor for penny shape

  \f$ \begin{aligned}
  & S^{II}_{1111} = S^{II}_{2222} = \frac{13 - 8\nu}{32 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{3333} = 1 - \frac{1 - 2\nu}{1-\nu} \frac{\pi}{4} \frac{c}{a}, \\
  & S^{II}_{1122} = S^{II}_{2211} = \frac{8\nu - 1}{32 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{1133} = S^{II}_{2233} = \frac{2\nu -1}{8 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{3311} = S^{II}_{3322} = \frac{\nu}{1-\nu} \left( 1 - \frac{4\nu + 1}{8\nu} \pi \frac{c}{a} \right), \\
  & S^{II}_{1212} = S^{II}_{1221} = S^{II}_{2112} = S^{II}_{2121} = \frac{7 - 8\nu}{32 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{1313} = S^{II}_{1331} = S^{II}_{3113} = S^{II}_{3131} = S^{II}_{2323} = S^{II}_{2332} = S^{II}_{3223} = S^{II}_{3232} = \frac{1}{2} \left( 1 + \frac{\nu - 2}{1 - \nu} \frac{\pi}{4} \frac{c}{a} \right).
  \end{aligned} \f$ \n
  (see Toshio Mura, Micromechanics of Defects in Solid, Kluwer Academic Publishers, ISBN 90-247-3256-5)

  \param C elasticity tensor
  \param geometry penny shape length \f$ (a,a,c) \f$
  \return Eshelby tensor
*/

//---------------------------------------------------------------------------------------------------------------------
template <class T> tensor4<T> Eshelby::penny_shape (const isotropic_elasticity_tensor<T>& C, const vector<T>& geometry)
//---------------------------------------------------------------------------------------------------------------------
{
  assert (geometry.dim() == 3);
  assert (geometry[1] == geometry[2]);
  assert (geometry[1] > geometry[3]);

  T nu = C.Poisson();
  T invnu = 1/(1-nu);
  T PIa3_a1 = pi() * geometry[3] / geometry[1];
  //  T value = invnu * PIa3_a1 / (T) 32;
  T value = invnu * PIa3_a1 / 32.;

  tensor4<T> SII(3);
  SII(1,1,1,1) = SII(2,2,2,2) = (13 - 8*nu) * value;
  SII(3,3,3,3) = 1 - (8 - 16*nu) * value;
  SII(1,1,2,2) = SII(2,2,1,1) = (8*nu - 1) * value;
  SII(1,1,3,3) = SII(2,2,3,3) = (8*nu - 4) * value;
  SII(3,3,1,1) = SII(3,3,2,2) = nu * invnu - (16*nu + 4) * value;
  SII(1,2,1,2) = SII(1,2,2,1) = SII(2,1,1,2) = SII(2,1,2,1) = (7 - 8*nu) * value;
  SII(1,3,1,3) = SII(1,3,3,1) = SII(3,1,1,3) = SII(3,1,3,1) = SII(2,3,2,3) = SII(2,3,3,2) = SII(3,2,2,3) = SII(3,2,3,2) = .5 + (4*nu - 8) * value;

  return SII;
}


/*!
  \brief Computes Eshelby tensor 2323, 2332, 3223 and 3232 coordinates for oblate spheroid

  \f$ \displaystyle S^{II}_{2323} = S^{II}_{2332} = S^{II}_{3223} = S^{II}_{3232} = 
  \frac{1}{2 (1-\nu)(a^2 - c^2)} \left[ a^2 - \nu (a^2 - c^2) + \frac{1}{2} \left( \displaystyle \frac{a^2 c \arccos \left( \frac{c}{a} \right)}{\sqrt{a^2 - c^2}} - c^2 \right) \left( \nu - \frac{2a^2 + c^2}{a^2 - c^2} \right) \right]. \f$

  \param C elasticity tensor
  \param geometry penny shape length \f$ (a,a,c) \f$
  \return Eshelby tensor coordiantes 2323 2332 3223 3232
*/

//---------------------------------------------------------------------------------------------------------------------
template <class T> T Eshelby::oblate_spheroid_2332 (const isotropic_elasticity_tensor<T>& C, const vector<T>& geometry)
//---------------------------------------------------------------------------------------------------------------------
{
  assert (geometry.dim() == 3);
  assert (geometry[1] == geometry[2]);

  T nu = C.Poisson();
  T a = geometry[1];
  T a2 = a*a;
  T c = geometry[3];
  T c2 = c*c;

  assert (a > c);

  T coef = a2-c2;

  T Tii = .5 * ( a2 * c * acos(c/a) / sqrt(coef) - c2 ) * (nu - (2*a2 + c2) / coef);
  Tii += a2 - nu*coef;
  Tii *= .5 / ((1-nu)*coef);

  return Tii;
}


#endif
