/***************************************************************
 *                   Materials Object Library                  *
 *     Class MatOoLbox : declaration for materials toolbox     *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 0.1.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2005,2006 COLLARD Christophe
 * copyright  2005,2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  2005,2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class MatOoLbox
    \brief Toolbox for materials object libraries

    \htmlonly 
    <FONT color="#838383">

    MatOoLbox belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2005, 2006 Christophe COLLARD \n
            copyright \htmlonly &#169; \endhtmlonly 2005, 2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	    copyright \htmlonly &#169; \endhtmlonly 2005, 2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.1.0
    \date 2005-2006
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type MatOoLbox
#endif

#ifndef _materials_toolbox_h
#define _materials_toolbox_h


#if !defined (__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__FSTREAM_H)
#include <fstream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__MATHS_H)
#include <math.h>
#endif

#if !defined(__STRING_H)
#include <string.h>
#endif

#if !defined(__TENSORS2_H)
#include "../MOL++/tensors2.h"
#endif

#if !defined(__TENSORS4_H)
#include "../MOL++/tensors4.h"
#endif

#if !defined(__ISOTROPIC_TENSORS_H)
#include "../MateriOL++/isotropic tensors.h"
#endif

#if !defined(__ISOTROPIC_ELASTICITY_TENSORS_H)
#include "../MateriOL++/isotropic elasticity tensors.h"
#endif

using namespace std;


//=============
class MatOoLbox
//=============
{
  template <class T> friend T Von_Mises_stress (const tensor2<T>&);
  template <class T> friend T Von_Mises_strain (const tensor2<T>&);
  template <class T, template <class> class tensor_type> tensor2<T> constitutive_equation (const tensor_type<T>&, const tensor2<T>&, const tensor2<T>&, const T&, int);
};


//=====Public methods for materials toolbox=============================================


/*!
  \brief Computes the Von Mises equivalent stress

  \f$ \displaystyle \sigma_m = \sqrt{\frac{3}{2} dev(\sigma) :: dev(\sigma)} \f$

  \param stress stress tensor \f$ \sigma \f$
  \return Von Mises equivalent stress tensor
  \sa \ref tensor2::dev "dev"
*/

//==============================================================
template <class T> T Von_Mises_stress (const tensor2<T>& stress)
//==============================================================
{
  assert (stress.dim1());

  tensor2<T> deviatoric =& (dev(stress));

  return sqrt((T) 1.5 * (deviatoric|deviatoric));
}


/*!
  \brief Computes the Von Mises equivalent strain

  \f$ \displaystyle \EuScript{E}_m = \sqrt{\frac{2}{3} \EuScript{E} :: \EuScript{E}} \f$

  \param stress strain tensor (\f$ \EuScript{E} \f$)
  \return Von Mises equivalent strain tensor
*/

//==============================================================
template <class T> T Von_Mises_strain (const tensor2<T>& strain)
//==============================================================
{
  assert (strain.dim1());

  return sqrt((T) 2 * (strain|strain) / (T) 3);
}


/*!
  \brief computes the constitutive equations \f$ \Sigma = L E - M T \f$ and \f$ E = G \Sigma + N T \f$

  The stress tensor \f$ \Sigma \f$ and the strain tensor \f$ E \f$ are related through the following constitutive equations : \n
  \f$ \Sigma = L E - M T \f$, with \f$ T4 = L \; ; \; T2 = M \; ; \; stn = E \; ; \; str = \Sigma \; ; \; sgn = -1\f$. \n
  \f$ E = G \Sigma + N T \f$, with \f$ T4 = G \; ; \; T2 = N \; ;\; stn = \Sigma \; ;\; str = E \; ;\; sgn = 1 \f$. \n

  \param str strain / stress tensor (E / \f$ \Sigma \f$)
  \param To temperature T
  \param T4 L / G 4th order tensor
  \param T2 M / N 2nd order tensor
  \param sgn sign $\pm 1$
  \return tsr stress / strain tensor
*/

//----------------------------------------------------------------------------------------------------------------------------
template <class T, template <class> class tensor_type> 
tensor2<T> constitutive_equation (const tensor_type<T>& T4, const tensor2<T>& T2, const tensor2<T>& str, const T& To, int sgn)
//----------------------------------------------------------------------------------------------------------------------------
{
  // dS_ij = L_ijkl dE_kl - M_ij dT
  tensor2<T> tsr =& (T4 || str);

  if (sgn == 1) tsr += T2 * To;
  else if (sgn == -1) tsr-= T2 * To;
  else assert(false); // sgn value error

  return tsr;
}


#endif
