/***************************************************************
 *                   Material Object Library                   *
 *     Class Eshelby : declaration for the Eshelby tensor      *
 *                    simula.plus@cemes.fr                     *
 *		     GNU/linux version 3.0.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2005,2006,2007,2008,2012 COLLARD Christophe
 * copyright © 2005,2006,2007,2008,2012 Centre National de la Recherche Scientifique
 * copyright © 2005,2006,2007,2008 Arts et Métiers ParisTech
 * copyright © 2005,2006,2007 Université de Valenciennes et du Hainaut-Cambrésis
 * copyright © 2005,2006,2007,2008 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::Eshelby
    \brief Numerical computation of Eshelby tensor for isotropic spheroidal inclusions

    \htmlonly 
    <FONT color="#838383">

    Eshelby belongs to Material Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    We note \f$ C \f$ the elasticity tensor (see \ref isotropic_tensor), \f$ T^{II} \f$ the interaction tensor (see \ref TII) and V the inclusion volume. Then, Eshelby tensor reads \n
\f$ \displaystyle S^{II}_{ijkl} = T^{II}_{ijpq} C_{pqkl} \f$.

    \author copyright \htmlonly &#169; \endhtmlonly 2005, 2006, 2007, 2008, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; 2005, 2006, 2007, 2008, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006, 2007, 2008 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006, 2007 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
            copyright \htmlonly &#169; 2005, 2006, 2007, 2008 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006, 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
            copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.0.0
    \date 2005-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type Eshelby
#endif

#if !defined __Eshelby_hpp
#define __Eshelby_hpp


#ifndef __maths_hpp
#include "MOL++/maths.hpp"
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __tensors4_hpp
#include "MOL++/tensors4.hpp"
#endif

#ifndef __isotropic_elasticity_tensors_hpp
#include "MateriOL++/isotropic elasticity tensors.hpp"
#endif

#ifndef __TII_hpp
#include "MateriOL++/TII.hpp"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//===========
class Eshelby
//===========
{
  public :
    Eshelby () {}   // default constructor
    ~Eshelby () {}  // destructor

    template <class T, template<class T> class tensor_type> tensor4<T> ellipsoid (const tensor_type<T>&, const vector<T>&, int, int);
    template <class T> tensor4<T> sphere (const isotropic_elasticity_tensor<T>&);
    template <class T> tensor4<T> penny_shape (const isotropic_elasticity_tensor<T>&, const vector<T>&);
    // computes interaction TII = T_2323 = T_2332 for spheroid oblates
    template <class T> T oblate_spheroid_2332 (const isotropic_elasticity_tensor<T>&, const vector<T>&);
};

//=====Private methods for Eshelby==================================================


//=====Public methods for Eshelby===================================================


/*!
  \brief Computes Eshelby tensor in a general case

  First we compute \f$ T^{II} \f$. Then we compute Eshelby tensor. \n
  (see detailed description and \ref TII)

  \param L elasticity tensor
  \param geometry inclusion geometry
  \param nItg1 number of points for the integration on \f$ \theta \f$
  \param nItg2 number of points for the integration on \f$ \varphi \f$
  \return Eshelby tensor
*/

//------------------------------------------------------------------------------------------------------
template <class T, template<class T> class tensor_type>
tensor4<T> Eshelby::ellipsoid (const tensor_type<T>& L, const vector<T>& geometry, int nItg1, int nItg2)
//------------------------------------------------------------------------------------------------------
{
  assert (geometry.dim() == 3);

  TII<T> tii(nItg1,nItg2);

  return (tii.ellipsoid(L,geometry) || L);
}


/*!
  \brief Computes Eshelby tensor for sphere

  \f$ \begin{aligned}
  & S^{II}_{1111} = S^{II}_{2222} = S^{II}_{3333} = \frac{7 - 5\nu}{15(1-\nu)}, \\
  & S^{II}_{1122} = S^{II}_{2233} = S^{II}_{3311} = S^{II}_{1133} = S^{II}_{2211} = S^{II}_{3322} = \frac{5\nu - 1}{15(1-\nu)}, \\
  & S^{II}_{1212} = S^{II}_{1221} = S^{II}_{2112} = S^{II}_{2121} = S^{II}_{2323} = S^{II}_{2332} = S^{II}_{3223} = S^{II}_{3232} = S^{II}_{1313} = S^{II}_{1331} = S^{II}_{3113} = S^{II}_{3131} = \frac{4 - 5\nu}{15(1-\nu)}.
  \end{aligned} \f$ \n
  (see Toshio Mura, Micromechanics of Defects in Solid, Kluwer Academic Publishers, ISBN 90-247-3256-5)

  \param C elasticity tensor
  \return Eshelby tensor
*/

//-------------------------------------------------------------------------------------
template <class T> tensor4<T> Eshelby::sphere (const isotropic_elasticity_tensor<T>& C)
//-------------------------------------------------------------------------------------
{
  T nu = C.Poisson();
  T coef = 1/(15 * (1-nu));

  tensor4<T> SII(3);
  SII(1,1,1,1) = SII(2,2,2,2) = SII(3,3,3,3) = (7 - 5*nu) * coef;
  SII(1,1,2,2) = SII(2,2,3,3) = SII(3,3,1,1) = SII(1,1,3,3) = SII(2,2,1,1) = SII(3,3,2,2) = (5*nu - 1) * coef;
  SII(1,2,1,2) = SII(1,2,2,1) = SII(2,1,1,2) = SII(2,1,2,1) = SII(2,3,2,3) = SII(2,3,3,2) = SII(3,2,2,3) = SII(3,2,3,2) = SII(1,3,1,3) = SII(1,3,3,1) = SII(3,1,1,3) = SII(3,1,3,1) = (4 - 5*nu) * coef;

  return SII;
}


/*!
  \brief Computes Eshelby tensor for penny shape

  \f$ \begin{aligned}
  & S^{II}_{1111} = S^{II}_{2222} = \frac{13 - 8\nu}{32 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{3333} = 1 - \frac{1 - 2\nu}{1-\nu} \frac{\pi}{4} \frac{c}{a}, \\
  & S^{II}_{1122} = S^{II}_{2211} = \frac{8\nu - 1}{32 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{1133} = S^{II}_{2233} = \frac{2\nu -1}{8 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{3311} = S^{II}_{3322} = \frac{\nu}{1-\nu} \left( 1 - \frac{4\nu + 1}{8\nu} \pi \frac{c}{a} \right), \\
  & S^{II}_{1212} = S^{II}_{1221} = S^{II}_{2112} = S^{II}_{2121} = \frac{7 - 8\nu}{32 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{1313} = S^{II}_{1331} = S^{II}_{3113} = S^{II}_{3131} = S^{II}_{2323} = S^{II}_{2332} = S^{II}_{3223} = S^{II}_{3232} = \frac{1}{2} \left( 1 + \frac{\nu - 2}{1 - \nu} \frac{\pi}{4} \frac{c}{a} \right).
  \end{aligned} \f$ \n
  (see Toshio Mura, Micromechanics of Defects in Solid, Kluwer Academic Publishers, ISBN 90-247-3256-5)

  \param C elasticity tensor
  \param geometry penny shape length \f$ (a,a,c) \f$
  \return Eshelby tensor
*/

//---------------------------------------------------------------------------------------------------------------------
template <class T> tensor4<T> Eshelby::penny_shape (const isotropic_elasticity_tensor<T>& C, const vector<T>& geometry)
//---------------------------------------------------------------------------------------------------------------------
{
  assert (geometry.dim() == 3);
  assert (abs(geometry[1] - geometry[2]) < epsilon);
  assert (geometry[1] > geometry[3]);

  T nu = C.Poisson();
  T invnu = 1/(1-nu);
  T PIa3_a1 = pi() * geometry[3] / geometry[1];
  //  T value = invnu * PIa3_a1 / (T) 32;
  T value = invnu * PIa3_a1 / 32.;

  tensor4<T> SII(3);
  SII(1,1,1,1) = SII(2,2,2,2) = (13 - 8*nu) * value;
  SII(3,3,3,3) = 1 - (8 - 16*nu) * value;
  SII(1,1,2,2) = SII(2,2,1,1) = (8*nu - 1) * value;
  SII(1,1,3,3) = SII(2,2,3,3) = (8*nu - 4) * value;
  SII(3,3,1,1) = SII(3,3,2,2) = nu * invnu - (16*nu + 4) * value;
  SII(1,2,1,2) = SII(1,2,2,1) = SII(2,1,1,2) = SII(2,1,2,1) = (7 - 8*nu) * value;
  SII(1,3,1,3) = SII(1,3,3,1) = SII(3,1,1,3) = SII(3,1,3,1) = SII(2,3,2,3) = SII(2,3,3,2) = SII(3,2,2,3) = SII(3,2,3,2) = .5 + (4*nu - 8) * value;

  return SII;
}


/*!
  \brief Computes Eshelby tensor 2323, 2332, 3223 and 3232 coordinates for oblate spheroid

  \f$ \displaystyle S^{II}_{2323} = S^{II}_{2332} = S^{II}_{3223} = S^{II}_{3232} = 
  \frac{1}{2 (1-\nu)(a^2 - c^2)} \left[ a^2 - \nu (a^2 - c^2) + \frac{1}{2} \left( \displaystyle \frac{a^2 c \arccos \left( \frac{c}{a} \right)}{\sqrt{a^2 - c^2}} - c^2 \right) \left( \nu - \frac{2a^2 + c^2}{a^2 - c^2} \right) \right]. \f$ \n
  If we note \f$ \zeta =  \displaystyle \frac{c}{a} \f$ then \n
  \f$ \displaystyle S^{II}_{2323} =
  \frac{1}{2 (1-\nu)(1 - \zeta^2)} \left[ 1 - \nu (1 - \zeta^2) + \frac{1}{2} \left( \displaystyle \frac{\zeta \arccos(\zeta)}{\sqrt{1 - \zeta^2}} - \zeta^2 \right) \left( \nu - \frac{2 + \zeta^2}{1 - \zeta^2} \right) \right]. \f$ \n
  
  \param C elasticity tensor
  \param geometry penny shape length \f$ (a,a,c) \f$
  \return Eshelby tensor coordiantes 2323 2332 3223 3232
*/

//---------------------------------------------------------------------------------------------------------------------
template <class T> T Eshelby::oblate_spheroid_2332 (const isotropic_elasticity_tensor<T>& C, const vector<T>& geometry)
//---------------------------------------------------------------------------------------------------------------------
{
  assert (geometry.dim() == 3);
  assert (abs(geometry[1] - geometry[2]) < epsilon);

  T nu = C.Poisson();
  T zeta = geometry[3] / geometry[1];
  assert (zeta < 1);
  T zeta2 = pow(zeta,2);
  T coef = 1/(1-zeta2);

  T Tii = .5 * (coef-nu)  +  0.25 * zeta * coef * (nu - (2+zeta2)*coef) * (acos(zeta) * sqrt(coef) - zeta);
  Tii /= (1-nu);

  return Tii;
}


}


#endif
