/***************************************************************
 *                   Material Object Library                   *
 *     Class TII : declaration for TII interaction tensor      *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.0.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2005,2006,2008,2009,2012 COLLARD Christophe
 * copyright © 2006 PARGUEZ Cristelle
 * copyright © 2005,2006,2008,2009,2012 Centre National de la Recherche Scientifique
 * copyright © 2005,2006,2008,2009 Arts et Métiers ParisTech
 * copyright © 2005,2006 Université de Valenciennes et du Hainaut-Cambrésis
 * copyright © 2005,2006,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2005,2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::TII
    \brief Numerical computation of the interaction tensor \f$T^{II} \f$ for spheroid inclusions

    \htmlonly 
    <FONT color="#838383">

    TII belongs to Material Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    Let us note  \f$ C \f$ the \f$ 4^\text{th} \f$ order elastic tensor, \f$ V_I \f$ the inclusion volume, \f$ \delta_{pj} \f$ the Kronecker symbol and \f$ \delta \f$ the Dirac measure.
    We compute the interaction tensor \n
    \f$ \displaystyle T^{II}_{ijkl} = - \frac{1}{2 V_I} \int_{V^I} \int_{V^I} \left( G_{ik,jl}(r-r^\prime) + G_{jk,il}(r-r^\prime) \right) dV dV^\prime = \frac{1}{2} \left( t^{II}_{ijkl} + t^{II}_{jikl} \right) \f$, \n
    where the Green tensor \f$ G_{ik,jl} \f$ solves the Navier equation \f$ C_{plik} G_{ij,kl}(r) + \delta_{pj} \delta(r) = 0 \f$, and where \f$ t^{II} \f$ is given by \n
    \f$ \displaystyle t^{II}_{ikjl} = - \frac{1}{V_I} \int_{V^I} \int_{V^I} G_{ik,jl} (r - r^\prime) dr dr^\prime \f$. \n
    In the particular case of spheroids, \f$ t^{II} \f$ reads \n
    \f$ \displaystyle t^{II}_{ijkl} = \frac{a b c}{3 V_I} \displaystyle \int_0^\pi \int_0^{2\pi} \sin \theta \left[ C_{pqst} \chi_t (\theta,\varphi) \chi_q (\theta,\varphi) \right]_{ik}^{-1} \chi_j (\theta,\varphi) \chi_l (\theta,\varphi) d\varphi d\theta = \frac{1}{4 \pi} \displaystyle \int_0^\pi \int_0^{2\pi} \sin \theta \left[ C_{pqst} \chi_t (\theta,\varphi) \chi_q (\theta,\varphi) \right]_{ik}^{-1} \chi_j (\theta,\varphi) \chi_l (\theta,\varphi) d\varphi d\theta \f$ \n
    with \n
    \f$ \left\{ \begin{aligned} \chi_1 &= \sin\theta \cos\varphi, \\ \chi_2 &= \frac{a}{b} \sin\theta \sin\varphi, \\ \chi_3 &= \frac{a}{c} \cos\theta, \end{aligned} \right. \f$ \n
    where \f$ (\theta,\varphi) \f$ are the spherical coordinates angles and where \f$ (a,b,c) \f$ are the spheroid length.

    (see Omar Fassi-Fehri - Th&#232;se de Doctorat d'Etat, ISGMP, LPMM - 1985). \n

    \author copyright \htmlonly &#169; \endhtmlonly 2005, 2006, 2008, 2009, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; 2005, 2006, 2008, 2009, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006, 2008, 2009 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
            copyright \htmlonly &#169; \endhtmlonly 2006 Cristelle PARGUEZ \n
            copyright \htmlonly &#169; 2005, 2006, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
            copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.0.0
    \date 2005-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type TII
#endif

#if !defined _TII_h
#define _TII_h


#ifndef __iostream
#include <iostream>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __tensors2_hpp
#include "MOL++/tensors2.hpp"
#endif

#ifndef __tensors3_hpp
#include "MOL++/tensors3.hpp"
#endif

#ifndef __tensors4_hpp
#include "MOL++/tensors4.hpp"
#endif

#ifndef __symtensors_hpp
#include "MOL++/symtensors4.hpp"
#endif

#ifndef __integration_hpp
#include "MOL++/integration.hpp"
#endif

#ifndef __isotropic_elasticity_tensors_hpp
#include "isotropic elasticity tensors.hpp"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//==========================
template <class T> class TII
//==========================
{
  private :
    integration<T> Itg1, Itg2;
    tensor3<T> Khi;

  public :
    TII () {}        // default constructor
    TII (int, int);  // uses a Gauss integration on tetha and phi to compute the Green tensor
    ~TII () {}       // destructor

    template<template <class> class tensor_type> tensor4<T> ellipsoid (const tensor_type<T>&, const vector<T>&); // computes TII for ellipsoidal inclusion
    tensor4<T> sphere (const isotropic_elasticity_tensor<T>& C); // computes TII for isotropic spheric inclusion
};


//=====Private methods for TII=============================================


//=====Public methods for TII==============================================


/*!
  \brief Constructor for TII with the number of points for integration

  We use a Gauss-Legendre integration to compute \f$ t^{II} \f$ (see Detailed description). \n
  So we compute the integration points \f$ x_i \f$ and weights \f$ \omega_i \f$ using the library \ref integration, as well as : \n
  \f$ \begin{aligned} & \zeta_1(x_i,x_j) = \sin(x_i) \cos(x_j) \\ & \zeta_2(x_i,x_j) = \sin(x_i) \sin(x_j) \\ & \zeta_3(x_i,x_j) = \cos(x_i) \end{aligned} \f$ \n
  which are independant of the inclusion geometry.

  \param nItg1 number of points for the integration on \f$ \theta \f$
  \param nItg2 number of points for the integration on \f$ \varphi \f$
*/

//---------------------------------------------------
template <class T> TII<T>::TII (int nItg1, int nItg2)
//---------------------------------------------------
{
  assert ((nItg1>0) && (nItg2>0));

  // Gauss-Legendre numerical integration
  Itg1.Gauss(nItg1,0,pi());
  Itg2.Gauss(nItg2,0,2*pi());

  Khi.assign(nItg1,nItg2,3);
  for (int i=1; i<=Itg1.NbIntPts(); i++)   // integration on tetha
    for (int j=1; j<=Itg2.NbIntPts(); j++) // integration on phi
      { Khi(i,j,1) = sin(Itg1.point(i)) * cos(Itg2.point(j));
	Khi(i,j,2) = sin(Itg1.point(i)) * sin(Itg2.point(j));
	Khi(i,j,3) = cos(Itg1.point(i));
      }
}


/*!
  \brief Numerical computation of \f$ T^{II} \f$ 

  We approximate \f$ t^{II} \f$ with \n
  \f$ t^{II}_{pqkl} \approx \frac{1}{4 \pi}\underset{i}{\sum} \underset{j}{\sum} \sin(x_i) \chi_l(x_i,x_j) \chi_q(x_i,x_j) \left[ L_{rsmn} \chi_s(x_i,x_j) \chi_n(x_i,x_j) \right]^{-1}_{pk} \omega_i \omega_j \f$, \n
  where \f$ \chi_k(x_i,x_j) = R_k \zeta_k(x_i,x_j)\f$, where \f$ \zeta_i \f$ and the Gauss-Legendre points and weights \f$ (x_i, \omega_i) \f$ are computed in the constructor. So we get \n
  \f$ T^{II}_{ijkl} \approx \frac{1}{2} \left( t^{II}_{ijkl} + t^{II}_{jikl} \right) \f$

  \param L elasticity tensor
  \param geometry spheroid geometry \f$ R = (a,b,c) \f$
  \return \f$ T^{II} \f$
*/

//-------------------------------------------------------------------------------
template <class T> template <template <class> class tensor_type>
tensor4<T> TII<T>::ellipsoid (const tensor_type<T>& L, const vector<T>& geometry)
//-------------------------------------------------------------------------------
{
  assert (geometry.dim() == 3);
  assert (Itg1.NbIntPts() && Itg2.NbIntPts());

  tensor2<T> M;
  tensor4<T> tii(3);
  T scaling1 = geometry[1] / geometry[2];
  T scaling2 = geometry[1] / geometry[3];
  T scaling = 1 / (4 * pi());

  // Gauss-Legendre numerical integration
  for (int i=1; i<=Itg1.NbIntPts(); i++)   // integration over tetha
    { T wi_sin_xi = Itg1.weight(i) * sin (Itg1.point(i)) * scaling;
      for (int j=1; j<=Itg2.NbIntPts(); j++) // integration over phi
	{ vector<T> khi (3,false);
	  khi[1] = Khi(i,j,1);
	  khi[2] = scaling1 * Khi(i,j,2);
	  khi[3] = scaling2 * Khi(i,j,3);
	  M = (khi |  L) * khi; // M(p,q) = L(p,k,q,l) * khi[k] * khi[l]
	  M = gauss (M);       // (L(i,k,j,l) * khi[k] * khi[l])^(-1)
	  T wi_wj_sin_xi = wi_sin_xi * Itg2.weight(j);
	
	  for (int p=1; p<=3; p++)
	    for (int q=1; q<=3; q++)
	      for (int k=1; k<=3; k++)
		for (int l=1; l<=3; l++)
		  tii(p,q,k,l) += wi_wj_sin_xi * khi[l] * khi[q] * M(p,k);
	}
    }
  tii.approximation();

  tensor4<T> T2I(3,false);
  for (int p=1; p<=3; p++)
    for (int q=1; q<=3; q++)
      for (int k=1; k<=3; k++)
  	for (int l=1; l<=3; l++)
  	  T2I(p,q,k,l) = .5 * (tii(p,q,k,l) + tii(q,p,k,l));

  return T2I;
}


/*!
  \brief Numerical computation of \f$ T^{II} \f$ for a spherical isotropic material

  In this particular case, the expression of \f$ t^{II} \f$ reads \n
  \f$ \displaystyle t^{II}_{ijkl} = \frac{1}{30 \mu (1-\nu)} \left[ (9-10\nu) \delta_{ik} \delta_{jl} - ( \delta_{ij} \delta_{kl} + \delta_{il} \delta_{jk} ) \right] \f$, \n
  where \f$ \mu \f$ is the shear modulus and \f$ \nu \f$ is the Poisson's ration (see \ref isotropic_tensor). Hence we get \f$ T^{II}_{ijkl} = \frac{1}{2} \left( t^{II}_{ijkl} + t^{II}_{jikl} \right) \f$

  \param C isotropic elastic tensor
  \param geometry sphere radius R
  \return \f$ T^{II} \f$
*/

//------------------------------------------------------------------------------------
template <class T> tensor4<T> TII<T>::sphere (const isotropic_elasticity_tensor<T>& C)
//------------------------------------------------------------------------------------
{
  tensor4<T> tii(3); // use and return a cubic tensor when class cubic elasticity tensor is ready
  T nu = C.Poisson();
  T coef = 1 / (30 * C.Shear() * (1-nu));

  tii(1,1,1,1) = tii(2,2,2,2) = tii(3,3,3,3) = (7 - 10 * nu) * coef;
  tii(1,1,2,2) = tii(2,2,3,3) = tii(3,3,1,1) = tii(1,1,3,3) = tii(2,2,1,1) = tii(3,3,2,2) = -coef;
  tii(1,2,1,2) = tii(2,1,1,2) = tii(1,2,2,1) = tii(2,1,2,1) = tii(2,3,2,3) = tii(3,2,2,3) = tii(2,3,3,2) = tii(3,2,3,2) = tii(3,1,3,1) = tii(1,3,3,1) = tii(3,1,1,3) = tii(1,3,1,3) = (4 - 5*nu) * coef;

  return tii;
}


}


#endif
