/***************************************************************
 *                   Materials Object Library                  *
 *       Conversion tools Library using Walpole notations      *
 *                    simula.plus@cemes.fr                     *
 *	             GNU/linux version 3.0.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2009,2012 COLLARD Christophe
 * copyright © 2009 BERBENNI Stephane
 * copyright © 2009,2012 Centre National de la Recherche Scientifique
 * copyright © 2009 Arts et Métiers ParisTech
 * copyright © 2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace walpole
    \brief Walpole conversion tools
*/

/*! \class walpole::walpole
    \brief conversion tools library using Walpole notations for basic mathematical objects \n

    \htmlonly 
    <FONT color="#838383">

    walpole belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2009, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; \endhtmlonly 2009 Stephane BERBENNI \n
            copyright \htmlonly &#169; 2009, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2009 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
            copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.0.0
    \date 2009-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type walpole
#endif

#ifndef __walpole_hpp
#define __walpole_hpp

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __parameters_h
#include "parameters.h"
#endif

#ifndef __maths_hpp
#include "MOL++/maths.hpp"
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __symtensors2_hpp
#include "MOL++/symtensors2.hpp"
#endif

#ifndef __tensors4_hpp
#include "MOL++/tensors4.hpp"
#endif

using namespace std;

namespace walpole
{


//===========
class walpole
//===========
{
  public :
    template <class T> friend symtensor2<T> vector2symtensor  (vector<T>);
    template <class T> friend tensor4<T>    matrix2symtensor  (matrix<T>);
    template <class T> friend vector<T>     symtensor2vector  (const symtensor2<T>&);
    template <class T> friend matrix<T>     symtensor2matrix  (const tensor4<T>&);
};


//=====Public methods for conversion tools===================================


/*!
  \brief Converts a vector into a symmetric \f$ 2^\text{nd} \f$ order tensor.

  Let \f$ v \f$ be a vector of \f$ \mathbb{R}^6 \f$. We define \f$ T \in \mathbb{R}^3 \times \mathbb{R}^3 \f$ by
  \f$ Tn = \left( \begin{matrix}
      T_{11} = v_1 & T_{12} = \frac{1}{\sqrt{2}} v_6 & T_{13} = \frac{1}{\sqrt{2}} v_5 \\ \\
      T_{12} & T_{22} = v_2 & T_{23} = \frac{1}{\sqrt{2}} v_4 \\ \\
      T_{13} & T_{23} & T_{33} = v_3
      \end{matrix} \right) \f$.

  \param v vector
  \return \f$ T \f$
  \sa symtensor2vector
*/

//-------------------------------------------------------------
template <class T> symtensor2<T> vector2symtensor (vector<T> v)
//-------------------------------------------------------------
{
  assert (v.dim() == 6);

  v.permute(4,6);
  symtensor2<T> tsr =& mol::vector2symtensor (v);

  return tsr;
}


/*!
  \brief Converts a matrix into symmetric \f$ 4^\text{th} \f$ order tensor.

  Let \f$ M \f$ be a matrix in  \f$ \mathbb{R}^6 \times \mathbb{R}^6 \f$. We define \f$ C \f$ the \f$ 4^\text{th} \f$ order tensor in \f$ \mathbb{R}^3 \times \mathbb{R}^3 \times \mathbb{R}^3 \times \mathbb{R}^3 \f$ such as \f$ C_{ijkl} = C_{jikl} = C_{ijlk} \f$ and
  \f$ C_{ij} = \alpha_{ij} * M_{ij} \f$, where \n
  \f$ \alpha_{ij} = \left\{
      \begin{aligned}
      & 1 \text{ if } i=j \text{ and } k=l, \\
      & \frac{1}{2} \text{ if } i \neq j \text{ and } k \neq l, \\
      & \frac{1}{\sqrt{2}} \text{ elsewhere},
      \end{aligned}
  \right.
  \f$ \n
  and where we use the following notation for \f$ C \f$ indices \n
  \f$ \begin{array}{| c | c | c | c | c | c | c |}
  \hline C_{pq} : p \equiv & 1 & 2 & 3 & 4 & 5 & 6 \\
  \hline C_{ijkl} : ij \equiv & 11 & 22 & 33 & 23 & 13 & 12 \\
  \hline
  \end{array}
  \f$
  \param mat matrix M
  \return C
  \n\n
  \f$ \textbf{Example :} \text{ elasticity constitutive equation} \f$ \n
  \f$ \begin{pmatrix}
      \sigma_{11} \\ \sigma_{22} \\ \sigma_{33} \\ \sqrt{2} \sigma_{23} \\ \sqrt{2} \sigma_{13} \\ \sqrt{2} \sigma_{12}
      \end{pmatrix}
      =
      \begin{pmatrix}
      C_{1111} & C_{1122} & C_{1133} & \sqrt{2} C_{1112} & \sqrt{2} C_{1123} & \sqrt{2} C_{1113} \\
      C_{2211} & C_{2222} & C_{2233} & \sqrt{2} C_{2223} & \sqrt{2} C_{2213} & \sqrt{2} C_{2212} \\
      C_{3311} & C_{3322} & C_{3333} & \sqrt{2} C_{3323} & \sqrt{2} C_{3313} & \sqrt{2} C_{3312} \\
      \sqrt{2} C_{2311} & \sqrt{2} C_{2322} & \sqrt{2} C_{2333} & 2 C_{2323} & 2 C_{2313} & 2 C_{2312} \\
      \sqrt{2} C_{1311} & \sqrt{2} C_{1322} & \sqrt{2} C_{1333} & 2 C_{1323} & 2 C_{1313} & 2 C_{1312} \\
      \sqrt{2} C_{1211} & \sqrt{2} C_{1222} & \sqrt{2} C_{1233} & 2 C_{1223} & 2 C_{1213} & 2 C_{1212} \\
      \end{pmatrix}
      \begin{pmatrix}
      \EuScript{E}_{11} \\ \EuScript{E}_{22} \\ \EuScript{E}_{33} \\ \sqrt{2} \EuScript{E}_{23} \\ \sqrt{2} \EuScript{E}_{13} \\ \sqrt{2} \EuScript{E}_{12}
      \end{pmatrix}
  \f$ \n
*/

//------------------------------------------------------------
template <class T> tensor4<T> matrix2symtensor (matrix<T> mat)
//------------------------------------------------------------
{
  assert (mat.Rows()==6 && mat.Columns()==6);

  mat.permute_columns(4,6);
  mat.permute_rows(4,6);

  tensor4<T> tsr =& mol::matrix2symtensor (mat);

  return tsr;
}


/*!
  \brief Converts a symmetric \f$ 2^\text{nd} \f$ order tensor into a vector.

  Let \f$ T \in \mathbb{R}^3 \times \mathbb{R}^3 \f$. 
  We define vector \f$ v  \in \mathbb{R}^6 \f$ through \f$ T \f$ by \n
  \f$ T = \left( \begin{matrix}
      T_{11} = v_1 & T_{12} = \frac{1}{\sqrt{2}} v_6 & T_{13} = \frac{1}{\sqrt{2}} v_5 \\ \\
      T_{12} & T_{22} = v_2 & T_{23} = \frac{1}{\sqrt{2}} v_4 \\ \\
      T_{13} & T_{23} & T_{33} = v_3
      \end{matrix} \right) \f$.

  \param tsr symmetric \f$ 2^\text{nd} \f$ order tensor T
  \return vector \f$ v \f$
*/

//----------------------------------------------------------------------
template <class T> vector<T> symtensor2vector (const symtensor2<T>& tsr)
//----------------------------------------------------------------------
{
  assert (tsr.dim1()==3 && tsr.dim2()==3);

  vector<T> v = mol::symtensor2vector(tsr);
  v.permute(4,6);

  return v;
}


/*!
  \brief Converts a symmetric \f$ 4^\text{th} \f$ order tensor into a matrix.

  Let \f$ C \f$ be a \f$ 4^\text{th} \f$ order tensor in \f$ \mathbb{R}^3 \times \mathbb{R}^3 \times \mathbb{R}^3 \times \mathbb{R}^3 \f$ such as \f$ C_{ijkl} = C_{jikl} = C_{ijlk} \f$.
  We define the matrix \f$ M \in \mathbb{R}^6 \times \mathbb{R}^6 \f$ such as \n
  \f$ M_{ij} = \alpha_{ij} * C_{ij} \f$, where \n
  \f$ \alpha_{ij} = \left\{
      \begin{aligned}
      & 1 \text{ if } i=j \text{ and } k=l, \\
      & 2 \text{ if } i \neq j \text{ and } k \neq l, \\
      & \sqrt{2} \text{ elsewhere},
      \end{aligned}
  \right.
  \f$ \n
  and where we use the following notation for \f$ C \f$ indices \n
  \f$ \begin{array}{| c | c | c | c | c | c | c |}
  \hline C_{pq} : p \equiv & 1 & 2 & 3 & 4 & 5 & 6 \\
  \hline C_{ijkl} : ij \equiv & 11 & 22 & 33 & 23 & 13 & 12 \\
  \hline
  \end{array}
  \f$
  \param tsr \f$ 4^\text{th} \f$ order tensor C
  \return M
  \n\n
  \f$ \textbf{Example :} \text{ elasticity constitutive equation} \f$ \n
  \f$ \begin{pmatrix}
      \sigma_{11} \\ \sigma_{22} \\ \sigma_{33} \\ \sqrt{2} \sigma_{23} \\ \sqrt{2} \sigma_{13} \\ \sqrt{2} \sigma_{12}
      \end{pmatrix}
      =
      \begin{pmatrix}
      C_{1111} & C_{1122} & C_{1133} & \sqrt{2} C_{1112} & \sqrt{2} C_{1123} & \sqrt{2} C_{1113} \\
      C_{2211} & C_{2222} & C_{2233} & \sqrt{2} C_{2223} & \sqrt{2} C_{2213} & \sqrt{2} C_{2212} \\
      C_{3311} & C_{3322} & C_{3333} & \sqrt{2} C_{3323} & \sqrt{2} C_{3313} & \sqrt{2} C_{3312} \\
      \sqrt{2} C_{2311} & \sqrt{2} C_{2322} & \sqrt{2} C_{2333} & 2 C_{2323} & 2 C_{2313} & 2 C_{2312} \\
      \sqrt{2} C_{1311} & \sqrt{2} C_{1322} & \sqrt{2} C_{1333} & 2 C_{1323} & 2 C_{1313} & 2 C_{1312} \\
      \sqrt{2} C_{1211} & \sqrt{2} C_{1222} & \sqrt{2} C_{1233} & 2 C_{1223} & 2 C_{1213} & 2 C_{1212} \\
      \end{pmatrix}
      \begin{pmatrix}
      \EuScript{E}_{11} \\ \EuScript{E}_{22} \\ \EuScript{E}_{33} \\ \sqrt{2} \EuScript{E}_{23} \\ \sqrt{2} \EuScript{E}_{13} \\ \sqrt{2} \EuScript{E}_{12}
      \end{pmatrix}
  \f$ \n
*/

//-------------------------------------------------------------------
template <class T> matrix<T> symtensor2matrix (const tensor4<T>& tsr)
//-------------------------------------------------------------------
{
  assert (tsr.dim1()==3 && tsr.dim2()==3 && tsr.dim3()==3 && tsr.dim4()==3);

  matrix<T> mat = mol::symtensor2matrix(tsr);
  mat.permute_rows(4,6);
  mat.permute_columns(4,6);

  return mat;
}


}


#endif
