/***************************************************************
 *                   Materials Object Library                  *
 *     Class MatOoLbox : declaration for materials toolbox     *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.2.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2005,2006,2007,2008,2010,2012 COLLARD Christophe
 * copyright © 2005,2006,2007,2008,2010,2012 Centre National de la Recherche Scientifique
 * copyright © 2005,2006,2007,2008,2010 Arts et Métiers ParisTech
 * copyright © 2005,2006,2007 Université de Valenciennes et du Hainaut-Cambrésis
 * copyright © 2005,2006,2007,2008,2010 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::MatOoLbox
    \brief Toolbox for materials object libraries

    \htmlonly 
    <FONT color="#838383">

    MatOoLbox belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2005, 2006, 2007, 2008, 2010, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; 2005, 2006, 2007, 2008, 2010, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006, 2007, 2008, 2010 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006, 2007 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
            copyright \htmlonly &#169; 2005, 2006, 2007, 2008, 2010 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006, 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	    copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.2.0
    \date 2005-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type MatOoLbox
#endif

#ifndef __materials_toolbox_hpp
#define __materials_toolbox_hpp


#ifndef __iostream
#include <iostream>
#endif

#ifndef __fstream
#include <fstream>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __math_h
#include <math.h>
#endif

#ifndef __string_h
#include <string.h>
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __tensors2_hpp
#include "MOL++/tensors2.hpp"
#endif

#ifndef __tensorsS4_hpp
#include "MOL++/tensors4.hpp"
#endif

#ifndef __isotropic_tensors_hpp
#include "MateriOL++/isotropic tensors.hpp"
#endif

#ifndef __isotropic_elasticity_tensors_hpp
#include "MateriOL++/isotropic elasticity tensors.hpp"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//=============
class MatOoLbox
//=============
{
  template <class T> friend T Von_Mises_stress (const tensor2<T>&);
  template <class T> friend T Von_Mises_strain (const tensor2<T>&);
  template <class T, template <class> class tensor_type> friend tensor2<T> constitutive_equation (const tensor_type<T>&, const tensor2<T>&, const tensor2<T>&, const T&, int);
  template <class T> friend matrix<T> global2local_basis (T, T, T);
  template <class T> friend matrix<T> local2global_basis (T, T, T);
};


//=====Public methods for materials toolbox=============================================


/*!
  \brief Computes the Von Mises equivalent stress

  \f$ \displaystyle \sigma_m = \sqrt{\frac{3}{2} dev(\sigma) :: dev(\sigma)} \f$

  \param stress stress tensor \f$ \sigma \f$
  \return Von Mises equivalent stress tensor
  \sa \ref tensor2::dev "dev"
*/

//==============================================================
template <class T> T Von_Mises_stress (const tensor2<T>& stress)
//==============================================================
{
  assert (stress.dim1());

  tensor2<T> deviatoric =& (dev(stress));

  return sqrt((T) 1.5 * (deviatoric|deviatoric));
}


/*!
  \brief Computes the Von Mises equivalent strain

  \f$ \displaystyle \EuScript{E}_m = \sqrt{\frac{2}{3} \EuScript{E} :: \EuScript{E}} \f$

  \param stress strain tensor (\f$ \EuScript{E} \f$)
  \return Von Mises equivalent strain tensor
*/

//==============================================================
template <class T> T Von_Mises_strain (const tensor2<T>& strain)
//==============================================================
{
  assert (strain.dim1());

  return sqrt((T) 2 * (strain|strain) / (T) 3);
}


/*!
  \brief computes the constitutive equations \f$ \Sigma = L E - M T \f$ and \f$ E = G \Sigma + N T \f$

  The stress tensor \f$ \Sigma \f$ and the strain tensor \f$ E \f$ are related through the following constitutive equations : \n
  \f$ \Sigma = L E - M T \f$, with \f$ T4 = L \; ; \; T2 = M \; ; \; stn = E \; ; \; str = \Sigma \; ; \; sgn = -1\f$. \n
  \f$ E = G \Sigma + N T \f$, with \f$ T4 = G \; ; \; T2 = N \; ;\; stn = \Sigma \; ;\; str = E \; ;\; sgn = 1 \f$. \n

  \param str strain or stress tensor (E or \f$ \Sigma \f$)
  \param To temperature T
  \param T4 L or G 4th order tensor
  \param T2 M or N 2nd order tensor
  \param sgn sign \f$ \pm 1 \f$
  \return tsr stress or strain tensor
*/

//----------------------------------------------------------------------------------------------------------------------------
template <class T, template <class> class tensor_type> 
tensor2<T> constitutive_equation (const tensor_type<T>& T4, const tensor2<T>& T2, const tensor2<T>& str, const T& To, int sgn)
//----------------------------------------------------------------------------------------------------------------------------
{
  // dS_ij = L_ijkl dE_kl - M_ij dT
  tensor2<T> tsr = (T4 || str);

  if (sgn == 1) tsr += T2 * To;
  else if (sgn == -1) tsr-= T2 * To;
  else assert(false); // sgn value error

  return tsr;
}


/*!
  \brief Build the rotation matrix from the local to the global basis

  Let P be the rotation matrix from the local basis to the global basis (see \ref local2global_basis). Then the rotation matrix from the global basis to the local basis is \f$ P^t \f$ (since P is orthogonal).

  \param phi1 Euler angle \f$ \varphi_1 \f$
  \param phi Euler angle \f$ \phi \f$
  \param phi2 Euler angle \f$ \varphi2 \f$
  \return \f$ P^t \f$
*/

//---------------------------------------------------------------------
template <class T> matrix<T> global2local_basis (T phi1, T phi, T phi2)
//---------------------------------------------------------------------
{
  matrix<T> passgl (3,3);

  passgl(1,1) =  cos(phi1) * cos(phi2)  -  sin(phi1) * cos(phi) * sin(phi2);
  passgl(2,1) = -cos(phi1) * sin(phi2)  -  sin(phi1) * cos(phi) * cos(phi2);
  passgl(3,1) =  sin(phi1) * sin(phi);
  passgl(1,2) =  sin(phi1) * cos(phi2)  +  cos(phi1) * cos(phi) * sin(phi2);
  passgl(2,2) = -sin(phi1) * sin(phi2)  +  cos(phi1) * cos(phi) * cos(phi2);
  passgl(3,2) = -cos(phi1) * sin(phi);
  passgl(1,3) =  sin(phi)  * sin(phi2);
  passgl(2,3) =  sin(phi)  * cos(phi2);
  passgl(3,3) =  cos(phi);

  return passgl;
}


/*!
  \brief Build the rotation matrix from the local to the global basis

  Euler angles \f$ (\varphi_1,\phi,\varphi_2) \f$ are a means of representing the spatial orientation of any basis of the space (so called local basis or crystal orientation in material science) as a composition of three rotations from a reference basis (so called global basis or sample orientation in material science). The first rotation is by an angle \f$ \varphi_1 \f$ about the z-axis, the second is by an angle \f$ \phi \in [0,\pi] \f$ about the x-axis, and the third is by an angle \f$ \varphi_2 \f$ about the z-axis (again). The associated rotation matrices are respectively given by: \n
  \f$ P_1 = \begin{pmatrix} \cos\varphi_1 & -\sin\varphi_1 & 0 \\ \sin\varphi_1 & \cos\varphi_1 & 0 \\ 0 & 0 & 1 \end{pmatrix}, \; \; P_2 = \begin{pmatrix} 1 & 0 & 0 \\ 0 & \cos\phi & -\sin\phi \\ 0 & \sin\phi & \cos\phi \end{pmatrix}, \; \; P_3 = \begin{pmatrix} \cos\varphi_2 & -\sin\varphi_2 & 0 \\ \sin\varphi_2 & \cos\varphi_2 & 0 \\ 0 & 0 & 1 \end{pmatrix}. \f$ \n
  Let P be the rotation matrix from the local basis \f$ (\textbf{e}^l_1,\textbf{e}^l_2,\textbf{e}^l_3) \f$ to the global basis \f$ (\textbf{e}^g_1,\textbf{e}^g_2,\textbf{e}^g_3) \f$ : \f$ \displaystyle \textbf{e}^l_i = \underset{j}{\sum} P_{ji} \textbf{e}^g_j \f$. Then P reads : \n
    \f$ P = P_1 P_2 P_3 = \left(
        \begin{matrix} 
           \cos(\varphi_1) * \cos(\varphi_2)  -  \sin(\varphi_1) * \cos(\phi) * \sin(\varphi_2)
        & -\cos(\varphi_1) * \sin(\varphi_2)  -  \sin(\varphi_1) * \cos(\phi) * \cos(\varphi_2)
	&  \sin(\varphi_1) * \sin(\phi) \\
	   \sin(\varphi_1) * \cos(\varphi_2)  +  \cos(\varphi_1) * \cos(\phi) * \sin(\varphi_2)
	& -\sin(\varphi_1) * \sin(\varphi_2)  +  \cos(\varphi_1) * \cos(\phi) * \cos(\varphi_2)
	& -\cos(\varphi_1) * \sin(\phi) \\
	   \sin(\phi)  * \sin(\varphi_2)
	&  \sin(\phi)  * \cos(\varphi_2)
	&  \cos(\phi)
	\end{matrix}
	\right) \f$ \n
    So we can express the coordinates \f$ x^g_i \f$ of a vector \f$ \textbf{u} \f$ in the global basis through its coordinates in the local basis \f$x^l_i \f$ :\n
    \f$ \textbf{u} = \underset{i}{\sum} x^g_i \textbf{e}^g_i = \underset{j}{\sum} x^l_j \textbf{e}^l_j = \underset{j}{\sum} x^l_j ( \underset{i}{\sum} P_{ij} \textbf{e}^g_i ) = \underset{i}{\sum} (\underset{j}{\sum} P_{ij} x^l_j) \textbf{e}^g_i \f$ \n
\f$ \implies X^g = P X^l \;\; (x^g_i = P_{ij} x^l_j) \f$.

  \param phi1 Euler angle \f$ \varphi_1 \f$
  \param phi Euler angle \f$ \phi \f$
  \param phi2 Euler angle \f$ \varphi2 \f$
  \return P
*/

//---------------------------------------------------------------------
template <class T> matrix<T> local2global_basis (T phi1, T phi, T phi2)
//---------------------------------------------------------------------
{
  matrix<T> passlg(3,3,false); // allocates memory without initializing data to zero

  passlg(1,1) =  cos(phi1) * cos(phi2)  -  sin(phi1) * cos(phi) * sin(phi2);
  passlg(1,2) = -cos(phi1) * sin(phi2)  -  sin(phi1) * cos(phi) * cos(phi2);
  passlg(1,3) =  sin(phi1) * sin(phi);
  passlg(2,1) =  sin(phi1) * cos(phi2)  +  cos(phi1) * cos(phi) * sin(phi2);
  passlg(2,2) = -sin(phi1) * sin(phi2)  +  cos(phi1) * cos(phi) * cos(phi2);
  passlg(2,3) = -cos(phi1) * sin(phi);
  passlg(3,1) =  sin(phi)  * sin(phi2);
  passlg(3,2) =  sin(phi)  * cos(phi2);
  passlg(3,3) =  cos(phi);

  return passlg;
}


}


#endif
