/***************************************************************
 *                   Materials Object Library                  *
 *         Class systems: declaration for slip sytems          *
 *                                    or martensitic variants  *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.2.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2008,2009,2011m2012 COLLARD Christophe
 * copyright © 2008,2009,2011,2012 Centre National de la Recherche Scientifique
 * copyright © 2008,2009,2011 Arts et Métiers ParisTech
 * copyright © 2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2011 Laboratoire d'Etude des Microstructures et de Mécanique des Matériaux (LEM3 - CNRS)
 * copyright © 2011,2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::systems
    \brief slip systems or martensitic variants properties

    \htmlonly
    <FONT color="#838383">

    systems belongs to Materials Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    We define here : \n
    the number of systems (plastic slip systems or martensitic variants) \n
    the Schmid tensor for each system
    \sa Schmid_tensor

    \author copyright \htmlonly &#169; \endhtmlonly 2008, 2009, 2011, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; 2008, 2009, 2011, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2008, 2009, 2011 Arts et M&#233;tiers ParisTech \endhtmlonly \n
            copyright \htmlonly &#169; 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2011 Laboratoire d'Etude des Microstructures et de MM&#233;canique des MatM&#233;riaux (LEM3 - CNRS) \endhtmlonly \n
            copyright \htmlonly &#169; 2011, 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.2.0
    \date 2008-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type martensite
#endif

#ifndef __systems_hpp
#define __systems_hpp

#ifndef __stdlib_h
#include <stdlib.h>
#endif

#ifndef __iostream
#include <iostream>
#endif

#ifndef __sstream
#include <sstream>
#endif

#ifndef __fstream
#include <fstream>
#endif

#ifndef __iomanip
#include <iomanip>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __string_h
#include <string.h>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __symmatrix_hpp
#include "MOL++/symmatrix.hpp"
#endif

#ifndef __Schmid_tensor_hpp
#include "MateriOL++/Schmid tensor.hpp"
#endif

using namespace std;
using namespace materiol;

namespace materiol
{


//==============================
template <class T> class systems
//==============================
{
  private:
    int nbsystems;         // number of slip systems or martensitic variants
    Schmid_tensor<T> *Rij;  // Schmid tensors

    matrix<T> build_N24 (const vector<T>&);
    matrix<T> build_M24 (const vector<T>&);
    matrix<T> build_N12 (const vector<T>&);
    matrix<T> build_M12 (const vector<T>&);

  public:
    //    systems () {nbsystems=0;}  // default constructor
    systems (const char* = "/data/systems.dat");
    systems (const string = "/data/systems.dat");
    systems (ifstream&, int);
    systems (const string&, T);
    systems (const systems<T>&);                 // copy constructor
    ~systems ();                                 // destructor

    systems<T>& operator = (const systems<T>&);
    int nb_systems () {return nbsystems;}  // returns the number of slip systems
    vector<T> n(int);
    vector<T> m(int);
    Schmid_tensor<T>& R (int) const;   // returns the ith slip system
    template <class Tf> friend bool operator == (const systems<Tf>&, const systems<Tf>&);
    template <class Tf> friend bool operator != (const systems<Tf>&, const systems<Tf>&);
    template <class Tf> friend ostream& operator << (ostream&, const systems<Tf>&);
};


//=====Private methods for systems================================================


/*!
  \brief Computes the unit normal vector to the habit plane for 24 slip systems.

  \param n normal unit vector to the slip direction

  \return matrix with the 24 normal to the slip direction
*/

//---------------------------------------------------------------------
template <class T> matrix<T> systems<T>::build_N24 (const vector<T>& n)
//---------------------------------------------------------------------
{
  matrix<T> N(nbsystems,3);

  N(1,1) = N(2,1) = N(5,2) = N(6,2) = N(9,1) = N(10,1) = N(13,2) = N(14,2) = -n[1];
  N(3,1) = N(4,1) = N(7,2) = N(8,2) = N(11,1) = N(12,1) = N(15,2) = N(16,2) = n[1];
  for (int i=17; i<= 24; i++) N(i,3) = n[1];
  N(5,1) = N(7,1) = N(9,2) = N(11,2) = N(18,2) = N(19,1) = N(21,2) = N(22,1) = - n[2];
  N(1,2) = N(2,3) = N(3,2) = N(4,3) = N(6,3) = N(8,3) = N(10,3) = N(12,3) = N(13,3) = N(14,1) = N(15,3) = N(16,1) = N(17,1) = N(20,2) = N(23,2) = N(24,1) = n[2];
  N(6,1) = N(8,1) = N(10,2) = N(12,2) = N(17,2) = N(20,1) = N(21,1) = N(22,2) = -n[3];
  N(1,3) = N(2,2) = N(3,3) = N(4,2) = N(5,3) = N(7,3) = N(9,3) = N(11,3) = N(13,1) = N(14,3) = N(15,1) = N(16,3) = N(18,1) = N(19,2) = N(23,1) = N(24,2) = n[3];

  return N;
}


/*!
  \brief Computes the unit vector in the displacement direction.

  \param m normal unit vector in the displacement direction

  \return matrix with the 24 vectors in the slip direction
*/

//---------------------------------------------------------------------
template <class T> matrix<T> systems<T>::build_M24 (const vector<T>& m)
//---------------------------------------------------------------------
{
  matrix<T> M(nbsystems,3);

  M(1,1) = M(2,1) = M(5,2) = M(6,2) = M(9,1) = M(10,1) = M(13,2) = M(14,2) = -m[1];
  M(3,1) = M(4,1) = M(7,2) = M(8,2) = M(11,1) = M(12,1) = M(15,2) = M(16,2) = m[1];
  for (int i=17; i<= 24; i++) M(i,3) = m[1];
  M(6,1) = M(8,1) = M(10,2) = M(12,2) = M(17,2) = M(20,1) = M(21,1) = M(22,2) = - m[2];
  M(1,3) = M(2,2) = M(3,3) = M(4,2) = M(5,3) = M(7,3) = M(9,3) = M(11,3) = M(13,1) = M(14,3) = M(15,1) = M(16,3) = M(18,1) = M(19,2) = M(23,1) = M(24,2) = m[2];
  M(1,2) = M(2,3) = M(3,2) = M(4,3) = M(6,3) = M(8,3) = M(10,3) = M(12,3) = M(13,3) = M(14,1) = M(15,3) = M(16,1) = M(17,1) = M(20,2) = M(23,2) = M(24,1) = -m[3];
  M(5,1) = M(7,1) = M(9,2) = M(11,2) = M(18,2) = M(19,1) = M(21,2) = M(22,1) = m[3];

  return M;
}



//---------------------------------------------------------------------
template <class T> matrix<T> systems<T>::build_N12 (const vector<T>& n)
//---------------------------------------------------------------------
{
  matrix<T> N(nbsystems,3);

  N(1,1) = N(2,1) = N(3,1)= -n[1];
  for (int i=4; i<= 12; i++) N(i,1) = n[1];
  N(10,2) = N(11,2) = N(12,2)= -n[2];
  for (int i=1; i<= 9; i++) N(i,2) = n[2];
  N(7,3) = N(8,3) = N(9,3)= -n[3];
  for (int i=1; i<= 6; i++) N(i,3) = n[3];
  for (int i=10;i<=12; i++) N(i,3) = n[3];

  return N;
}


//---------------------------------------------------------------------
template <class T> matrix<T> systems<T>::build_M12 (const vector<T>& m)
//---------------------------------------------------------------------
{
  matrix<T> M(nbsystems,3);

  M(1,1) = M(4,1) = M(7,1) = M(10,1) = M(2,2) = M(5,2) = M(8,2) = M(11,2) = M(3,3) = M(6,3) = M(9,3) = M(12,3)= m[3];
  M(5,1) = M(11,1) = M(1,2) = M(4,2) = M(6,2) = M(9,2) = m[2];
  M(1,3) = M(2,1) = M(2,3) = M(3,1) = M(3,2) = M(4,3) = M(5,3) = M(6,1) = M(7,2) = M(7,3) = M(8,1) = M(8,3)  = M(9,1)= M(10,2) = M(10,3) = M(11,3) = M(12,1) = M(12,2) = m[1];
  
  return M;
}

//=====Public methods for systems=================================================


/*!
  \brief Reads systems properties in a file and builds the associated Schmid tensors.

  (default file is located in the current directory under 'data/systems.dat'). \n
  This file contains the following informations : \n
  the number of systems \n
  the unit normal vector to the habit / slip plane and the unit vector in the displacement / slip direction for the first system.\n
  \n
  Then the constructor builds the Schmid tensor for each system.
  \param path filename
*/

//-------------------------------------------------------
template <class T> systems<T>::systems (const char* path)
//-------------------------------------------------------
{
  string text;

  ifstream file(path, ios::in);
  assert (!file.fail());

  for (char ch=file.peek(); !isdigit(ch); ch=file.peek())
    if (ch == ' ') file.get();
    else getline (file, text);

  file >> nbsystems;
  Rij = new Schmid_tensor<T> [nbsystems];
  for (int i=0; i<nbsystems; i++)
    Rij[i] = Schmid_tensor<T>(file);  // construction of Schmid tensor for each martensitic variants
  file.close();
}


/*!
  \brief Reads systems properties in a file and builds the associated Schmid tensors.

  (default file is located in the current directory under 'data/systems.dat'). \n
  This file contains the following informations : \n
  the number of systems \n
  the unit normal vector to the habit / slip plane and the unit vector in the displacement / slip direction for the first system.\n
  \n
  Then the constructor builds the Schmid tensor for each system.
  \param filename filename
*/

//------------------------------------------------------------
template <class T> systems<T>::systems (const string filename)
//------------------------------------------------------------
{
  string text;

  ifstream file(filename.c_str(), ios::in);
  assert (!file.fail());

  for (char ch=file.peek(); !isdigit(ch); ch=file.peek())
    if (ch == ' ') file.get();
    else getline (file, text);

  file >> nbsystems;
  Rij = new Schmid_tensor<T> [nbsystems];
  for (int i=0; i<nbsystems; i++)
    Rij[i] = Schmid_tensor<T>(file);  // construction of Schmid tensor for each martensitic variants
  file.close();
}


/*!
  \brief Reads systems properties in a file and builds the associated Schmid tensors.

  This file is pointing on the unit normal vector to the habit / slip plane and the unit vector in the displacement / slip direction for the first system.\n
  \n
  Then the constructor builds the Schmid tensor for each system.
  \param file opened filename
*/

//--------------------------------------------------------------
 template <class T> systems<T>::systems (ifstream& file, int nb)
//--------------------------------------------------------------
{
  string text;

  nbsystems = nb;
  Rij = new Schmid_tensor<T> [nbsystems];
  for (int i=0; i<nbsystems; i++)
    Rij[i] = Schmid_tensor<T>(file);  // construction of Schmid tensor for each martensitic variants
}


/*!
  \brief Builds the Schmid tensor for each system of a given material.

  \param material_type material type : values are "CuZnAl" and "CuAlBe"
  \param nsystems number of systems
  \param path filename
*/

//------------------------------------------------------------------------------
template <class T> systems<T>::systems (const string& material_type, T nsystems)
//------------------------------------------------------------------------------
{
  string text;

  nbsystems = nsystems;

  vector<T> n(3,false), m(3,false);
  if (material_type == "CuZnAl")
    { n[1]=0.182; n[2]=0.669; n[3]=0.721;
      m[1]=0.165; m[2]=0.655; m[3]=0.737;
      Rij = new Schmid_tensor<T> [nbsystems];
      matrix<T> N, M;
      N = build_N24(n);
      M = build_M24(m);
      for (int i=1; i<=nbsystems; i++)
      Rij[i-1] = Schmid_tensor<T>(N[i],M[i]);  // construction of martensitic variants
    }
  else if (material_type == "CuAlBe")
    { n[1]=0.168; n[2]=0.688; n[3]=0.705;
      m[1]=0.121; m[2]=0.678; m[3]=0.725;
      Rij = new Schmid_tensor<T> [nbsystems];
      matrix<T> N, M;
      N = build_N24(n);
      M = build_M24(m);
      for (int i=1; i<=nbsystems; i++)
      Rij[i-1] = Schmid_tensor<T>(N[i],M[i]);  // construction of martensitic variants
    }
  else if (material_type == "FCC")
    { n[1]=1; n[2]=1; n[3]=1;
      m[1]=1; m[2]=-1; m[3]=0;
      Rij = new Schmid_tensor<T> [nbsystems];
      matrix<T> N, M;
      N = build_N12(n);
      M = build_M12(m);
      for (int i=1; i<=nbsystems; i++)
	Rij[i-1] = Schmid_tensor<T>(N[i],M[i]);  // construction of martensitic variants
    }
  else
    { cout << "No data for this material \n";
      exit(0);
    }
}


//----------------------------------------------------------
template <class T> systems<T>::systems (const systems& syst)
//----------------------------------------------------------
{
  Rij = new Schmid_tensor<T> [nbsystems=syst.nbsystems];
  for (int i=0; i<nbsystems; i++)
    Rij[i] = syst.Rij[i];
}


//----------------------------------------
template <class T> systems<T>::~systems ()
//----------------------------------------
{
  if (nbsystems) delete [] Rij;
}


//-------------------------------------------------------------------------
template <class T> systems<T>& systems<T>::operator = (const systems& syst)
//-------------------------------------------------------------------------
{
  assert (syst.nbsystems);

  if (nbsystems) delete [] Rij;

  Rij = new Schmid_tensor<T> [nbsystems=syst.nbsystems];
  for (int i=0; i<nbsystems; i++)
    Rij[i] = syst.Rij[i];

  return *this;
}


/*!
  \brief Returns the normal unit vector to the slip plane associated to the \f$ i^\text{th} \f$ system.
  \param i system number
  \return normal unit vector to the slip plane for the \f$ i^\text{th} \f$ system
*/

//------------------------------------------------
template <class T> vector<T> systems<T>::n (int i)
//------------------------------------------------
{
  assert ((i>0) && (i<=nbsystems));
  return Rij[i-1].n();
}


/*!
  \brief Returns the unit vector in the slip direction associated to the \f$ i^\text{th} \f$ system.
  \param i system number
  \return unit vector in the slip direction for the \f$ i^\text{th} \f$ system
*/

//------------------------------------------------
template <class T> vector<T> systems<T>::m (int i)
//------------------------------------------------
{
  assert ((i>0) && (i<=nbsystems));
  return Rij[i-1].m();
}


/*!
  \brief Returns the \ref Schmid_tensor "Schmid tensor" associated to the \f$ i^\text{th} \f$ system.
  \param i system number
  \return the Schmid tensor for the \f$ i^\text{th} \f$ system
*/

//--------------------------------------------------------------
template <class T> Schmid_tensor<T>& systems<T>::R (int i) const
//--------------------------------------------------------------
{
  assert ((i>0) && (i<=nbsystems));
  return Rij[i-1];
}


/*!
  \brief Compare two systems.
  \param syst1 first system
  \param syst2 second system
  \return true if syst1 and syst2 have the same characteristics, false elsewhere
*/

//---------------------------------------------------------------------------------------
template <class Tf> bool operator == (const systems<Tf>& syst1, const systems<Tf>& syst2)
//---------------------------------------------------------------------------------------
{
  bool test = true;

  test *= (syst1.nbsystems == syst2.nbsystems);
  for (int i=1; i<=syst1.nbsystems && test; i++)
    test *= (syst1.Rij[i-1] == syst2.Rij[i-1]);

  return test;
}


/*!
  \brief Compare two systems.
  \param syst1 first system
  \param syst2 second system
  \return true if syst1 and syst2 don't have the same characteristics, false elsewhere
*/

//---------------------------------------------------------------------------------------
template <class Tf> bool operator != (const systems<Tf>& syst1, const systems<Tf>& syst2)
//---------------------------------------------------------------------------------------
{
  return !(syst1 == syst2);
}


//----------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, const systems<Tf>& syst)
//----------------------------------------------------------------------------
{
  s << "Schmid tensor of slip systems\n";
  s << "-----------------------------\n";
  for (int i=0; i<=syst.nbsystems; i++)
    s << i << " Schmid tensor: \n" << syst.Rij[i];
  s << endl;
  return s;
}


}


#endif
