/***************************************************************
 *                   Material Object Library                   *
 *     Class Eshelby : declaration for the Eshelby tensor      *
 *                    simula.plus@cemes.fr                     *
 *		     GNU/linux version 3.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2005,2006,2007,2008,2012,2013,2014 COLLARD Christophe
 * copyright © 2012,2013 BERBENNI Stephane
 * copyright © 2005,2006,2007,2008,2012,2013,2014 Centre National de la Recherche Scientifique
 * copyright © 2005,2006,2007,2008 Arts et Métiers ParisTech
 * copyright © 2005,2006,2007 Université de Valenciennes et du Hainaut-Cambrésis
 * copyright © 2005,2006,2007,2008 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012,2013 Laboratoire d'Etude des Microstructures et de Mécanique des Matériaux (LEM3 - CNRS)
 * copyright © 2012,2013,2014 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::Eshelby
    \brief Numerical computation of Eshelby tensor for isotropic spheroidal inclusions

    \htmlonly 
    <FONT color="#838383">

    Eshelby belongs to Material Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    We note \f$ C \f$ the elasticity tensor (see \ref isotropic_tensor), \f$ T^{II} \f$ the interaction tensor (see \ref TII) and V the inclusion volume. Then, Eshelby tensor reads \n
\f$ \displaystyle S^{II}_{ijkl} = T^{II}_{ijpq} C_{pqkl} \f$.

    \author copyright \htmlonly &#169; \endhtmlonly 2005, 2006, 2007, 2008, 2012, 2013, 2014 Christophe COLLARD \n
            copyright \htmlonly &#169; 2012, 2013 St&#233;phane BERBENNI \endhtmlonly \n
            copyright \htmlonly &#169; 2005, 2006, 2007, 2008, 2012, 2013, 2014 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006, 2007, 2008 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006, 2007 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
            copyright \htmlonly &#169; 2005, 2006, 2007, 2008 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006, 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	    copyright \htmlonly &#169; 2012, 2013 Laboratoire d'Etude des Microstructures et de MM&#233;canique des MatM&#233;riaux (LEM3 - CNRS) \endhtmlonly \n
            copyright \htmlonly &#169; 2012, 2013, 2014 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.4.0
    \date 2005-2019
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type Eshelby
#endif

#if !defined __Eshelby_hpp
#define __Eshelby_hpp


#ifndef __maths_hpp
#include "MOL++/maths.hpp"
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __symtensors2_hpp
#include "MOL++/symtensors2.hpp"
#endif

#ifndef __tensors3_hpp
#include "MOL++/tensors3.hpp"
#endif

#ifndef __tensors4_hpp
#include "MOL++/tensors4.hpp"
#endif

#ifndef __isotropic_elasticity_tensors_hpp
#include "MateriOL++/isotropic elasticity tensors.hpp"
#endif

#ifndef __TII_hpp
#include "MateriOL++/TII.hpp"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//===========
class Eshelby
//===========
{
  private:

  public:
    Eshelby () {}   // default constructor
    ~Eshelby () {}  // destructor

    template <class T, template<class> class tensor_type> tensor4<T> ellipsoid (const tensor_type<T>&, const vector<T>&, int, int);
    template <class T> symtensor4<T> sphere (const isotropic_elasticity_tensor<T>&); // interior points Eshelby's tensor
    template <class T> symtensor4<T> sphere (const isotropic_elasticity_tensor<T>&, const vector<T>&, const vector<T>&); // exterior points Eshelby's tensor
    template <class T> tensor4<T> penny_shape (const isotropic_elasticity_tensor<T>&, const vector<T>&);
    template <class T> symtensor4<T> spheroid (const isotropic_elasticity_tensor<T>&, const vector<T>&, const vector<T>&); // Mura's formula for exterior points Eshelby's tensor
    template <class T> symtensor4<T> spheroid (const isotropic_elasticity_tensor<T>&, const vector<T>&); // Mura's formula for interior points Eshelby's tensor
    // computes interaction TII = T_2323 = T_2332 for spheroid oblates
    template <class T> T oblate_spheroid_2332 (const isotropic_elasticity_tensor<T>&, const vector<T>&);
    template <class T> symtensor4<T> spheroid_exterior_points (const isotropic_elasticity_tensor<T>&, const vector<T>&, const vector<T>&);  // Ju & Sun formula
    template <class T> symtensor4<T> spheroid_interior_points (const isotropic_elasticity_tensor<T>&, const vector<T>&);  // Ju & Sun formula
};


//=====Private methods for Eshelby==================================================


//=====Public methods for Eshelby===================================================


/*!
  \brief Computes Eshelby tensor in a general case

  First we compute \f$ T^{II} \f$. Then we compute Eshelby tensor. \n
  (see detailed description and \ref TII)

  \param L elasticity tensor
  \param geometry inclusion geometry
  \param nItg1 number of points for the integration on \f$ \theta \f$
  \param nItg2 number of points for the integration on \f$ \varphi \f$
  \return Eshelby tensor
*/

//------------------------------------------------------------------------------------------------------
template <class T, template<class> class tensor_type>
tensor4<T> Eshelby::ellipsoid (const tensor_type<T>& L, const vector<T>& geometry, int nItg1, int nItg2)
//------------------------------------------------------------------------------------------------------
{
  assert (geometry.dim() == 3);

  TII<T> tii(nItg1,nItg2);

  return (tii.ellipsoid(L,geometry) || L);
}


/*!
  \brief Computes Eshelby tensor for sphere

  \f$ \begin{aligned}
  & S^{II}_{1111} = S^{II}_{2222} = S^{II}_{3333} = \frac{7 - 5\nu}{15(1-\nu)}, \\
  & S^{II}_{1122} = S^{II}_{2233} = S^{II}_{3311} = S^{II}_{1133} = S^{II}_{2211} = S^{II}_{3322} = \frac{5\nu - 1}{15(1-\nu)}, \\
  & S^{II}_{1212} = S^{II}_{1221} = S^{II}_{2112} = S^{II}_{2121} = S^{II}_{2323} = S^{II}_{2332} = S^{II}_{3223} = S^{II}_{3232} = S^{II}_{1313} = S^{II}_{1331} = S^{II}_{3113} = S^{II}_{3131} = \frac{4 - 5\nu}{15(1-\nu)}.
  \end{aligned} \f$ \n
  (see Toshio Mura, Micromechanics of Defects in Solid, Kluwer Academic Publishers, ISBN 90-247-3256-5)

  \param C elasticity tensor
  \return Eshelby tensor
*/

//----------------------------------------------------------------------------------------
template <class T> symtensor4<T> Eshelby::sphere (const isotropic_elasticity_tensor<T>& C)
//----------------------------------------------------------------------------------------
{
  T nu = C.Poisson();
  T coef = 1/(15 * (1-nu));

  symtensor4<T> SII(3);
  SII(1,1,1,1) = SII(2,2,2,2) = SII(3,3,3,3) = (7 - 5*nu) * coef;
  SII(1,1,2,2) = SII(2,2,3,3) = SII(3,3,1,1) = SII(1,1,3,3) = SII(2,2,1,1) = SII(3,3,2,2) = (5*nu - 1) * coef;
  SII(1,2,1,2) = SII(1,2,2,1) = SII(2,1,1,2) = SII(2,1,2,1) = SII(2,3,2,3) = SII(2,3,3,2) = SII(3,2,2,3) = SII(3,2,3,2) = SII(1,3,1,3) = SII(1,3,3,1) = SII(3,1,1,3) = SII(3,1,3,1) = (4 - 5*nu) * coef;

  return SII;
}


//---------------------------------------------------------------------------------------------------------------------------------------
template <class T> symtensor4<T> Eshelby::sphere (const isotropic_elasticity_tensor<T>& C, const vector<T>& geometry, const vector<T>& x)
//---------------------------------------------------------------------------------------------------------------------------------------
{
  assert ((geometry[1] == geometry[2]) && (geometry[2] == geometry[3]));
  T nu = C.Poisson ();

  vector<T> a_2 (3);
  a_2[1] = pow (geometry[1], 2);
  a_2[2] = a_2[3] = pow (geometry[2], 2);

  T r = sqrt (pow (x[1], 2) + pow (x[2], 2) + pow (x[3], 2));
  assert (r / geometry[1] > 1); // checks that x is outside the ellipse centered in (0,0,0)

  T delta_x1 = x[1]; // prolate spheroid centered in (0,0,0)

  vector<T> rho (3, true, geometry[1] / r);

  T rho_3 = pow (rho[1], 3);
  T rho2 = pow (rho_3, 2./3.);

  vector<T> theta_(3);
  for (int i=1; i<=3; i++)
    theta_[i] = x[i] / pow (r, 2);
  T theta = theta_ * theta_;

  vector<T> n = theta_;
  n /= sqrt (theta);

  symtensor4<T> G(3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  G(i,j,k,l) = (3 * rho2 + 10 * nu - 5) * (i==j) * (k==l) + (3 * rho2 - 10 * nu + 5) * ((i==k) * (j==l) + (i==l) * (j==k)) + 15 * (1 - rho2) * (i==j) * n[k] * n[l] + 15 * (1 - 2 * nu - rho2) * (k==l) * n[i] * n[j] + 15 * (nu - rho2) * ((i==k) * n[j] * n[l] + (i==l) * n[j] * n[k] + (j==k) * n[i] * n[l] + (j==l) * n[i] * n[k]) + 15 * (7 * rho2 - 5) * n[i] * n[j] * n[k] * n[l];
  G *= rho_3 / (30 * (1 - nu));

  return G;
}


/*!
  \brief Computes Eshelby tensor for penny shape

  \f$ \begin{aligned}
  & S^{II}_{1111} = S^{II}_{2222} = \frac{13 - 8\nu}{32 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{3333} = 1 - \frac{1 - 2\nu}{1-\nu} \frac{\pi}{4} \frac{c}{a}, \\
  & S^{II}_{1122} = S^{II}_{2211} = \frac{8\nu - 1}{32 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{1133} = S^{II}_{2233} = \frac{2\nu -1}{8 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{3311} = S^{II}_{3322} = \frac{\nu}{1-\nu} \left( 1 - \frac{4\nu + 1}{8\nu} \pi \frac{c}{a} \right), \\
  & S^{II}_{1212} = S^{II}_{1221} = S^{II}_{2112} = S^{II}_{2121} = \frac{7 - 8\nu}{32 (1-\nu)} \pi \frac{c}{a}, \\
  & S^{II}_{1313} = S^{II}_{1331} = S^{II}_{3113} = S^{II}_{3131} = S^{II}_{2323} = S^{II}_{2332} = S^{II}_{3223} = S^{II}_{3232} = \frac{1}{2} \left( 1 + \frac{\nu - 2}{1 - \nu} \frac{\pi}{4} \frac{c}{a} \right).
  \end{aligned} \f$ \n
  (see Toshio Mura, Micromechanics of Defects in Solid, Kluwer Academic Publishers, ISBN 90-247-3256-5)

  \param C elasticity tensor
  \param geometry penny shape length \f$ (a,a,c) \f$
  \return Eshelby tensor
*/

//---------------------------------------------------------------------------------------------------------------------
template <class T> tensor4<T> Eshelby::penny_shape (const isotropic_elasticity_tensor<T>& C, const vector<T>& geometry)
//---------------------------------------------------------------------------------------------------------------------
{
  assert (geometry.dim() == 3);
  assert (abs(geometry[1] - geometry[2]) < epsilon);
  assert (geometry[1] > geometry[3]);

  T nu = C.Poisson();
  T invnu = 1/(1-nu);
  T PIa3_a1 = pi() * geometry[3] / geometry[1];
  //  T value = invnu * PIa3_a1 / (T) 32;
  T value = invnu * PIa3_a1 / 32.;

  tensor4<T> SII(3);
  SII(1,1,1,1) = SII(2,2,2,2) = (13 - 8*nu) * value;
  SII(3,3,3,3) = 1 - (8 - 16*nu) * value;
  SII(1,1,2,2) = SII(2,2,1,1) = (8*nu - 1) * value;
  SII(1,1,3,3) = SII(2,2,3,3) = (8*nu - 4) * value;
  SII(3,3,1,1) = SII(3,3,2,2) = nu * invnu - (16*nu + 4) * value;
  SII(1,2,1,2) = SII(1,2,2,1) = SII(2,1,1,2) = SII(2,1,2,1) = (7 - 8*nu) * value;
  SII(1,3,1,3) = SII(1,3,3,1) = SII(3,1,1,3) = SII(3,1,3,1) = SII(2,3,2,3) = SII(2,3,3,2) = SII(3,2,2,3) = SII(3,2,3,2) = .5 + (4*nu - 8) * value;

  return SII;
}


//-----------------------------------------------------------------------------------------------------------------------------------------
template <class T> symtensor4<T> Eshelby::spheroid (const isotropic_elasticity_tensor<T>& C, const vector<T>& geometry, const vector<T>& x)
//-----------------------------------------------------------------------------------------------------------------------------------------
{
  assert (pow (x[1] / geometry[1], 2) + pow (x[2] / geometry[2], 2) + pow (x[3] / geometry[3], 2) >= 1); // checks that x is outside the ellipse centered in (0,0,0)
  T nu = C.Poisson();

  vector<T> a_2 (3);
  for (int i=1; i<=3; i++)
    a_2[i] = pow (geometry[i], 2);

  //  assert (a_2[1] > a_2[3]); // prolate spheroid - compute I_2_2pi for oblate spheroids.

  T r_2 = pow (x[1], 2) + pow (x[2], 2) + pow (x[3], 2);

  T delta_x1 = x[1]; // prolate spheroid centered in (0,0,0)

  T lambda = r_2 - a_2[1] - a_2[3] + sqrt (pow (r_2 + a_2[1] - a_2[3], 2) - 4 * (a_2[1] - a_2[3]) * pow (delta_x1, 2));
  lambda *= 0.5;
//  lambda = 0;  // Eshelby

  // I-integrals

  T b = sqrt ((a_2[1] + lambda) / (a_2[3] + lambda));
  vector<T> Ii_2pi (3);

  // prolate spheroid
  if (geometry[1] > geometry[3])
    { T d = sqrt ((a_2[1] - a_2[3]) / (a_2[3] + lambda));
      Ii_2pi [2] = Ii_2pi [3] = geometry[1] * a_2[3] * (b*d - acosh (b)) / pow (a_2[1] - a_2[3], 1.5);
    }

  // oblate spheroid
  else
    { T d = sqrt ((a_2[3] - a_2[1]) / (a_2[3] + lambda));
      Ii_2pi [2] = Ii_2pi [3] = geometry[1] * a_2[3] * (acos(b) - b*d) / pow (a_2[3] - a_2[1], 1.5);
    }

  T Delta = sqrt (a_2[1] + lambda) * (a_2[3] + lambda);
  Ii_2pi [1] = 2 * geometry[1] * a_2[3] / Delta - 2 * Ii_2pi[3];

  symmatrix<T> Iij_2pi (3);
  Iij_2pi (1,2) = Iij_2pi (1,3) = (Ii_2pi[3] - Ii_2pi[1]) / (a_2[1] - a_2[3]);
  Iij_2pi (1,1) = 2 * geometry[1] * a_2[3] / (3 * (a_2[1] + lambda) * Delta) - (T) 2 / (T) 3 * Iij_2pi (1,3);
  Iij_2pi (2,2) = Iij_2pi (3,3) =  Iij_2pi (2,3) = 0.5 * geometry[1] * a_2[3] / ((a_2[3] + lambda) * Delta) - 0.25 * Iij_2pi (1,3);

  vector<T> deltav(3);
  symtensor4<T> S (3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=i; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=k; l++)
	  S(i,j,k,l) = (i==j) * (k==l) * ((2 * nu - 1) * Ii_2pi[i] + a_2[k] * Iij_2pi(k,i)) +
	    ((i==k) * (j==l) + (j==k) * (i==l)) * (a_2[k] * Iij_2pi(k,l) - nu * Ii_2pi[l] + (1 - nu) * Ii_2pi[k]);
  S *= .25 / (1 - nu);

  // 1st order derivatives

  T theta = 0;
  for (int i=1; i<=3; i++)
    theta += pow (x[i] / (a_2[i] + lambda), 2);

  vector<T> dlambda_dx (3);
  for (int i=1; i<=3; i++)
    dlambda_dx[i] = 2 * x[i] / ((a_2[i] + lambda) * theta);

  // 2nd order derivatives

  T d2F_part = 0;
  for (int i=1; i<=3; i++)
    d2F_part += 2 * pow (x[i], 2) / pow (a_2[i] + lambda, 3);

  matrix<T> d2lambda_dx2 (3,3);
  for (int i=1; i<=3; i++)
    { d2lambda_dx2 (i,i) = 2 / ((a_2[i] + lambda) * theta); // do not use dlambda_dx[i] / x[i] since x[i] can be equal to 0;
      for (int j=1; j<=3; j++)
	d2lambda_dx2 (i,j) += (d2F_part / theta - 1 / (a_2[i] + lambda) - 1 / (a_2[j] + lambda)) * dlambda_dx[i] * dlambda_dx[j];
    }

  // I-integrals derivatives

  T dDelta_dlambda = 0.5 * (a_2[3] + lambda) / sqrt (a_2[1] + lambda) + sqrt (a_2[1] + lambda);
  T dDelta_Delta = (T) 0.5 / (a_2[1] + lambda) + (T) 1 / (a_2[2] + lambda);

  tensor2<T> Ii_p_2pi (3);
  tensor3<T> Ii_pq_2pi (3), Iij_p_2pi (3);
  tensor4<T> Iij_pq_2pi (3);
  tensor2<T> Phi_pq_2pi (3);
  for (int i=1; i<=3; i++)
    { Phi_pq_2pi (i,i) = - Ii_2pi[i];
      for (int j=1; j<=3; j++)
	{ Ii_p_2pi (i,j) = - geometry[1] * a_2[2] * dlambda_dx[j] / ((a_2[i] + lambda) * Delta);
	  Phi_pq_2pi (i,j) -= x[i] * Ii_p_2pi (i,j);
	  for (int k=1; k<=3; k++)
	    { Iij_p_2pi (i,k,j) = Ii_p_2pi (i,j) / (a_2[k] + lambda);
	      Ii_pq_2pi (i,j,k) = - Ii_p_2pi (i,j) * ((T) 1 / (a_2[i] + lambda) + dDelta_Delta) * dlambda_dx [k] - geometry[1] * a_2[2] / ((a_2[i] + lambda) * Delta) * d2lambda_dx2 (j,k); // OK
	    }
	}
    }
  for (int i=1; i<=3; i++)
    Iij_pq_2pi [i] = (Ii_pq_2pi - (Ii_p_2pi ^ dlambda_dx) / (a_2[i] + lambda)) / (a_2[i] + lambda); // OK

  tensor3<T> Phi_pqr_2pi (3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	Phi_pqr_2pi (i,j,k) = - (i==j) * Ii_p_2pi (i,k) - (i==k) * Ii_p_2pi (i,j) - x[i] * Ii_pq_2pi (i,j,k);

  symtensor4<T> Psi_pqrs_2pi (3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=i; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=k; l++)
	  Psi_pqrs_2pi (i,j,k,l) = (i==j) * Phi_pq_2pi (k,l) + (i==k) * Phi_pq_2pi (j,l) + (i==l) * Phi_pq_2pi (j,k) + x[i] * Phi_pqr_2pi (j,k,l)
	    + a_2[i] * ( (i==j) * (k==l) * Iij_2pi (i,k) + ((i==k) * (j==l) + (j==k) * (i==l)) * Iij_2pi (i,j)
			+ (i==j) * x[k] * Iij_p_2pi (i,k,l) + ((i==k) * x[j] + (j==k) * x[i]) * Iij_p_2pi (i,j,l) + ((i==l) * x[j] + (j==l) * x[i]) * Iij_p_2pi (i,j,k)
			+ x[i] * x[j] * Iij_pq_2pi (i,j,k,l));

  symtensor4<T> D (3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=i; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=k; l++)
	  D (i,j,k,l) = Psi_pqrs_2pi (k,l,i,j) - 2 * nu * (k==l) * Phi_pq_2pi (i,j) - (1 - nu) * (Phi_pq_2pi (k,j) * (i==l) + Phi_pq_2pi (k,i) * (j==l) + Phi_pq_2pi (l,j) * (i==k) + Phi_pq_2pi (l,i) * (j==k));
  D *= 0.25 / (1 - nu);

  return D;
}


/*!
  \brief Computes (interior points) Eshelby's tensor for a spheroid using Mura's formulae

  \f$ \begin{aligned}
  & S^{II}_{1111} = \frac{3}{8 \pi (1-\nu)} a_1^2 I_{11} + \frac{1 - 2 \nu}{8 \pi (1 - \nu)} I_1, \\
  & S^{II}_{2222} = S^{II}_{3333} = \frac{3}{8 \pi (1-\nu)} a_3^2 I_{22} + \frac{1 - 2 \nu}{8 \pi (1 - \nu)} I_2, \\
  & S^{II}_{1122} = S^{II}_{1133} = \frac{1}{8 \pi (1-\nu)} a_2^2 I_{12} - \frac{1 - 2 \nu}{8 \pi (1 - \nu)} I_1, \\
  & S^{II}_{2211} = S^{II}_{3311} = \frac{1}{8 \pi (1-\nu)} a_1^2 I_{12} - \frac{1 - 2 \nu}{8 \pi (1 - \nu)} I_2, \\
  & S^{II}_{2233} = S^{II}_{3322} = \frac{1}{8 \pi (1-\nu)} a_3^2 I_{23} - \frac{1 - 2 \nu}{8 \pi (1 - \nu)} I_2, \\
  & S^{II}_{1212} = S^{II}_{1221} = S^{II}_{2112} = S^{II}_{2121} = S^{II}_{1313} = S^{II}_{1331} = S^{II}_{3113} = S^{II}_{3131} = \frac{a_1^2 + a_2^2}{16 \pi (1 - \nu)} I_{12} + \frac{1 - 2 \nu}{16 \pi (1 - \nu)} (I_1 + I_2), \\
  & S^{II}_{2323} = S^{II}_{2332} = S^{II}_{3223} = S^{II}_{3232} = \frac{a_2^2}{8 \pi (1 - \nu)} I_{23} + \frac{1 - 2 \nu}{8 \pi (1 - \nu)} I_2,
  \end{aligned} \f$ \n
  where \f$ \nu \f$ is the Poisson's ratio and: \n
  \f$ \begin{aligned}
  & I_2 = I_3 = \frac{2 \pi a_1 a_3^2}{(a_1^2 - a_3^2)^{\frac{3}{2}}} \left\{ \frac{a_1}{a_3} \left( \frac{a_1^2}{a_3^2} - 1 \right)^\frac{1}{2} - \mathrm{argch} \frac{a_1}{a_3} \right\}, \text{ if } a_1 > a_2 = a_3 \text{ (prolate spheroid)}, \\
  & I_2 = I_3 = \frac{2 \pi a_1 a_3^2}{(a_1^2 - a_3^2)^{\frac{3}{2}}} \left\{ \arccos \frac{a_1}{a_3} - \frac{a_1}{a_3} \left( 1 - \frac{a_1^2}{a_3^2} \right) \right\}, \text{ if } a_1 < a_2 = a_3 \text{ (oblate spheroid)}, \\
  & I_1 = 4 \pi - 2 I_2, \\
  & I_{12} = I_{13} = \frac{I_2 - I_1}{a_1^2 - a_2^2}, \\
  & 3 I_{11} = \frac{4 \pi}{a_1^2} - 2 I_{12}, \\
  & I_{23} = I_{22} = I_{33} = \frac{\pi}{a_2^2} - \frac{I_2 - I_1}{4 (a_1^2 - a_2^2)}.
  \end{aligned} \f$ \n
  (See Toshio Mura, Micromechanics of Defects in Solid, Kluwer Academic Publishers, ISBN 90-247-3256-5)

  \param C elasticity tensor
  \param geometry penny shape length \f$ (a_1,a_2,a_3 = a_2) \f$
  \return (interior points) Eshelby's tensor for a spheroid
*/

//---------------------------------------------------------------------------------------------------------------------
template <class T> symtensor4<T> Eshelby::spheroid (const isotropic_elasticity_tensor<T>& C, const vector<T>& geometry)
//---------------------------------------------------------------------------------------------------------------------
{
  assert (geometry[1] != geometry[2]); // use the sphere subroutine if a1 = a2 = a3
  assert (geometry[2] == geometry[3]);
  T nu = C.Poisson();
  T alpha = geometry[1] / geometry[3];
  T alpha_2 = pow (alpha, 2);

  T a1_2 = pow (geometry[1], 2);
  T a3_2 = pow (geometry[2], 2);
  T I2_2pi;
  if (geometry[1] > geometry[2]) I2_2pi = alpha / pow (alpha_2 - 1, 1.5) * (alpha * sqrt (alpha_2 - 1) - acosh (alpha));
  else I2_2pi = alpha / pow (1 - alpha_2, 1.5) * (acos (alpha) - alpha * sqrt (1 - alpha_2));
  T I1_2pi =  2 - 2 * I2_2pi;
  T I12_2pi = (I2_2pi - I1_2pi) / (a1_2 - a3_2);
  T I11_2pi = (2 / a1_2 - 2 * I12_2pi) / (T) 3;
  T I23_2pi = 0.5 / a3_2 - 0.25 * (I2_2pi - I1_2pi) / (a1_2 - a3_2);
  T I22_2pi = I23_2pi;

  symtensor4<T> S (3,3);

  S(1,1,1,1) = 0.75 * a1_2 * I11_2pi + 0.25 * (1 - 2 * nu) * I1_2pi;
  S(2,2,2,2) = S(3,3,3,3) = 0.75 * a3_2 * I22_2pi + 0.25 * (1 - 2 * nu) * I2_2pi;

  S(1,1,2,2) = S(1,1,3,3) = 0.25 * a3_2 * I12_2pi - 0.25 * (1 - 2 * nu) * I1_2pi;
  S(2,2,1,1) = S(3,3,1,1) = 0.25 * a1_2 * I12_2pi - 0.25 * (1 - 2 * nu) * I2_2pi;

  S(2,2,3,3) = S(3,3,2,2) = 0.25 * a3_2 * I23_2pi - 0.25 * (1 - 2 * nu) * I2_2pi;


  S(1,2,1,2) = S(2,1,1,2) = S(1,3,1,3) = S(3,1,1,3) = 0.125 * (a1_2 + a3_2) * I12_2pi + 0.125 * (1 - 2 * nu) * (2 - I2_2pi);
  S(2,3,2,3) = S(3,2,2,3) = 0.25 * a3_2 * I23_2pi + 0.25 * (1 - 2 * nu) * I2_2pi;

  S *= 1 / (1 - nu);

  return S;
}


/*!
  \brief Computes Eshelby tensor 2323, 2332, 3223 and 3232 coordinates for oblate spheroid

  \f$ \displaystyle S^{II}_{2323} = S^{II}_{2332} = S^{II}_{3223} = S^{II}_{3232} = 
  \frac{1}{2 (1-\nu)(a^2 - c^2)} \left[ a^2 - \nu (a^2 - c^2) + \frac{1}{2} \left( \displaystyle \frac{a^2 c \arccos \left( \frac{c}{a} \right)}{\sqrt{a^2 - c^2}} - c^2 \right) \left( \nu - \frac{2a^2 + c^2}{a^2 - c^2} \right) \right]. \f$ \n
  If we note \f$ \zeta =  \displaystyle \frac{c}{a} \f$ then \n
  \f$ \displaystyle S^{II}_{2323} =
  \frac{1}{2 (1-\nu)(1 - \zeta^2)} \left[ 1 - \nu (1 - \zeta^2) + \frac{1}{2} \left( \displaystyle \frac{\zeta \arccos(\zeta)}{\sqrt{1 - \zeta^2}} - \zeta^2 \right) \left( \nu - \frac{2 + \zeta^2}{1 - \zeta^2} \right) \right]. \f$ \n
  
  \param C elasticity tensor
  \param geometry penny shape length \f$ (a,a,c) \f$
  \return Eshelby tensor coordiantes 2323 2332 3223 3232
*/

//---------------------------------------------------------------------------------------------------------------------
template <class T> T Eshelby::oblate_spheroid_2332 (const isotropic_elasticity_tensor<T>& C, const vector<T>& geometry)
//---------------------------------------------------------------------------------------------------------------------
{
  assert (geometry.dim() == 3);
  assert (abs(geometry[1] - geometry[2]) < epsilon);

  T nu = C.Poisson();
  T zeta = geometry[3] / geometry[1];
  assert (zeta < 1);
  T zeta2 = pow(zeta,2);
  T coef = 1/(1-zeta2);

  T Tii = .5 * (coef-nu)  +  0.25 * zeta * coef * (nu - (2+zeta2)*coef) * (acos(zeta) * sqrt(coef) - zeta);
  Tii /= (1-nu);

  return Tii;
}


/*!
  \brief Computes (exterior points) Eshelby's tensor for a spheroid using Ju & Sun formulae

  We consider and inclusion centered in \f$ x^{(1)} \f$ in an infinite matrix. \n
  The (exterior points) Eshelby's tensor reads: \n
  \f$ S^{II}_{ijkl} = \frac{1}{4 (1 - \nu)} \left[ S^{(1)}_{IK}(\lambda) \delta_{ij} \delta_{kl} + S^{(2)}_{IJ}(\lambda) \left( \delta_{ik} \delta_{jl} + \delta_{il} \delta_{jk} \right) + S^{(3)}_I(\lambda) \delta_{ij} \hat{n}_k \hat{n}_l + S^{(4)}_K(\lambda) \delta_{kl} \hat{n}_i \hat{n}_j + S^{(5)}_I(\lambda) \left( \delta_{ik} \hat{n}_j \hat{n}_l + \delta_{il} \hat{n}_j \hat{n}_k \right) + S^{(6)}_J(\lambda) \left( \delta_{jk} \hat{n}_i \hat{n}_l + \delta_{jl} \hat{n}_i \hat{n}_k \right) + S^{(7)}_{IJKL}(\lambda) \hat{n}_i \hat{n}_j \hat{n}_k \hat{n}_l \right] \f$ \n
  where \n
  \f$ \begin{aligned}
  & S^{(1)}_{11} (\lambda) = \left[ 4 \nu + \frac{2}{\alpha^2 - 1} \right] g(\lambda) - \frac{2}{3 (\alpha^2 - 1)} \rho^3_1(\lambda) + \left[ 4 \nu + \frac{2}{\alpha^2 -1} \rho_1(\lambda) \rho^2_2(\lambda) \right], \\
  & S^{(1)}_{12} (\lambda) = S^{(1)}_{13} (\lambda) = \left[ 4 \nu - \frac{2 \alpha^2 + 1}{\alpha^2 - 1} \right] g(\lambda) + \left[ 4 \nu - \frac{2 \alpha^2}{\alpha^2 - 1} \right] \rho_1(\lambda) \rho^2_2(\lambda), \\
  & S^{(1)}_{21} (\lambda) = S^{(1)}_{31} (\lambda) = \left[ - 2 \nu - \frac{1 + 2 \alpha^2}{\alpha^2 -1} \right] g(\lambda) - \frac{2 \alpha^2}{\alpha^2 - 1} \rho_1(\lambda) \rho^2_2(\lambda), \\
  & S^{(1)}_{22} (\lambda) = S^{(1)}_{23} (\lambda) = S^{(1)}_{32} (\lambda) = S^{(1)}_{33} (\lambda) = \left[ - 2 \nu + \frac{4 \alpha^2 - 1}{4 ( \alpha^2 - 1)} \right] g(\lambda) + \frac{\alpha^2}{2 ( \alpha^2 - 1 )} \frac{\rho^4_2(\lambda)}{\rho_1(\lambda)}, \\
  & S^{(2)}_{11} (\lambda) = \left[ - 4 \nu + \frac{4 \alpha^2 - 2}{\alpha^2 - 1} \right] g(\lambda) - \frac{2}{3 (\alpha^2 - 1)} \rho^3_1(\lambda) - \left[4 \nu - \frac{4 \alpha^2 - 2}{\alpha^2 - 1} \right] \rho_1(\lambda) \rho^2_2(\lambda), \\
  & S^{(2)}_{12} (\lambda) = S^{(2)}_{21} (\lambda) = S^{(2)}_{13} (\lambda) = S^{(2)}_{31} (\lambda) = \left[ - \nu - \frac{\alpha^2 + 2}{\alpha^2 - 1} \right] g(\lambda) - \left[ 2 \nu + \frac{2}{\alpha^2 - 1} \right] \rho_1(\lambda) \rho^2_2(\lambda), \\
  & S^{(2)}_{22} (\lambda) = S^{(2)}_{23} (\lambda) = S^{(2)}_{32} (\lambda) = S^{(2)}_{33} (\lambda) = \left[2 \nu - \frac{4 \alpha^2 - 7}{4 (\alpha^2 - 1)} \right] g(\lambda) + \frac{\alpha^2}{2(\alpha^2 - 1)} \frac{\rho^4_2(\lambda)}{\rho_1(\lambda)}, \\
  & S^{(3)}_I (\lambda) = 2 \rho^3(\lambda) \left[ 1 - \rho^2_I(\lambda) \right], \\
  & S^{(4)}_I (\lambda) = 2 \rho^3(\lambda) \left[ 1 - 2 \nu - \rho^2_I(\lambda) \right], \\
  & S^{(5)}_I (\lambda) = S^{(6)}_I (\lambda) = 2 \rho^3(\lambda) \left[ \nu - \rho^2_I(\lambda) \right], \\
  & S^{(7)}_{IJKL} (\lambda) = 2 \rho^3(\lambda) \left[ 2 \left( \rho^2_I(\lambda) + \rho^2_J(\lambda) + \rho^2_K(\lambda) + \rho^2_L(\lambda) \right) + \rho_m(\lambda) \rho_m(\lambda) - \frac{4 \rho^2_M(\lambda) \Theta_m(\lambda) \Theta_m(\lambda)}{\Theta(\lambda)} - 5 \right],
  \end{aligned} \f$ \n
  and where \f$ \nu \f$ is the Poisson's ratio, \f$ \delta_{ij} \f$ is the kronecker symbol, \f$ \alpha = \frac{a_1}{a_3} \f$ is the shape ratio of the spheroid, and: \n
  \f$ \begin{aligned}
  & g(\lambda) = - \frac{\alpha^2}{\alpha^2 - 1} \frac{\rho^2_2(\lambda)}{\rho_1(\lambda)} + \frac{\alpha}{(\alpha^2 - 1)^\frac{3}{2}} \ln \left[ (\alpha^2 - 1)^\frac{1}{2} \rho_2(\lambda) + \frac{\alpha \rho_2(\lambda)}{\rho_1(\lambda)} \right], \text{ if } a_1 > a_2 = a_3 \text{ (prolate spheroid)}, \\
  & g(\lambda) = - \frac{\alpha^2}{\alpha^2 - 1} \frac{\rho^2_2(\lambda)}{\rho_1(\lambda)} + \frac{\alpha}{(1 - \alpha^2)^\frac{3}{2}} \left[ \arctan \frac{\alpha}{(1 - \alpha^2)^\frac{1}{2} \rho_1(\lambda)} - \frac{\pi}{2} \right], \text{ if } a_1 < a_2 = a_3 \text{ (oblate spheroid)},
  \end{aligned} \f$ \n
  and \f$ \lambda \f$ is given by: \n
  \f$ \lambda = \frac{1}{2} \left[ r^2 - a^2_1 - a^2_2 + \sqrt{\left( r^2 + a^2_1 - a^2_2 \right)^2 - 4 \left( a^2_1 - a^2_2 \right) \left( x_1 - x_1^{(1)} \right)^2} \right], \f$ \n
  \f$ r^2 = \left( x_i - x_i^{(1)} \right)  \left( x_i - x_i^{(1)} \right) \f$, \n
  \f$ \rho_I (\lambda) = \frac{a_I}{\sqrt{a_I^2 + \lambda}} \f$, \n
  \f$ \rho^3 (\lambda) = \rho_1(\lambda) \rho_2(\lambda) \rho_3(\lambda) \f$, \n
  \f$ \Theta_i (\lambda) = \frac{x_i - x_i^{(1)}}{a_I^2 + \lambda} \f$, \n
  \f$ \Theta (\lambda) = \Theta_i(\lambda) \Theta_i(\lambda) \f$, \n
  \f$ \hat{n}_i = \frac{\Theta_i(\lambda)}{\sqrt{\Theta(\lambda)}} \f$. \n
  Note that \f$ I_2 = - 2 \pi g(\lambda) \f$, where \f$ I_2 \f$ is given by \ref Eshelby::spheroid(const isotropic_elasticity_tensor<T>&, const vector<T>&, const vector<T>&) "Mura's formulae" \n
  The summation convention on repeated indices does not apply on capital letters. \n
  (See J.W. Ju and L.Z. Sun , International Journal of Solids and Structures, 38 (2001), pp 183-201.

  \param C elasticity tensor
  \param geometry penny shape length \f$ (a_1, a_2, a_3 = a_2) \f$
  \param x position \f$ x \f$ where the Eshelby's tensor is calculated
  \return (interior points) Eshelby's tensor for a spheroid
*/

//---------------------------------------------------------------------------------------------------------------------------------------------------------
template <class T> symtensor4<T> Eshelby::spheroid_exterior_points (const isotropic_elasticity_tensor<T>& C, const vector<T>& geometry, const vector<T>& x)
//---------------------------------------------------------------------------------------------------------------------------------------------------------
{
  assert ((geometry[1] != geometry[2]) && (geometry[2] == geometry[3]));

  T nu = C.Poisson ();

  vector<T> a_2 (3);
  a_2[1] = pow (geometry[1], 2);
  a_2[2] = a_2[3] = pow (geometry[2], 2);
  T alpha_2 = a_2[1] / a_2[2];

  assert (pow (x[1] / geometry[1], 2) + pow (x[2] / geometry[2], 2) + pow (x[3] / geometry[3], 2) >= 1); // checks that x is outside the ellipse centered in (0,0,0)

  T r_2 = pow (x[1], 2) + pow (x[2], 2) + pow (x[3], 2);

  T delta_x1 = x[1]; // prolate spheroid centered in (0,0,0)

  T lambda = r_2 - a_2[1] - a_2[2] + sqrt (pow (r_2 + a_2[1] - a_2[2], 2) - 4 * (a_2[1] - a_2[2]) * pow (delta_x1, 2));
  lambda *= 0.5;
  //  lambda = 0;

  vector<T> rho(3);
  rho[1] = geometry[1] / sqrt (a_2[1] + lambda);
  rho[2] = rho[3] = geometry[2] / sqrt (a_2[2] + lambda);

  vector<T> rho_2(3);
  rho_2[1] = pow (rho[1], 2);
  rho_2[2] = rho_2[3] = pow (rho[2], 2);

  T rho_3 = rho[1] * rho_2[2];
  T inv_alpha2_1 = (T) 1 / (alpha_2 - 1);

  T g = 0;
  if (alpha_2 >= 1) g = - alpha_2 * inv_alpha2_1 * rho_2[2] / rho[1] + sqrt (alpha_2 * pow (inv_alpha2_1, 3)) * log (sqrt ((alpha_2 - 1) * rho_2[2]) + sqrt (alpha_2 * rho_2[2]) / rho[1]);
  else if (alpha_2 < 1) g = - alpha_2 * inv_alpha2_1 * rho_2[2] / rho[1] + sqrt (- alpha_2 * pow (inv_alpha2_1, 3)) * (atan (sqrt (- alpha_2 * inv_alpha2_1) / rho[1]) - 0.5 * pi());

  tensor2<T> S1(3);
  S1(1,1) = (4 * nu + 2 * inv_alpha2_1) * g - pow (rho[1], 3) * inv_alpha2_1 / (T) 1.5 + (4 * nu + 2 * inv_alpha2_1) * rho_3;
  S1(1,2) = S1(1,3) = (4 * nu - (2 * alpha_2 + 1) * inv_alpha2_1) * g + (4 * nu - 2 * alpha_2 * inv_alpha2_1) * rho_3;
  S1(2,1) = S1(3,1) = (- 2 * nu - (2 * alpha_2 + 1) * inv_alpha2_1) * g - 2 * alpha_2 * inv_alpha2_1 * rho_3;
  S1(2,2) = S1(2,3) = S1(3,2) = S1(3,3) = (- 2 * nu + (alpha_2 - 0.25) * inv_alpha2_1) * g + 0.5 * alpha_2 * inv_alpha2_1 * pow (rho_2[2], 2) / rho[1];

  symtensor2<T> S2(3);
  S2(1,1) = (- 4 * nu + (4 * alpha_2 - 2) * inv_alpha2_1) * g - 2 * inv_alpha2_1 * pow (rho[1], 3) / (T) 3 - (4 * nu - (4 * alpha_2 - 2) * inv_alpha2_1) * rho_3;
  S2(1,2) = S2(1,3) = (- nu - (alpha_2 + 2) * inv_alpha2_1) * g - (2 * nu + 2 * inv_alpha2_1) * rho_3;
  S2(2,2) = S2(2,3) = S2(3,3) = (2 * nu - (alpha_2 - (T) 7 / (T) 4) * inv_alpha2_1) * g + 0.5 * alpha_2 * inv_alpha2_1 * pow (rho_2[2], 2) / rho[1];

  vector<T> Id (3, true, 1);
  vector<T> S3 = 2 * rho_3 * (Id - rho_2);
  vector<T> S4 = 2 * rho_3 * ((1 - 2 * nu) * Id - rho_2);
  vector<T> S5 = 2 * rho_3 * (nu * Id - rho_2);

  vector<T> theta_(3);
  for (int i=1; i<=3; i++)
    theta_[i] = x[i] / (a_2[i] + lambda);

  T theta_2 = theta_ * theta_;
  T rho_theta_theta = rho_2[1] * pow (theta_[1], 2) + rho_2[2] * pow (theta_[2], 2) + rho_2[3] * pow (theta_[3], 2);
  T cst =  rho * rho - (T) 4 * rho_theta_theta / theta_2 - (T) 5;

  symtensor4<T> S7 (3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=i; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=k; l++)
	  S7 (i,j,k,l) = (T) 2 * rho_3 * ((T) 2 * (rho_2[i] + rho_2[j] + rho_2[k] + rho_2[l]) + cst);

  vector<T> n = theta_;
  n /= sqrt (theta_2);

  symtensor4<T> G(3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=i; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=k; l++)
	  G(i,j,k,l) = S1(i,k) * (i==j) * (k==l) + S2(i,j) * ((i==k) * (j==l) + (i==l) * (j==k)) + S3[i] * (i==j) * n[k] * n[l] + S4[k] * (k==l) * n[i] * n[j] + S5[i] * ((i==k) * n[j] * n[l] + (i==l) * n[j] * n[k]) + S5[j] * ((j==k) * n[i] * n[l] + (j==l) * n[i] * n[k]) + S7(i,j,k,l) * n[i] * n[j] * n[k] * n[l];
  G *=0.25 / (1 - nu);

  return G;
}


/*!
  \brief Computes (interior points) Eshelby's tensor for a spheroid using Ju & Sun formulae

  The (interior points) Eshelby's tensor reads:
  \f$ S^{II}_{ijkl} = \frac{1}{4 (1 - \nu)} \left[ S^{(1)}_{IK}(0) \delta_{ij} \delta_{kl} + S^{(2)}_{IJ}(0) \left( \delta_{ik} \delta_{jl} + \delta_{il} \delta_{jk} \right) \right] \f$ \n
  where \n
  \f$ \begin{aligned}
  & S^{(1)}_{11} (0) = \left[ 4 \nu + \frac{2}{\alpha^2 - 1} \right] g(0) + 4 \nu + \frac{4}{3 ( \alpha^2 -1 )}, \\
  & S^{(1)}_{12} (0) = S^{(1)}_{13} (0) = \left[ 4 \nu - \frac{2 \alpha^2 + 1}{\alpha^2 - 1} \right] g(0) + 4 \nu - \frac{2 \alpha^2}{\alpha^2 - 1}, \\
  & S^{(1)}_{21} (0) = S^{(1)}_{31} (0) = \left[ - 2 \nu - \frac{1 + 2 \alpha^2}{\alpha^2 -1} \right] g(0) - \frac{2 \alpha^2}{\alpha^2 - 1}, \\
  & S^{(1)}_{22} (0) = S^{(1)}_{23} (0) = S^{(1)}_{32} (0) = S^{(1)}_{33} (0) = \left[ - 2 \nu + \frac{4 \alpha^2 - 1}{4 ( \alpha^2 - 1)} \right] g(0) + \frac{\alpha^2}{2 ( \alpha^2 - 1 )}, \\
  & S^{(2)}_{11} (0) = \left[ - 4 \nu + \frac{4 \alpha^2 - 2}{\alpha^2 - 1} \right] g(0) - 4 \nu + \frac{12 \alpha^2 - 8}{3 (\alpha^2 - 1)}, \\
  & S^{(2)}_{12} (0) = S^{(2)}_{21} (0) = S^{(2)}_{13} (0) = S^{(2)}_{31} (0) = \left[ - \nu - \frac{\alpha^2 + 2}{\alpha^2 - 1} \right] g(0) - 2 \nu - \frac{2}{\alpha^2 - 1}, \\
  & S^{(2)}_{22} (0) = S^{(2)}_{23} (0) = S^{(2)}_{32} (0) = S^{(2)}_{33} (0) = \left[2 \nu - \frac{4 \alpha^2 - 7}{4 (\alpha^2 - 1)} \right] g(0) + \frac{\alpha^2}{2(\alpha^2 - 1)},
  \end{aligned} \f$ \n
  and where \f$ \nu \f$ is the Poisson's ratio, \f$ \alpha = \frac{a_1}{a_3} \f$ is the shape ratio of the spheroid, \f$ \delta_{ij} \f$ is the kronecker symbol, and: \n
  \f$ \begin{aligned}
  & g(0) = \frac{\alpha}{(\alpha^2 - 1)^\frac{3}{2}} \left[ \mathrm{argch} \alpha - \alpha (\alpha^2 - 1)^\frac{1}{2} \right], \text{ if } a_1 > a_2 = a_3 \text{ (prolate spheroid)}, \\
  & g(0) = \frac{\alpha}{(1 - \alpha^2)^\frac{3}{2}} \left[ \alpha (1 - \alpha^2)^\frac{1}{2} - \arccos \alpha\right], \text{ if } a_1 < a_2 = a_3 \text{ (oblate spheroid)}.
  \end{aligned} \f$ \n
  Note that \f$ I_2 = - 2 \pi g(0) \f$, where \f$ I_2 \f$ is given by \ref Eshelby::spheroid(const isotropic_elasticity_tensor<T>&, const vector<T>&) "Mura's formulae" \n
  The summation convention on repeated indices does not apply on capital letters. \n
  (See J.W. Ju and L.Z. Sun , International Journal of Solids and Structures, 38 (2001), pp 183-201.

  \param C elasticity tensor
  \param geometry penny shape length \f$ (a_1, a_2, a_3 = a_2) \f$
  \return (interior points) Eshelby's tensor for a spheroid
*/

//-------------------------------------------------------------------------------------------------------------------------------------
template <class T> symtensor4<T> Eshelby::spheroid_interior_points (const isotropic_elasticity_tensor<T>& C, const vector<T>& geometry)
//-------------------------------------------------------------------------------------------------------------------------------------
{
  assert ((geometry[1] != geometry[2]) && (geometry[2] == geometry[3]));

  T nu = C.Poisson ();

  vector<T> a_2 (3);
  a_2[1] = pow (geometry[1], 2);
  a_2[2] = a_2[3] = pow (geometry[2], 2);

  T alpha_2 = a_2[1] / a_2[2];
  T alpha = geometry[1] / geometry[2];
  T inv_alpha2_1 = (T) 1 / (alpha_2 - 1);

  T g = 0;
  if (alpha_2 > 1) g = - alpha_2 * inv_alpha2_1 + sqrt (alpha_2 * pow (inv_alpha2_1, 3)) * acosh (alpha);
  else if (alpha_2 < 1) g =  - alpha_2 * inv_alpha2_1 - sqrt (alpha_2 * pow (-inv_alpha2_1, 3)) * acos (alpha);

  tensor2<T> S1(3);
  S1(1,1) = (4 * nu + (T) 2 * inv_alpha2_1) * g + 4 * nu + 4 / (T) 3 * inv_alpha2_1;
  S1(1,2) = S1(1,3) = (4 * nu - (2 * alpha_2 + 1) * inv_alpha2_1) * g + 4 * nu - 2 * alpha_2 * inv_alpha2_1;
  S1(2,1) = S1(3,1) = (- 2 * nu - (2 * alpha_2 + 1) * inv_alpha2_1) * g - 2 * alpha_2 * inv_alpha2_1;
  S1(2,2) = S1(2,3) = S1(3,2) = S1(3,3) = (- 2 * nu + (alpha_2 - 0.25) * inv_alpha2_1) * g + 0.5 * alpha_2 * inv_alpha2_1;

  symtensor2<T> S2(3);
  S2(1,1) = (- 4 * nu + (4 * alpha_2 - 2) * inv_alpha2_1) * g - 4 * nu + (12 * alpha_2 - 8) * inv_alpha2_1 / (T) 3;
  S2(1,2) = S2(1,3) = (- nu - (alpha_2 + 2) * inv_alpha2_1) * g - 2 * nu - 2 * inv_alpha2_1;
  S2(2,2) = S2(2,3) = S2(3,3) = (2 * nu - (alpha_2 - 1.75) * inv_alpha2_1) * g + 0.5 * alpha_2 * inv_alpha2_1;

  // to be expressed as a matrix (see calculation of exterior points)
  symtensor4<T> S(3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  S(i,j,k,l) = S1(i,k) * (i==j) * (k==l) + S2(i,j) * ((i==k) * (j==l) + (i==l) * (j==k));
  S *= 0.25 / (1 - nu);

  return S;
}


}


#endif
