/***************************************************************
 *                   Material Object Library                   *
 *     Class PII : declaration for PII interaction tensor      *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2009,2012 COLLARD Christophe
 * copyright © 2008 BERBENNI Stephane
 * copyright © 2008,2009,2012 Centre National de la Recherche Scientifique
 * copyright © 2008,2009 Arts et Métiers ParisTech
 * copyright © 2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::PII
    \brief Numerical computation of the polarization tensor \f$P^{II} \f$ for spheroid inclusions

    \htmlonly 
    <FONT color="#838383">

    PII belongs to Material Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    Let us note \f$ \displaystyle T^{II}_{ijkl} \f$ the \f$ 4^\text{th} \f$ order \ref TII "interaction tensor". The polarization tensor \f$ \displaystyle P^{II}_{ijkl} \f$ can be defined through \f$ \displaystyle T^{II}_{ijkl} \f$ by \n
    \f$ \displaystyle P^{II}_{ijkl} = \frac{1}{2} \left( T^{II}_{ijkl} + T^{II}_{ijlk} \right) \f$. \n
    It verifies the following symmetries: \n
    \f$ \displaystyle P^{II}_{ijkl} = P^{II}_{jikl} = P^{II}_{ijlk} \f$. \n

    \author copyright \htmlonly &#169; \endhtmlonly 2008, 2009 Stephane BERBENNI \n
            copyright \htmlonly &#169; \endhtmlonly 2009, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; 2008, 2009, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2008, 2009 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
            copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.4.0
    \date 2008-2019
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type PII
#endif

#ifndef __PII_hpp
#define __PII_hpp


#ifndef __iostream
#include <iostream>
#endif

#ifndef __tensors4_hpp
#include "MOL++/tensors4.h"
#endif

#ifndef __TII_hpp
#include "TII.hpp"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//==========================================
template <class T> class PII : public TII<T>          // Hill tensor P also called Polarization tensor
//==========================================
{
  public :
    PII () {}        // default constructor
    PII (int, int);  // uses a Gauss integration on tetha and phi to compute the Green tensor
    ~PII () {}       // destructor

    template<template <class> class tensor_type> tensor4<T> ellipsoid (const tensor_type<T>&, const vector<T>&); // computes P for ellipsoidal inclusion
    tensor4<T> sphere (const isotropic_elasticity_tensor<T>&); // computes PII for isotropic spherical inclusion
};


//=====Private methods for PII=============================================


//=====Public methods for PII==============================================


/*!
  \brief Constructor for PII with the number of integration points

  see \ref TII::TII(int,int) "TII constructor"

  \param nItg1 number of points for the integration on \f$ \theta \f$
  \param nItg2 number of points for the integration on \f$ \varphi \f$
*/

//---------------------------------------------------------------------------
template <class T> PII<T>::PII (int nItg1, int nItg2) : TII<T> (nItg1, nItg2)
//---------------------------------------------------------------------------
{
}


/*!
  \brief Numerical computation of \f$ P^{II} \f$ 

  see computation of \ref TII::ellipsoid "TII for an ellipsoid"

  \param L elasticity tensor
  \param geometry spheroid geometry \f$ R = (a,b,c) \f$
  \return \f$ P^{II} \f$
*/

//-------------------------------------------------------------------------------
template <class T> template <template <class> class tensor_type>
tensor4<T> PII<T>::ellipsoid (const tensor_type<T>& L, const vector<T>& geometry)
//-------------------------------------------------------------------------------
{
  tensor4<T> Tii = TII<T>::ellipsoid (L, geometry);

  tensor4<T> Pii (3, false);
  for (int p=1; p<=3; p++)
    for (int q=1; q<=3; q++)
      Pii[p][q] = (T) 0.5 * (Tii[p][q] + t(Tii[p][q]));

  return Pii;
}


/*!
  \brief Numerical computation of \f$ P^{II} \f$ 

  see computation of \ref TII::sphere "TII for a sphere"

  \param C isotropic elasticity tensor
  \return \f$ P^{II} \f$
*/

//------------------------------------------------------------------------------------
template <class T> tensor4<T> PII<T>::sphere (const isotropic_elasticity_tensor<T>& C)
//------------------------------------------------------------------------------------
{
  tensor4<T> Tii = TII<T>::sphere (C);

  tensor4<T> Pii (3, false);
  for (int p=1; p<=3; p++)
    for (int q=1; q<=3; q++)
      Pii[p][q] = (T) 0.5 * (Tii[p][q] + t(Tii[p][q]));

  return Pii;
}


}


#endif
