/***************************************************************
 *                   Material Object Library                   *
 *  Class loading : declaration for different loading dstrain  *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2008,2010,2012 COLLARD Christophe
 * copyright © 2008 NICAISE Nicolas
 * copyright © 2008,2010,2012 Centre National de la Recherche Scientifique
 * copyright © 2008,2010 Arts et Métiers ParisTech
 * copyright © 2008,2010 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace materiol
    \brief Materials Object Libraries
*/

/*! \class materiol::loading
    \brief Numerical computation of strain increment for different kind of loadings

    \htmlonly 
    <FONT color="#838383">

    loading belongs to Material Object Libraries (MateriOL++) </br>
    MateriOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2008, 2010, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; \endhtmlonly 2008 Nicolas NICAISE \n
            copyright \htmlonly &#169; 2008, 2010, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2008, 2010 Arts et M&#233;tiers ParisTech \endhtmlonly \n
            copyright \htmlonly &#169; 2008, 2010 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
            copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.4.0
    \date 2008-2019
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type loading
#endif

#ifndef __loading_hpp
#define __loading_hpp


#ifndef __iostream
#include <iostream>
#endif

#ifndef __vectors_hpp
#include "MOL++/vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __tensors2_hpp
#include "MOL++/tensors2.hpp"
#endif

#ifndef __conversion_tools_hpp
#include "MOL++/conversion tools.hpp"
#endif

using namespace std;
using namespace mol;

namespace materiol
{


//==================================================================
template <class T, template <class> class tensor_type> class loading
//==================================================================
{
  private:
//  tensor2 <T> dstrain_macro;
//  tensor2 <T> dstress_macro;
    int stock_macro, stock_micro;
    double increment, end_loading;
    string end_criterium;

  public:
    loading() {}
    loading(ifstream&);
    ~loading() {}
    tensor2<T> UT (const tensor_type <T>&, T, int);
    tensor2<T> PS (const tensor_type <T>&, T, int);
};


//----------------------------------------------------
template <class T, template <class> class tensor_type>
loading<T, tensor_type>::loading (ifstream& file3)
//----------------------------------------------------
{ 
  string text, keyword, loading_type, end_crit;
}


/*!
  \brief Uniaxial tensile strain


  \param L elasticity tensor
  \param dE strain increment
  \param axis direction for tensile stress (possible values are 11 or 22 or 33)
  \return \f$ d\EuScript{E} \f$
*/

//-------------------------------------------------------------------------------
template <class T, template <class> class tensor_type>
tensor2<T> loading<T, tensor_type>::UT (const tensor_type <T>& L, T dE, int axis)
//-------------------------------------------------------------------------------
{ 
  assert (axis == 11 || axis == 22 || axis ==33);

  axis /= 11;
  tensor_type<T> Linv = L.inv();

  T dSigma = dE / Linv (axis, axis, axis, axis);
  symtensor2<T> dotE (3);
  for (int i=1; i<=3; i++)
    for (int j=i; j<=3; j++)
      dotE (i,j) = Linv (i, j, axis, axis) * dSigma;

  return dotE;
}


/*!
  \brief Uniaxial tensile strain


  \param L elasticity tensor
  \param dE strain increment
  \param axis direction for tensile stress (possible values are 11 or 22 or 33)
  \return \f$ d\EuScript{E} \f$
*/

//-------------------------------------------------------------------------------
template <class T, template <class> class tensor_type>
tensor2<T> loading<T, tensor_type>::PS (const tensor_type <T>& L, T dE, int axis)
//-------------------------------------------------------------------------------
{ assert (axis == 12 || axis == 13 || axis == 23);

  if (axis == 12) axis = 4;
  if (axis == 13) axis = 5;
  if (axis == 23) axis = 6;

  matrix<T> Linv = gauss (symtensor2matrix (L));
  T dSigma = dE / Linv (axis, axis);
  vector<T> dotE (6);
  for (int i=1; i<=6; i++)
    dotE[i] = sqrt(2) * Linv(i, axis) * dSigma;

  return vector2symtensor (dotE);
}

}


#endif
