/***************************************************************
 *                    simula.plus@cemes.fr                     *
 *	             GNU/linux version 3.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2005,2006,2007,2008,2009,2012,2013,2014 COLLARD Christophe
 * copyright © 2005,2006,2007,2008,2009,2012,2013,2014 Centre National de la Recherche Scientifique
 * copyright © 2005,2006,2007,2008,2009 Arts et Métiers ParisTech
 * copyright © 2005,2006,2007 Université de Valenciennes et du Hainaut-Cambrésis
 * copyright © 2005,2006,2007,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012,2013,2014 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*
    Eshelby-tests belongs to Material Object Libraries (MateriOL++)
    MateriOL++ is part of Simula+

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef __cplusplus
#error Must use C++ for the type Eshelby test
#endif

#ifndef __Eshelby_test_hpp
#define __Eshelby_test_hpp


#ifndef __iostream
#include <iostream>
#endif

#ifndef __stdio_h
#include <stdio.h>
#endif

#ifndef __stdlib_h
#include <stdlib.h>
#endif

#ifndef __colors_hpp
#include "colors.hpp"
#endif

#ifndef __parameters_h
#include "parameters.h"
#endif

#ifndef __matrix_hpp
#include "MOL++/matrix.hpp"
#endif

#ifndef __tensors2_hpp
#include "MOL++/tensors2.hpp"
#endif

#ifndef __tensors3_hpp
#include "MOL++/tensors3.hpp"
#endif

#ifndef __isotropic_elasticity_tensors_hpp
#include "MateriOL++/isotropic elasticity tensors.hpp"
#endif

#ifndef __TII_hpp
#include "MateriOL++/TII.hpp"
#endif

#ifndef __Eshelby_hpp
#include "MateriOL++/Eshelby.hpp"
#endif

#ifndef __affiche_hpp
#include "tests/affiche.hpp"
#endif

using namespace materiol;


//===========================
int test_Eshelby (int detail)
//===========================
{
  int result=1;
  bool test = true;

  if (detail < 0) 
    { cout << "============================================================== \n";
      cout << blue << "                     Eshelby test skipped" << reset;
      cout << "============================================================== \n";
      return result;
    }

  long double R = 5;
  long double V = 4/3. * pi() * power(R,3);
  long double nu = 0.3;
  long double mu = 80000.;
  long double lambda = 120000;

  long double c = 2.5E-1; // length of the oblate radius rotation axis

  isotropic_elasticity_tensor<long double> C;
  C.Lame(lambda);
  C.Poisson(nu);
  //  C.Shear(mu);

  tensor4<long double> T_analytique(3), Tiso_analytique(3);
  long double coef = V / (30 * mu * (1-nu));
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  T_analytique(i,j,k,l) = ( (9. - 10*nu) * (i==k) * (j==l) - ( (i==j)*(k==l) + (i==l)*(j==k)) ) * coef;

  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      Tiso_analytique[i][j] = (long double) .5 * (T_analytique[i][j] + T_analytique[j][i]);

  tensor4<long double> Siso_analytique;
  Siso_analytique = (Tiso_analytique || C) * (long double) (1/V);

  Eshelby SII;
  test = (Siso_analytique == SII.sphere(C));
  if (detail) affiche ("sphere", test);
  else result *= (test);

  vector<long double> geometry (3,true,R);
  test = (Siso_analytique == SII.ellipsoid (C, geometry, 14, 18));
  if (detail) affiche ("ellipsoid (sphere)", test);
  else result *= (test);

  geometry[3] = 5.e-2;
  tensor4<long double> Ti_pennyshape = SII.penny_shape(C,geometry);
  tensor4<long double> Ti = SII.ellipsoid (C, geometry, 500, 10);
  test = true;
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  test *= abs(Ti(i,j,k,l) - Ti_pennyshape(i,j,k,l)) < 5.e-4;
  geometry[3] = 5.e-5;
  Ti_pennyshape =& SII.penny_shape(C,geometry);
  Ti = SII.ellipsoid (C, geometry, 10, 10);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  test *= abs(Ti(i,j,k,l) - Ti_pennyshape(i,j,k,l)) < 2.e-5;
  if (detail) affiche ("penny shape / ellipsoid", test);
  else result *= (test);

  test = abs(Ti (2,3,2,3) - SII.oblate_spheroid_2332(C,geometry)) < 1.e-5;
  test *= abs(Ti_pennyshape (2,3,2,3) - SII.oblate_spheroid_2332(C,geometry)) < 2.e-10;
  if (detail) affiche ("oblate / penny shape (coord 2332)", test);
  else result *= (test);

  geometry[3] = R-5.e-4;
  test = abs(SII.ellipsoid(C,geometry,13,17)(2,3,3,2) -  SII.oblate_spheroid_2332(C,geometry)) < 4.e-10;
  test *= abs(Siso_analytique(2,3,3,2) -  SII.oblate_spheroid_2332(C,geometry)) < 4.e-6; // 6.e-7;
  if (detail) affiche ("oblate / sphere (coord 2332)", test);
  else result *= (test);

  geometry[3] = 2;
  test = abs(SII.ellipsoid(C,geometry,60,17)(2,3,3,2) -  SII.oblate_spheroid_2332(C,geometry)) < epsilon;
  if (detail) affiche ("oblate / ellipsoid (coord 2332)", test);
  else result *= (test);


  //-----------------------
  // prolate discretization
  //-----------------------

  // heterogeneity (spheroid)
  vector<long double> prolate_geo (3,0);
  prolate_geo[1] = 3;
  prolate_geo[2] = 1.2;
  prolate_geo[3] = 1.2;
  // position of a point on the virtual spheroid
  vector<long double> x (3);
  x[1] = 4;
  x[2] = 5;
  x[3] = 6;
  // center of the spheroid
  vector<long double> x1 (3);

  int npts = 100;
  long double r = 1;
  long double d_theta, d_phi;
  d_theta = 2 * pi() / npts;
  d_phi = pi() / npts;

  vector<long double> X(3);
  matrix<long double> sphere ((npts+1)*(npts+1), 3);
  matrix<long double> prolate ((npts+1)*(npts+1), 3);
  vector<long double> F ((npts+1)*(npts+1)), Gx ((npts+1)*(npts+1)), Hx ((npts+1)*(npts+1));
  long double Lb = 2;

  int pts = 0;
  for (long double nphi = -0.5 * pi(); nphi <= 0.5 * pi(); nphi += d_phi)
    for (long double ntheta = 0; ntheta <= 2*pi(); ntheta += d_theta)
      { X[1] = cos (ntheta) * cos (nphi);
	X[2] = sin (ntheta) * cos (nphi);
	X[3] = sin (nphi);
	//	assert (pts < npts*npts);
	sphere [++pts] = X;
	X[1] *= prolate_geo[1];
	X[2] *= prolate_geo[2];
	X[3] *= prolate_geo[3];
	prolate [pts] = X;
	F[pts] = 1 - pow(X[1]/prolate_geo[1], 2) - pow(X[2]/prolate_geo[2], 2) - pow(X[3]/prolate_geo[3], 2);
	Gx[pts] = 1 - pow(X[1], 2) / (pow(prolate_geo[1], 2) + Lb) - pow(X[2], 2) / (pow(prolate_geo[2], 2) + Lb)  - pow(X[3], 2) / (pow(prolate_geo[3], 2) + Lb);
	Hx[pts] = 1 - pow(X[1], 2) / pow(prolate_geo[1], 2) - pow(X[2], 2) / pow(prolate_geo[2], 2) - pow(X[3], 2) / pow(prolate_geo[3], 2);
      }
  Gx.approximation();
  Hx.approximation();

  sphere.save ("sphere.res");
  prolate.save ("prolate.res");
  system ("gnuplot 'data/courbe.gnuplot'");


  //----------------------------------------------------------------------------------
  // compute Eshelby's tensor for exterior points when a1 > a2 = a3 (prolate spheroid)
  //----------------------------------------------------------------------------------

  // calculate r
  long double r_2 = 0;
  for (int i=1; i<=3; i++)
    r_2 += pow (x[i] - x1[i], 2);

  long double a1_2 = pow (prolate_geo[1], 2);
  long double a2_2 = pow (prolate_geo[2], 2);
  long double a3_2 = pow (prolate_geo[3], 2);

  long double lambda_ = 0.5 * (r_2 - a1_2 - a2_2 + sqrt (pow (r_2 + a1_2 - a2_2, 2) - 4 * (a1_2 - a2_2) * pow (x[1] - x1[1], 2)));
  //  long double beta2_inv = 1 / pow (beta, 2);
  //  long double lambda_ = - prolate_geo[1] + sqrt (a1_2 * r_2 * beta2_inv + (1 - beta2_inv) * pow (x[1] - x1[1], 2));

  long double rho1 = prolate_geo[1] / sqrt (a1_2 + lambda_);
  long double rho2 = prolate_geo[2] / sqrt (a2_2 + lambda_);
  long double rho3 = prolate_geo[3] / sqrt (a3_2 + lambda_);

  long double alpha = prolate_geo[1] / prolate_geo[2];
  long double alpha_2 = pow (alpha, 2);
  long double g;
  if (alpha > 1) g = - alpha_2 / (alpha_2 - 1) * pow (rho2, 2) / rho1 + alpha / pow (alpha_2 - 1, 1.5) * log (sqrt (alpha_2 - 1) * rho2 + alpha * rho2 / rho1);
  else if (alpha <1) g = - alpha_2 / (alpha_2 - 1) * pow (rho2, 2) / rho1 + alpha / pow (1 - alpha_2, 1.5) * (atan (alpha / (sqrt (1 - alpha_2) * rho1)) - 0.5 * pi());

  tensor2<long double> S1(3);
  S1(1,1) = (4 * nu + 2. / (alpha_2 - 1)) * g - 2. / (3 * (alpha_2 - 1)) * pow (rho1, 3) + (4 * nu + 2. / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S1(1,2) = S1(1,3) = (4 * nu - (2 * alpha_2 + 1) / (alpha_2 - 1)) * g +  (4 * nu - 2 * alpha_2 / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S1(2,1) = S1(3,1) = (-2 * nu - (2 * alpha_2 + 1) / (alpha_2 - 1)) * g - 2 * alpha_2 / (alpha_2 -1) * rho1 * pow (rho2, 2);
  S1(2,2) = S1(2,3) = S1(3,2) = S1(3,3) = (-2 * nu + (alpha_2 - 0.25) / (alpha_2 - 1)) * g + 0.5 * alpha_2 / (alpha_2 - 1) * pow (rho2, 4) / rho1;

  tensor2<long double> S2(3);
  S2(1,1) = (-4 * nu + (4 * alpha_2 - 2) / (alpha_2 - 1)) * g - 2 * pow (rho1, 3) / (3 * (alpha_2 - 1)) - (4 * nu - (4 * alpha_2 - 2) / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S2(1,2) = S2(1,3) = S2(2,1) = S2(3,1) = (-nu - (alpha_2 + 2) / (alpha_2 - 1)) * g - (2*nu + 2. / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S2(2,2) = S2(2,3) = S2(3,2) = S2(3,3) = (2 * nu - (alpha_2 - 7*0.25) / (alpha_2 - 1)) * g + 0.5 * alpha_2 / (alpha_2 - 1) * pow (rho2, 4) / rho1;

  long double rho_3 = rho1 * rho2 * rho3;
  vector<long double> S3(3);
  S3[1] = 2 * rho_3 * (1 - pow (rho1, 2));
  S3[2] = 2 * rho_3 * (1 - pow (rho2, 2));
  S3[3] = 2 * rho_3 * (1 - pow (rho3, 2));

  vector<long double> S4(3);
  S4[1] = 2 * rho_3 * (1 - 2 * nu - pow (rho1, 2));
  S4[2] = 2 * rho_3 * (1 - 2 * nu - pow (rho2, 2));
  S4[3] = 2 * rho_3 * (1 - 2 * nu - pow (rho3, 2));

  vector<long double> S5(3);
  S5[1] = 2 * rho_3 * (nu - pow (rho1, 2));
  S5[2] = 2 * rho_3 * (nu - pow (rho2, 2));
  S5[3] = 2 * rho_3 * (nu - pow (rho3, 2));

  vector<long double> theta(3);
  theta[1] = (x[1] - x1[1]) / (pow (prolate_geo[1], 2) + lambda_);
  theta[2] = (x[2] - x1[2]) / (pow (prolate_geo[2], 2) + lambda_);
  theta[3] = (x[3] - x1[3]) / (pow (prolate_geo[3], 2) + lambda_);

  long double ntheta = 0;
  for (int i=1; i<=3; i++) ntheta += theta[i] * theta[i];

  vector<long double> rho(3);
  rho[1] = rho1;
  rho[2] = rho[3] = rho2;
  vector<long double> rho_2 (3);
  rho_2[1] = pow (rho1, 2);
  rho_2[2] = rho_2[3] = pow (rho2, 2);
  tensor4<long double> S7(3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  { long double rtt = pow (rho1, 2) * theta[1] * theta[1];
	    rtt += pow (rho2, 2) * theta[2] * theta[2];
	    rtt += pow (rho2, 2) * theta[3] * theta[3];
	    rtt *= 4. / ntheta;
	    S7(i,j,k,l) = 2 * rho_3 * (2 * (rho_2[i] + rho_2[j] + rho_2[k] + rho_2[l]) + rho * rho - rtt - 5);
	  }
  symtensor4<long double> symS7 (S7);

  vector<long double> n(3);
  for (int i=1; i<=3; i++)
    n[i] = (x[i] - x1[i]) / ((pow (prolate_geo[i], 2) + lambda_) * sqrt (ntheta));

  tensor4<long double> G(3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  G(i,j,k,l) = 0.25 / (1 - nu) * (S1(i,k) * (i==j) * (k==l) + S2(i,j) * ((i==k) * (j==l) + (i==l) * (j==k)) + S3[i] * (i==j) * n[k] * n[l] + S4[k] * (k==l) * n[i] * n[j] + S5[i] * ((i==k) * n[j] * n[l] + (i==l) * n[j] * n[k]) + S5[j] * ((j==k) * n[i] * n[l] + (j==l) * n[i] * n[k]) + S7(i,j,k,l) * n[i] * n[j] * n[k] * n[l]);

  test = (G == SII.spheroid_exterior_points (C, prolate_geo, x));
  test *= (SII.spheroid_exterior_points (C, prolate_geo, x) ==  SII.spheroid (C, prolate_geo, x));

  if (detail) affiche ("Eshelby exterior points - prolate spheroid (a1 > a2 = a3)", test);
  else result *= (test);

  // imaginary prolate spheroid -> real spheroid

  x[1] = 3;
  x[2] = x[3] = 0;

  theta[1] = (x[1] - x1[1]) / pow (prolate_geo[1], 2);
  theta[2] = (x[2] - x1[2]) / pow (prolate_geo[2], 2);
  theta[3] = (x[3] - x1[3]) / pow (prolate_geo[3], 2);

  ntheta = 0;
  for (int i=1; i<=3; i++) ntheta += theta[i] * theta[i];

  for (int i=1; i<=3; i++)
    n[i] = (x[i] - x1[i]) / (pow (prolate_geo[i], 2) * sqrt (ntheta));
  symtensor4<long double> jump (3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  jump (i,j,k,l) = - 4 * nu * (k==l) * n[i] * n[j] + 2 * (nu - 1) * ((i==k) * n[j] * n[l] + (i==l) * n[j] * n[k] + (j==k) * n[i] * n[l] + (j==l) * n[i] * n[k]) + 4 * n[i] * n[j] * n[k] * n[l];
  jump *= 0.25 / (1 - nu);

  test = (jump == SII.spheroid_exterior_points (C, prolate_geo, x) - SII.spheroid_interior_points (C, prolate_geo));
  if (detail) affiche ("Eshelby exterior points - imaginary prolate -> real prolate", test);
  else result *= (test);

  x[1] = 4;
  x[2] = 5;
  x[3] = 6;
  vector<long double> geo_sphere (3, true, 3);
  vector<long double> geo_sph = geo_sphere;
  geo_sph[1] += 1E-3;
  symtensor4<long double> Delta_Sph = SII.spheroid_exterior_points (C, geo_sph, x) - SII.sphere (C, geo_sphere, x);
  if (detail) affiche ("Eshelby exterior points - prolate spheroid -> sphere", Delta_Sph.p_norm (2) < 3E-5);
  else result *= (Delta_Sph.p_norm (2) < 3E-5);

  geo_sph[1] = 3 + 1E-4;
  x[1] = 3 + 1E-3;
  x[2] = x[3] = 0;
  Delta_Sph = SII.spheroid_exterior_points (C, geo_sph, x) - SII.sphere (C, geo_sphere, x);
  if (detail) affiche ("Eshelby exterior points - imaginary prolate -> real prolate -> sphere", Delta_Sph.p_norm (2) < 5E-4);
  else result *= (Delta_Sph.p_norm (2) < 5E-4);


  //---------------------------------------------
  // compute Eshelby's tensor for interior points
  //---------------------------------------------

  lambda_ = 0;

  rho1 = prolate_geo[1] / sqrt (a1_2 + lambda_);
  rho2 = prolate_geo[2] / sqrt (a2_2 + lambda_);
  rho3 = prolate_geo[3] / sqrt (a3_2 + lambda_);

  if (alpha > 1) g = - alpha_2 / (alpha_2 - 1) * pow (rho2, 2) / rho1 + alpha / pow (alpha_2 - 1, 1.5) * log (sqrt (alpha_2 - 1) * rho2 + alpha * rho2 / rho1);
  else if (alpha <1) g = - alpha_2 / (alpha_2 - 1) * pow (rho2, 2) / rho1 + alpha / pow (1 - alpha_2, 1.5) * (atan (alpha / (sqrt (1 - alpha_2) * rho1)) - 0.5 * pi());

  S1(1,1) = (4 * nu + 2. / (alpha_2 - 1)) * g - 2. / (3 * (alpha_2 - 1)) * pow (rho1, 3) + (4 * nu + 2. / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S1(1,2) = S1(1,3) = (4 * nu - (2 * alpha_2 + 1) / (alpha_2 - 1)) * g +  (4 * nu - 2 * alpha_2 / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S1(2,1) = S1(3,1) = (-2 * nu - (2 * alpha_2 + 1) / (alpha_2 - 1)) * g - 2 * alpha_2 / (alpha_2 -1) * rho1 * pow (rho2, 2);
  S1(2,2) = S1(2,3) = S1(3,2) = S1(3,3) = (-2 * nu + (alpha_2 - 0.25) / (alpha_2 - 1)) * g + 0.5 * alpha_2 / (alpha_2 - 1) * pow (rho2, 4) / rho1;

  S2(1,1) = (-4 * nu + (4 * alpha_2 - 2) / (alpha_2 - 1)) * g - 2 * pow (rho1, 3) / (3 * (alpha_2 - 1)) - (4 * nu - (4 * alpha_2 - 2) / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S2(1,2) = S2(1,3) = S2(2,1) = S2(3,1) = (-nu - (alpha_2 + 2) / (alpha_2 - 1)) * g - (2*nu + 2. / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S2(2,2) = S2(2,3) = S2(3,2) = S2(3,3) = (2 * nu - (alpha_2 - 7*0.25) / (alpha_2 - 1)) * g + 0.5 * alpha_2 / (alpha_2 - 1) * pow (rho2, 4) / rho1;

  symtensor4<long double> S(3);
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  S(i,j,k,l) = S1(i,k) * (i==j) * (k==l) + S2(i,j) * ((i==k) * (j==l) + (i==l) * (j==k));
  S *= 0.25 / (1 - nu);

  long double I_2 = 2 * pi() * prolate_geo[1] * pow (prolate_geo[3], 2) / sqrt (pow (pow (prolate_geo[1], 2) - pow (prolate_geo[3], 2), 3)) * (prolate_geo[1] / prolate_geo[3] * sqrt (pow (prolate_geo[1] / prolate_geo[3], 2) - 1) - acosh (prolate_geo[1] / prolate_geo[3]));
  long double I_1 = 4 * pi() - 2 * I_2;
  long double I_12 = (I_2 - I_1) / (pow (prolate_geo[1], 2) - pow (prolate_geo[2], 2));
  long double I_11 = 1 / 3. * (4 * pi() / pow (prolate_geo[1], 2) - 2 * I_12);
  long double I_23 = pi() / pow (prolate_geo[2], 2) - 0.25 * (I_2 - I_1) / (pow (prolate_geo[1], 2) - pow (prolate_geo[2],2));
  long double I_22, I_33;
  I_22 = I_33 = I_23;
  long double I_13 = 4 * pi() / pow(prolate_geo[1], 2) - I_12 - 3 * I_11;
  long double I_3 = 4 * pi() - I_1 - I_2;

  tensor4<long double> S_prolate (3,3,3,3);
  S_prolate(1,1,1,1) = 3 / (8 * pi() * (1 - nu)) * pow (prolate_geo[1], 2) * I_11 + (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_1;
  S_prolate(2,2,2,2) = S_prolate(3,3,3,3) = 3 / (8 * pi() * (1 - nu)) * pow (prolate_geo[2], 2) * I_22 + (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_2;
  S_prolate(1,1,2,2) = 1 / (8 * pi() * (1 - nu)) * pow (prolate_geo[2], 2) * I_12 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_1;
  S_prolate(2,2,1,1) = 1 / (8 * pi() * (1 - nu)) * pow (prolate_geo[1], 2) * I_12 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_2;
  S_prolate(2,2,3,3) = 1 / (8 * pi() * (1 - nu)) * pow (prolate_geo[3], 2) * I_23 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_2;
  S_prolate(3,3,2,2) = 1 / (8 * pi() * (1 - nu)) * pow (prolate_geo[2], 2) * I_23 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_3;
  S_prolate(1,1,3,3) = 1 / (8 * pi() * (1 - nu)) * pow (prolate_geo[3], 2) * I_13 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_1;
  S_prolate(3,3,1,1) = 1 / (8 * pi() * (1 - nu)) * pow (prolate_geo[1], 2) * I_13 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_3;
  S_prolate(1,2,1,2) = S_prolate(1,2,2,1) = S_prolate(2,1,1,2) = S_prolate(2,1,2,1) = (pow (prolate_geo[1], 2) + pow (prolate_geo[2], 2)) / (16 * pi() * (1 - nu)) * I_12 + (1 - 2 * nu) / (16 * pi() * (1 - nu)) * (I_1 + I_2);
  S_prolate(1,3,1,3) = S_prolate(1,3,3,1) = S_prolate(3,1,1,3) = S_prolate(3,1,3,1) = (pow (prolate_geo[1], 2) + pow (prolate_geo[3], 2)) / (16 * pi() * (1 - nu)) * I_13 + (1 - 2 * nu) / (16 * pi() * (1 - nu)) * (I_1 + I_3);
  S_prolate(2,3,2,3) = S_prolate(2,3,3,2) = S_prolate(3,2,2,3) = S_prolate(3,2,3,2) = (pow (prolate_geo[2], 2) + pow (prolate_geo[3], 2)) / (16 * pi() * (1 - nu)) * I_23 + (1 - 2 * nu) / (16 * pi() * (1 - nu)) * (I_2 + I_3);

  assert (S == S_prolate);

  if (detail) affiche ("Eshelby formula - prolate spheroid (a1 > a2 = a3)", S_prolate == SII.spheroid (C, prolate_geo));
  else result *= (S_prolate == SII.spheroid (C, prolate_geo));

  if (detail) affiche ("Eshelby interior points - prolate spheroid (a1 > a2 = a3)", S_prolate == SII.spheroid_interior_points (C, prolate_geo));
  else result *= (S_prolate == SII.spheroid_interior_points (C, prolate_geo));

  vector<long double> geo (3, true, 5);
  geo[1] += 0.001;
  symtensor4<long double> Delta_S = SII.spheroid_interior_points (C, geo) - SII.sphere(C);
  if (detail) affiche ("Eshelby interior points - prolate spheroid -> sphere", Delta_S.p_norm (2) < 1.E-4);
  else result *= (Delta_S.p_norm (2) < 1.E-4);


  prolate_geo[1] = 3;
  prolate_geo[2] = 1;
  prolate_geo[3] = 1;

  vector<long double> xi(3);
  xi[1] = 0;
  xi[2] = 1;
  xi[3] = 0;

  r_2 = xi * xi; // A MODIFIER
  vector<long double> a_2 (3);
  a_2[1] = pow (prolate_geo[1], 2);
  a_2[2] = a_2[3] = pow (prolate_geo[2], 2);
  long double beta = prolate_geo[2] / prolate_geo[1];
  long double dlambda = 0.5 * (r_2 - a_2[1] - a_2[2] + sqrt (pow (r_2 + a_2[1] - a_2[2], 2) + 4 * (beta - 1) * a_2[1] * (r_2 - a_2[2]) - 4 * (a_2[1] - a_2[2]) * pow (xi[1], 2)));
  dlambda /= beta;

  pts = 0;
  for (long double nphi = -0.5 * pi(); nphi <= 0.5 * pi(); nphi += d_phi)
    for (long double ntheta = 0; ntheta <= 2*pi(); ntheta += d_theta)
      { X[1] = cos (ntheta) * cos (nphi);
	X[2] = sin (ntheta) * cos (nphi);
	X[3] = sin (nphi);

	X[1] *= prolate_geo[1] + dlambda;
	X[2] *= prolate_geo[2] + beta * dlambda;
	X[3] *= prolate_geo[3] + beta * dlambda;
	prolate [++pts] = X;
      }
  prolate.save ("prolate2.res");
  system ("gnuplot 'data/courbe2.gnuplot'");


  //------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------


  //---------------------------------------------------------------------------------
  // compute Eshelby's tensor for exterior points when a1 < a2 = a3 (oblate spheroid)
  //---------------------------------------------------------------------------------

  // heterogeneity (spheroid)
  vector<long double> oblate_geo (3);
  oblate_geo[1] = 1.2;
  oblate_geo[2] = 3;
  oblate_geo[3] = 3;
  // position of a point on the virtual spheroid
  x[1] = 4;
  x[2] = 5;
  x[3] = 6;

  // calculate r
  r_2 = 0;
  for (int i=1; i<=3; i++)
    r_2 += pow (x[i] - x1[i], 2);

  a1_2 = pow (oblate_geo[1], 2);
  a2_2 = pow (oblate_geo[2], 2);
  a3_2 = pow (oblate_geo[3], 2);
  lambda_ = 0.5 * (r_2 - a1_2 - a2_2 + sqrt (pow (r_2 + a1_2 - a2_2, 2) - 4 * (a1_2 - a2_2) * pow (x[1] - x1[1], 2)));

  rho1 = oblate_geo[1] / sqrt (a1_2 + lambda_);
  rho2 = oblate_geo[2] / sqrt (a2_2 + lambda_);
  rho3 = oblate_geo[3] / sqrt (a3_2 + lambda_);

  alpha = oblate_geo[1] / oblate_geo[2];
  alpha_2 = pow (alpha, 2);
  if (alpha > 1) g = - alpha_2 / (alpha_2 - 1) * pow (rho2, 2) / rho1 + alpha / pow (alpha_2 - 1, 1.5) * log (sqrt (alpha_2 - 1) * rho2 + alpha * rho2 / rho1);
  else if (alpha <1) g = - alpha_2 / (alpha_2 - 1) * pow (rho2, 2) / rho1 + alpha / pow (1 - alpha_2, 1.5) * (atan (alpha / (sqrt (1 - alpha_2) * rho1)) - 0.5 * pi());

  S1(1,1) = (4 * nu + 2. / (alpha_2 - 1)) * g - 2. / (3 * (alpha_2 - 1)) * pow (rho1, 3) + (4 * nu + 2. / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S1(1,2) = S1(1,3) = (4 * nu - (2 * alpha_2 + 1) / (alpha_2 - 1)) * g +  (4 * nu - 2 * alpha_2 / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S1(2,1) = S1(3,1) = (-2 * nu - (2 * alpha_2 + 1) / (alpha_2 - 1)) * g - 2 * alpha_2 / (alpha_2 -1) * rho1 * pow (rho2, 2);
  S1(2,2) = S1(2,3) = S1(3,2) = S1(3,3) = (-2 * nu + (alpha_2 - 0.25) / (alpha_2 - 1)) * g + 0.5 * alpha_2 / (alpha_2 - 1) * pow (rho2, 4) / rho1;

  S2(1,1) = (-4 * nu + (4 * alpha_2 - 2) / (alpha_2 - 1)) * g - 2 * pow (rho1, 3) / (3 * (alpha_2 - 1)) - (4 * nu - (4 * alpha_2 - 2) / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S2(1,2) = S2(1,3) = S2(2,1) = S2(3,1) = (-nu - (alpha_2 + 2) / (alpha_2 - 1)) * g - (2*nu + 2. / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S2(2,2) = S2(2,3) = S2(3,2) = S2(3,3) = (2 * nu - (alpha_2 - 7*0.25) / (alpha_2 - 1)) * g + 0.5 * alpha_2 / (alpha_2 - 1) * pow (rho2, 4) / rho1;

  rho_3 = rho1 * rho2 * rho3;
  S3[1] = 2 * rho_3 * (1 - pow (rho1, 2));
  S3[2] = 2 * rho_3 * (1 - pow (rho2, 2));
  S3[3] = 2 * rho_3 * (1 - pow (rho3, 2));

  S4[1] = 2 * rho_3 * (1 - 2 * nu - pow (rho1, 2));
  S4[2] = 2 * rho_3 * (1 - 2 * nu - pow (rho2, 2));
  S4[3] = 2 * rho_3 * (1 - 2 * nu - pow (rho3, 2));

  S5[1] = 2 * rho_3 * (nu - pow (rho1, 2));
  S5[2] = 2 * rho_3 * (nu - pow (rho2, 2));
  S5[3] = 2 * rho_3 * (nu - pow (rho3, 2));

  theta[1] = (x[1] - x1[1]) / (a1_2 + lambda_);
  theta[2] = (x[2] - x1[2]) / (a2_2 + lambda_);
  theta[3] = (x[3] - x1[3]) / (a2_2 + lambda_);

  ntheta = 0;
  for (int i=1; i<=3; i++) ntheta += theta[i] * theta[i];

  rho[1] = rho1;
  rho[2] = rho[3] = rho2;
  rho_2[1] = pow (rho1, 2);
  rho_2[2] = rho_2[3] = pow (rho2, 2);

  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  { long double rtt = pow (rho1, 2) * theta[1] * theta[1];
	    rtt += pow (rho2, 2) * theta[2] * theta[2];
	    rtt += pow (rho2, 2) * theta[3] * theta[3];
	    rtt *= 4. / ntheta;
	    S7(i,j,k,l) = 2 * rho_3 * (2 * (rho_2[i] + rho_2[j] + rho_2[k] + rho_2[l]) + rho * rho - rtt - 5);
	  }

  symS7 = S7;

  for (int i=1; i<=3; i++)
    n[i] = (x[i] - x1[i]) / ((pow (oblate_geo[i], 2) + lambda_) * sqrt (ntheta));

  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  G(i,j,k,l) = 0.25 / (1 - nu) * (S1(i,k) * (i==j) * (k==l) + S2(i,j) * ((i==k) * (j==l) + (i==l) * (j==k)) + S3[i] * (i==j) * n[k] * n[l] + S4[k] * (k==l) * n[i] * n[j] + S5[i] * ((i==k) * n[j] * n[l] + (i==l) * n[j] * n[k]) + S5[j] * ((j==k) * n[i] * n[l] + (j==l) * n[i] * n[k]) + S7(i,j,k,l) * n[i] * n[j] * n[k] * n[l]);

  test = (G == SII.spheroid_exterior_points (C, oblate_geo, x));
  test *= (G == SII.spheroid (C, oblate_geo, x));
  if (detail) affiche ("Eshelby exterior points - oblate spheroid (a1 < a2 = a3)", test);
  else result *= (test);

  // imaginary oblate spheroid -> real spheroid

  x[1] = x[2] = 0;
  x[3] = 3;

  theta[1] = (x[1] - x1[1]) / pow (oblate_geo[1], 2);
  theta[2] = (x[2] - x1[2]) / pow (oblate_geo[2], 2);
  theta[3] = (x[3] - x1[3]) / pow (oblate_geo[3], 2);

  ntheta = 0;
  for (int i=1; i<=3; i++) ntheta += theta[i] * theta[i];

  for (int i=1; i<=3; i++)
    n[i] = (x[i] - x1[i]) / (pow (oblate_geo[i], 2) * sqrt (ntheta));
  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  jump (i,j,k,l) = - 4 * nu * (k==l) * n[i] * n[j] + 2 * (nu - 1) * ((i==k) * n[j] * n[l] + (i==l) * n[j] * n[k] + (j==k) * n[i] * n[l] + (j==l) * n[i] * n[k]) + 4 * n[i] * n[j] * n[k] * n[l];
  jump *= 0.25 / (1 - nu);

  test = (jump == SII.spheroid_exterior_points (C, oblate_geo, x) - SII.spheroid_interior_points (C, oblate_geo));
  if (detail) affiche ("Eshelby exterior points - imaginary oblate -> real oblate", test);
  else result *= (test);

  x[1] = 4;
  x[2] = 5;
  x[3] = 6;
  geo_sphere = vector<long double> (3, true, 3);
  geo_sph = geo_sphere;
  geo_sph[1] -= 1E-3;
  Delta_Sph = SII.spheroid_exterior_points (C, geo_sph, x) - SII.sphere (C, geo_sphere, x);
  if (detail) affiche ("Eshelby exterior points - oblate spheroid -> sphere", Delta_Sph.p_norm (2) < 1E-4);
  else result *= (Delta_Sph.p_norm (2) < 1E-4);

  geo_sph[1] = 3 - 1E-4;
  x[1] = x[2] = 0;
  x[3] = 3 + 1E-3;
  Delta_Sph = SII.spheroid_exterior_points (C, geo_sph, x) - SII.sphere (C, geo_sphere, x);
  if (detail) affiche ("Eshelby exterior points - imaginary oblate -> real oblate -> sphere", Delta_Sph.p_norm (2) < 5E-4);
  else result *= (Delta_Sph.p_norm (2) < 5E-4);


  //---------------------------------------------
  // compute Eshelby's tensor for interior points
  //---------------------------------------------

  lambda_ = 0;

  rho1 = oblate_geo[1] / sqrt (a1_2 + lambda_);
  rho2 = oblate_geo[2] / sqrt (a2_2 + lambda_);
  rho3 = oblate_geo[3] / sqrt (a3_2 + lambda_);

  if (alpha > 1) g = - alpha_2 / (alpha_2 - 1) * pow (rho2, 2) / rho1 + alpha / pow (alpha_2 - 1, 1.5) * log (sqrt (alpha_2 - 1) * rho2 + alpha * rho2 / rho1);
  else if (alpha <1) g = - alpha_2 / (alpha_2 - 1) * pow (rho2, 2) / rho1 + alpha / pow (1 - alpha_2, 1.5) * (atan (alpha / (sqrt (1 - alpha_2) * rho1)) - 0.5 * pi());

  S1(1,1) = (4 * nu + 2. / (alpha_2 - 1)) * g - 2. / (3 * (alpha_2 - 1)) * pow (rho1, 3) + (4 * nu + 2. / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S1(1,2) = S1(1,3) = (4 * nu - (2 * alpha_2 + 1) / (alpha_2 - 1)) * g +  (4 * nu - 2 * alpha_2 / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S1(2,1) = S1(3,1) = (-2 * nu - (2 * alpha_2 + 1) / (alpha_2 - 1)) * g - 2 * alpha_2 / (alpha_2 -1) * rho1 * pow (rho2, 2);
  S1(2,2) = S1(2,3) = S1(3,2) = S1(3,3) = (-2 * nu + (alpha_2 - 0.25) / (alpha_2 - 1)) * g + 0.5 * alpha_2 / (alpha_2 - 1) * pow (rho2, 4) / rho1;

  S2(1,1) = (-4 * nu + (4 * alpha_2 - 2) / (alpha_2 - 1)) * g - 2 * pow (rho1, 3) / (3 * (alpha_2 - 1)) - (4 * nu - (4 * alpha_2 - 2) / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S2(1,2) = S2(1,3) = S2(2,1) = S2(3,1) = (-nu - (alpha_2 + 2) / (alpha_2 - 1)) * g - (2*nu + 2. / (alpha_2 - 1)) * rho1 * pow (rho2, 2);
  S2(2,2) = S2(2,3) = S2(3,2) = S2(3,3) = (2 * nu - (alpha_2 - 7*0.25) / (alpha_2 - 1)) * g + 0.5 * alpha_2 / (alpha_2 - 1) * pow (rho2, 4) / rho1;

  for (int i=1; i<=3; i++)
    for (int j=1; j<=3; j++)
      for (int k=1; k<=3; k++)
	for (int l=1; l<=3; l++)
	  S(i,j,k,l) = S1(i,k) * (i==j) * (k==l) + S2(i,j) * ((i==k) * (j==l) + (i==l) * (j==k));
  S *= 0.25 / (1 - nu);

  oblate_geo[1] = 3;
  oblate_geo[2] = 3;
  oblate_geo[3] = 1.2;
  I_1 = I_2 = 2 * pi() * oblate_geo[3] * pow (oblate_geo[1], 2) / pow (pow (oblate_geo[1], 2) - pow (oblate_geo[3], 2), 1.5) * (acos (oblate_geo[3] / oblate_geo[1]) - oblate_geo[3] / oblate_geo[1] * sqrt (1 - pow (oblate_geo[3] / oblate_geo[1], 2)));
  I_3 = 4 * pi() - 2 * I_1;
  I_13 = I_23 = (I_1 - I_3) / (pow (oblate_geo[3], 2) - pow (oblate_geo[1], 2));
  I_11 = I_22 = I_12 = pi() / pow (oblate_geo[1], 2) - 0.25 * (I_1 - I_3) / (pow (oblate_geo[3], 2) - pow (oblate_geo[1],2));
  I_33 = (4 * pi() / pow (oblate_geo[3], 2) - 2 * I_13) / 3.;

  tensor4<long double> S_oblate (3,3,3,3);
  S_oblate(1,1,1,1) = S_oblate(2,2,2,2) = 3 / (8 * pi() * (1 - nu)) * pow (oblate_geo[1], 2) * I_11 + (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_1;
  S_oblate(3,3,3,3) = 3. / (8 * pi() * (1 - nu)) * pow (oblate_geo[3], 2) * I_33 + (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_3;
  S_oblate(1,1,2,2) = 1 / (8 * pi() * (1 - nu)) * pow (oblate_geo[2], 2) * I_12 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_1;
  S_oblate(2,2,1,1) = 1 / (8 * pi() * (1 - nu)) * pow (oblate_geo[1], 2) * I_12 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_2;
  S_oblate(2,2,3,3) = 1 / (8 * pi() * (1 - nu)) * pow (oblate_geo[3], 2) * I_23 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_2;
  S_oblate(3,3,2,2) = 1 / (8 * pi() * (1 - nu)) * pow (oblate_geo[2], 2) * I_23 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_3;
  S_oblate(1,1,3,3) = 1 / (8 * pi() * (1 - nu)) * pow (oblate_geo[3], 2) * I_13 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_1;
  S_oblate(3,3,1,1) = 1 / (8 * pi() * (1 - nu)) * pow (oblate_geo[1], 2) * I_13 - (1 - 2 * nu) / (8 * pi() * (1 - nu)) * I_3;
  S_oblate(1,2,1,2) = S_oblate(1,2,2,1) = S_oblate(2,1,1,2) = S_oblate(2,1,2,1) = (pow (oblate_geo[1], 2) + pow (oblate_geo[2], 2)) / (16 * pi() * (1 - nu)) * I_12 + (1 - 2 * nu) / (16 * pi() * (1 - nu)) * (I_1 + I_2);
  S_oblate(1,3,1,3) = S_oblate(1,3,3,1) = S_oblate(3,1,1,3) = S_oblate(3,1,3,1) = (pow (oblate_geo[1], 2) + pow (oblate_geo[3], 2)) / (16 * pi() * (1 - nu)) * I_13 + (1 - 2 * nu) / (16 * pi() * (1 - nu)) * (I_1 + I_3);
  S_oblate(2,3,2,3) = S_oblate(2,3,3,2) = S_oblate(3,2,2,3) = S_oblate(3,2,3,2) = (pow (oblate_geo[2], 2) + pow (oblate_geo[3], 2)) / (16 * pi() * (1 - nu)) * I_23 + (1 - 2 * nu) / (16 * pi() * (1 - nu)) * (I_2 + I_3);

  matrix<long double> P(3,3); // rotation matrix;
  P(1,3) = P(2,1) = P(3,2) = 1;

  S_oblate = change_basis (P, (symtensor4<long double>) S_oblate);
  oblate_geo[1] = 1.2;
  oblate_geo[2] = 3;
  oblate_geo[3] = 3;
  assert (S_oblate == S);
  if (detail) affiche ("Eshelby formula - oblate spheroid (a1 < a2 = a3)", S_oblate == SII.spheroid (C, oblate_geo));
  else result *= (S_oblate == SII.spheroid (C, oblate_geo));

  if (detail) affiche ("Eshelby interior points - oblate spheroid (a1 < a2 = a3)", S_oblate == SII.spheroid_interior_points (C, oblate_geo));
  else result *= (S_oblate == SII.spheroid_interior_points (C, oblate_geo));

  geo[1] = geo[2] = geo[3] = 5;
  geo[1] -= 0.001;
  Delta_S = SII.spheroid_interior_points (C, geo) - SII.sphere(C);
  if (detail) affiche ("Eshelby interior points - oblate spheroid -> sphere", Delta_S.p_norm (2) < 1.E-4);
  else result *= (Delta_S.p_norm (2) < 1.E-4);

  cout << endl;

  cout << "============================================================== \n";
  if (result) cout << green << "                     Eshelby test passed" << reset;
  else cout << red << "                     Eshelby test failed" << reset;
  cout << "============================================================== \n";

  return result;
}


#endif
