/*- <matfquad/net/socket/Socket.cpp> -*- C++ -*-
 *
 *
 *  matfquad
 *  Copyright (C) 2012  Márcio Adriano Tavares Fernandes
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
#include "matfquad/net/socket/Socket.hpp"
#include <cerrno>
#include <cstring>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
using namespace matfquad::net::socket;

#ifndef SO_DOMAIN
#define __SOCKET_COMPAT__
#endif
#ifndef SO_TYPE
#define __SOCKET_COMPAT__
#endif
#ifndef SO_PROTOCOL
#define __SOCKET_COMPAT__
#endif

Family *Socket::FAMILY_INET = Family_INET::getInstance();
Family *Socket::FAMILY_INET6 = Family_INET6::getInstance();

Type *Socket::TYPE_DGRAM = Type_DGRAM::getInstance();
Type *Socket::TYPE_STREAM = Type_STREAM::getInstance();

Protocol *Socket::PROTOCOL_IP = Protocol_IP::getInstance();
Protocol *Socket::PROTOCOL_TCP = Protocol_TCP::getInstance();
Protocol *Socket::PROTOCOL_UDP = Protocol_UDP::getInstance();

const int Socket::OPTION_LEVEL_SOCKET = SOL_SOCKET;

const int Socket::OPTION_BROADCAST = SO_BROADCAST;
const int Socket::OPTION_DONTROUTE = SO_DONTROUTE;
const int Socket::OPTION_ERROR = SO_ERROR;
const int Socket::OPTION_LINGER = SO_LINGER;
const int Socket::OPTION_REUSEADDR = SO_REUSEADDR;
#ifndef __SOCKET_COMPAT__
const int Socket::OPTION_FAMILY = SO_DOMAIN;
const int Socket::OPTION_TYPE = SO_TYPE;
const int Socket::OPTION_PROTOCOL = SO_PROTOCOL;
#endif

Socket::Socket()
{
  this->descriptor = -1;
}

Socket::Socket(int descriptor) throw (socket_error): Socket()
{
  if ( descriptor == -1 )
  {
    int errnum = errno;
    throw socket_error(strerror(errnum));
  }
  this->descriptor = descriptor;
}

Socket::Socket(int family_number, int type_number, int protocol_number) throw (socket_error): Socket(::socket(family_number, type_number, protocol_number))
{
#ifdef __SOCKET_COMPAT__
    this->family_number = family_number;
    this->type_number = type_number;
    this->protocol_number = protocol_number;
#endif
}

Socket::Socket(Family *family, Type *type, Protocol *protocol) throw (socket_error): Socket(family->getNumber(), type->getNumber(), protocol->getNumber())
{
}

Socket::~Socket()
{
  if ( this->descriptor != -1 )
  {
    ::shutdown(this->descriptor, SHUT_RD|SHUT_WR);
    ::close(this->descriptor);
    errno = 0;
  }
}

Socket *Socket::accept() throw (socket_accept_error)
{
  Socket *socket = NULL;
  size_t address_length = sizeof(struct sockaddr);
  struct sockaddr address;
  memset(&address, '\0', address_length);
  try
  {
    socket = new Socket(::accept(this->descriptor, &address, (socklen_t *)&address_length));
  }
  catch ( socket_error &e )
  {
    int errnum = errno;
    throw socket_accept_error(strerror(errnum));
  }
#ifdef __SOCKET_COMPAT__
    socket->family_number = this->family_number;
    socket->type_number = this->type_number;
    socket->protocol_number = this->protocol_number;
#endif
  return socket;
}

void Socket::bind(const void *address, size_t address_length) throw (socket_bind_error)
{
  if ( ::bind(this->descriptor, (const struct sockaddr *)address, (socklen_t)address_length) == -1 )
  {
    int errnum = errno;
    throw socket_bind_error(strerror(errnum));
  }
}

void Socket::bind(std::string hostname, std::string service) throw (socket_bind_error)
{
  Address address = Address(hostname, service, this->getFamilyNumber(), this->getTypeNumber(), this->getProtocolNumber(), 0);
  this->bind(&address);
}

void Socket::bind(Address *address) throw (socket_bind_error)
{
  int errnum = 0;
  for ( Address *address_next = address; address_next != NULL; )
  {
    try
    {
      this->bind(address_next->getAddress(), address_next->getAddressLength());
      address_next = NULL;
    } catch ( socket_bind_error &e )
    {
      errnum = errno;
      address_next = address_next->getNext();
    }
  }
  if ( errnum != 0 )
  {
    throw socket_bind_error(strerror(errnum));
  }
}

void Socket::connect(std::string hostname, std::string service) throw (socket_connect_error)
{
  Address address = Address(hostname, service, this->getFamilyNumber(), this->getTypeNumber(), this->getProtocolNumber(), 0);
  this->connect(&address);
}

void Socket::connect(const void *address, size_t address_length) throw (socket_connect_error)
{
  if ( ::connect(this->descriptor, (const struct sockaddr *)address, (socklen_t)address_length) == -1 )
  {
    int errnum = errno;
    throw socket_connect_error(strerror(errnum));
  }
}

void Socket::connect(Address *address) throw (socket_connect_error)
{
  int errnum = 0;
  for ( Address *address_next = address; address_next != NULL; )
  {
    try
    {
      this->connect(address_next->getAddress(), address_next->getAddressLength());
      address_next = NULL;
    } catch ( socket_connect_error &e )
    {
      errnum = errno;
      address_next = address_next->getNext();
    }
  }
  if ( errnum != 0 )
  {
    throw socket_connect_error(strerror(errnum));
  }
}

int Socket::getDescriptor()
{
  return this->descriptor;
}

Address Socket::getAddress()
{
  size_t address_length = sizeof(struct sockaddr);
  struct sockaddr address;
  memset(&address, '\0', address_length);
  this->getName(&address, &address_length);
  return Address(&address, address_length, this->getFamilyNumber(), this->getTypeNumber(), this->getProtocolNumber(), Address::NAMETOADDRESS_PASSIVE|Address::NAMETOADDRESS_CANONNAME, 0);
}

Address Socket::getAddressNumeric()
{
  size_t address_length = sizeof(struct sockaddr);
  struct sockaddr address;
  memset(&address, '\0', address_length);
  this->getName(&address, &address_length);
  return Address(&address, address_length, this->getFamilyNumber(), this->getTypeNumber(), this->getProtocolNumber(), Address::NAMETOADDRESS_PASSIVE|Address::NAMETOADDRESS_NUMERICHOST, Address::ADDRESSTONAME_NUMERICHOST);
}

int Socket::getError() throw (socket_getoptions_error)
{
  int value = 0;
  size_t value_length = sizeof(int);
  this->getOptions(Socket::OPTION_LEVEL_SOCKET, Socket::OPTION_ERROR, &value, &value_length);
  return value;
}

std::string Socket::getErrorDescription() throw (socket_getoptions_error)
{
  return std::string(strerror(this->getError()));
}

Family Socket::getFamily() throw (socket_getoptions_error)
{
  return Family(this->getFamilyNumber());
}

int Socket::getFamilyNumber() throw (socket_getoptions_error)
{
#ifndef __SOCKET_COMPAT__
  int value = 0;
  size_t value_length = sizeof(int);
  this->getOptions(Socket::OPTION_LEVEL_SOCKET, Socket::OPTION_FAMILY, &value, &value_length);
  return value;
#else
  return this->family_number;
#endif
}

void Socket::getName(void *address, size_t *address_length) throw (socket_getname_error)
{
  if ( ::getsockname(this->descriptor, (struct sockaddr *)address, (socklen_t *)address_length) == -1 )
  {
    int errnum = errno;
    throw socket_getname_error(strerror(errnum));
  }
}

bool Socket::getOptionBroadcast() throw (socket_getoptions_error)
{
  int value = 0;
  size_t value_length = sizeof(int);
  this->getOptions(Socket::OPTION_LEVEL_SOCKET, Socket::OPTION_BROADCAST, &value, &value_length);
  return value;
}

bool Socket::getOptionDontRoute() throw (socket_getoptions_error)
{
  int value = 0;
  size_t value_length = sizeof(int);
  this->getOptions(Socket::OPTION_LEVEL_SOCKET, Socket::OPTION_DONTROUTE, &value, &value_length);
  return value;
}

void Socket::getOptions(int level, int option_name, void *option_value, size_t *option_value_length) throw (socket_getoptions_error)
{
  if ( getsockopt(this->descriptor, level, option_name, option_value, (socklen_t *)option_value_length) != 0 )
  {
    int errnum = errno;
    throw socket_getoptions_error(strerror(errnum));
  }
}

Address Socket::getPeerAddress()
{
  size_t address_length = sizeof(struct sockaddr);
  struct sockaddr address;
  memset(&address, '\0', address_length);
  this->getPeerName(&address, &address_length);
  return Address(&address, address_length, this->getFamilyNumber(), this->getTypeNumber(), this->getProtocolNumber(), Address::NAMETOADDRESS_PASSIVE|Address::NAMETOADDRESS_CANONNAME, 0);
}

Address Socket::getPeerAddressNumeric()
{
  size_t address_length = sizeof(struct sockaddr);
  struct sockaddr address;
  memset(&address, '\0', address_length);
  this->getPeerName(&address, &address_length);
  return Address(&address, address_length, this->getFamilyNumber(), this->getTypeNumber(), this->getProtocolNumber(), Address::NAMETOADDRESS_PASSIVE|Address::NAMETOADDRESS_NUMERICHOST, Address::ADDRESSTONAME_NUMERICHOST);
}

void Socket::getPeerName(void *address, size_t *address_length) throw (socket_getpeername_error)
{
  if ( ::getpeername(this->descriptor, (struct sockaddr *)address, (socklen_t *)address_length) == -1 )
  {
    int errnum = errno;
    throw socket_getpeername_error(strerror(errnum));
  }
}

Protocol Socket::getProtocol() throw (socket_getoptions_error)
{
  return Protocol(this->getProtocolNumber());
}

int Socket::getProtocolNumber() throw (socket_getoptions_error)
{
#ifndef __SOCKET_COMPAT__
  int value = 0;
  size_t value_length = sizeof(int);
  this->getOptions(Socket::OPTION_LEVEL_SOCKET, Socket::OPTION_PROTOCOL, &value, &value_length);
  return value;
#else
  return this->protocol_number;
#endif
}

Type Socket::getType() throw (socket_getoptions_error)
{
  return Type(this->getTypeNumber());
}

int Socket::getTypeNumber() throw (socket_getoptions_error)
{
#ifndef __SOCKET_COMPAT__
  int value = 0;
  size_t value_length = sizeof(int);
  this->getOptions(Socket::OPTION_LEVEL_SOCKET, Socket::OPTION_TYPE, &value, &value_length);
  return value;
#else
  return this->type_number;
#endif
}

void Socket::listen(int backlog) throw (socket_listen_error)
{
  if ( ::listen(this->descriptor, backlog) == -1 )
  {
    int errnum = errno;
    throw socket_listen_error(strerror(errnum));
  }
}

ssize_t Socket::send(const void *buffer, size_t buffer_length) throw (socket_send_error)
{
  return this->send(buffer, buffer_length, 0);
}

ssize_t Socket::send(const void *buffer, size_t buffer_length, int flags) throw (socket_send_error)
{
  ssize_t result = ::send(this->descriptor, buffer, buffer_length, flags);
  if ( result == -1 )
  {
    int errnum = errno;
    throw socket_send_error(strerror(errnum));
  }
  return result;
}

ssize_t Socket::sendTo(const void *buffer, size_t buffer_length, const void *destination_address, size_t destination_address_length) throw (socket_sendto_error)
{
  return this->sendTo(buffer, buffer_length, destination_address, destination_address_length, 0);
}

ssize_t Socket::sendTo(const void *buffer, size_t buffer_length, const void *destination_address, size_t destination_address_length, int flags) throw (socket_sendto_error)
{
  ssize_t result = ::sendto(this->descriptor, buffer, buffer_length, flags, (const struct sockaddr *)destination_address, (socklen_t)destination_address_length);
  if ( result == -1 )
  {
    int errnum = errno;
    throw socket_send_error(strerror(errnum));
  }
  return result;
}

void Socket::setOptionBroadcast(bool status) throw (socket_setoptions_error)
{
  int value = status;
  size_t value_length = sizeof(int);
  this->setOptions(Socket::OPTION_LEVEL_SOCKET, Socket::OPTION_BROADCAST, &value, value_length);
}

void Socket::setOptionDontRoute(bool status) throw (socket_setoptions_error)
{
  int value = status;
  size_t value_length = sizeof(int);
  this->setOptions(Socket::OPTION_LEVEL_SOCKET, Socket::OPTION_DONTROUTE, &value, value_length);
}

void Socket::setOptions(int level, int option_nome, const void *option_value, size_t option_value_length) throw (socket_setoptions_error)
{
  if ( ::setsockopt(this->descriptor, level, option_nome, option_value, (socklen_t)option_value_length) != 0 )
  {
    int errnum = errno;
    throw socket_setoptions_error(strerror(errnum));
  }
}

ssize_t Socket::receive(void *buffer, size_t buffer_length) throw (socket_receive_error)
{
  return this->receive(buffer, buffer_length, 0);
}

ssize_t Socket::receive(void *buffer, size_t buffer_length, int flags) throw (socket_receive_error)
{
  ::memset(buffer, '\0', sizeof(buffer_length));
  ssize_t result = ::recv(this->descriptor, buffer, buffer_length, flags);
  if ( result == -1 )
  {
    int errnum = errno;
    throw socket_receive_error(strerror(errnum));
  }
  return result;
}

ssize_t Socket::receiveFrom(void *buffer, size_t buffer_length, void *source_address, size_t *source_address_length) throw (socket_receivefrom_error)
{
  return this->receiveFrom(buffer, buffer_length, source_address, source_address_length, 0);
}

ssize_t Socket::receiveFrom(void *buffer, size_t buffer_length, void *source_address, size_t *source_address_length, int flags) throw (socket_receivefrom_error)
{
  ::memset(buffer, '\0', sizeof(buffer_length));
  ssize_t result = ::recvfrom(this->descriptor, buffer, buffer_length, flags, (struct sockaddr *)source_address, (socklen_t *)source_address_length);
  if ( result == -1 )
  {
    int errnum = errno;
    throw socket_receive_error(strerror(errnum));
  }
  return result;
}
