/*- <matfquad/net/socket/Socket.hpp> -*- C++ -*-
 *
 *
 *  matfquad
 *  Copyright (C) 2012  Márcio Adriano Tavares Fernandes
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
#ifndef __MATFQUAD_NET_SOCKET_SOCKET_HPP__
#define __MATFQUAD_NET_SOCKET_SOCKET_HPP__
#include <stdexcept>
#include "Family.hpp"
#include "Type.hpp"
#include "Protocol.hpp"
#include "Address.hpp"

namespace matfquad
{
namespace net
{
namespace socket
{
class socket_error: public std::runtime_error
{
 public:
  explicit socket_error(): socket_error("Error")
  {
  }
  explicit socket_error(const std::string &msg): std::runtime_error(msg)
  {
  }
};

class socket_accept_error: public socket_error
{
 public:
  explicit socket_accept_error(): socket_accept_error("Accept error")
  {
  }
  explicit socket_accept_error(const std::string &msg): socket_error(msg)
  {
  }
};

class socket_bind_error: public socket_error
{
 public:
  explicit socket_bind_error(): socket_bind_error("Bind error")
  {
  }
  explicit socket_bind_error(const std::string &msg): socket_error(msg)
  {
  }
};

class socket_connect_error: public socket_error
{
 public:
  explicit socket_connect_error(): socket_connect_error("Connect error")
  {
  }
  explicit socket_connect_error(const std::string &msg): socket_error(msg)
  {
  }
};

class socket_getname_error: public socket_error
{
 public:
  explicit socket_getname_error(): socket_getname_error("GetName error")
  {
  }
  explicit socket_getname_error(const std::string &msg): socket_error(msg)
  {
  }
};

class socket_getoptions_error: public socket_error
{
 public:
  explicit socket_getoptions_error(): socket_getoptions_error("GetOptions error")
  {
  }
  explicit socket_getoptions_error(const std::string &msg): socket_error(msg)
  {
  }
};

class socket_getpeername_error: public socket_error
{
 public:
  explicit socket_getpeername_error(): socket_getpeername_error("GetPeerName error")
  {
  }
  explicit socket_getpeername_error(const std::string &msg): socket_error(msg)
  {
  }
};

class socket_listen_error: public socket_error
{
 public:
  explicit socket_listen_error(): socket_listen_error("Listen error")
  {
  }
  explicit socket_listen_error(const std::string &msg): socket_error(msg)
  {
  }
};

class socket_receive_error: public socket_error
{
 public:
  explicit socket_receive_error(): socket_receive_error("Receive error")
  {
  }
  explicit socket_receive_error(const std::string &msg): socket_error(msg)
  {
  }
};

class socket_receivefrom_error: public socket_receive_error
{
 public:
  explicit socket_receivefrom_error(): socket_receivefrom_error("ReceiveFrom error")
  {
  }
  explicit socket_receivefrom_error(const std::string &msg): socket_receive_error(msg)
  {
  }
};

class socket_send_error: public socket_error
{
 public:
  explicit socket_send_error(): socket_send_error("Send error")
  {
  }
  explicit socket_send_error(const std::string &msg): socket_error(msg)
  {
  }
};

class socket_sendto_error: public socket_send_error
{
 public:
  explicit socket_sendto_error(): socket_sendto_error("SendTo error")
  {
  }
  explicit socket_sendto_error(const std::string &msg): socket_send_error(msg)
  {
  }
};

class socket_setoptions_error: public socket_error
{
 public:
  explicit socket_setoptions_error(): socket_setoptions_error("SetOptions error")
  {
  }
  explicit socket_setoptions_error(const std::string &msg): socket_error(msg)
  {
  }
};

class Socket
{
 public:
  static Family *FAMILY_INET;
  static Family *FAMILY_INET6;
  static Type *TYPE_DGRAM;
  static Type *TYPE_STREAM;
  static Protocol *PROTOCOL_IP;
  static Protocol *PROTOCOL_TCP;
  static Protocol *PROTOCOL_UDP;
  static const int OPTION_LEVEL_SOCKET;
  static const int OPTION_BROADCAST;
  static const int OPTION_DONTROUTE;
  static const int OPTION_ERROR;
  static const int OPTION_LINGER;
  static const int OPTION_REUSEADDR;
  static const int OPTION_FAMILY;
  static const int OPTION_TYPE;
  static const int OPTION_PROTOCOL;
  static const int MESSAGE_DONTROUTE;
  static const int MESSAGE_DONTWAIT;
  static const int MESSAGE_NOSIGNAL;
  static const int MESSAGE_OUTOFBAND;
  static const int MESSAGE_PEEK;
  static const int MESSAGE_WAITALL;
private:
  int descriptor;
 protected:
  Socket();
 public:
  Socket(int descriptor) throw (socket_error);
  Socket(int family_number, int type_number, int protocol_number) throw (socket_error);
  Socket(Family *family, Type *type, Protocol *protocol) throw (socket_error);
  ~Socket();
  Socket *accept() throw (socket_accept_error);
  void bind(const void *address, size_t address_length) throw (socket_bind_error);
  void bind(std::string hostname, std::string service) throw (socket_bind_error);
  void bind(Address *address) throw (socket_bind_error);
  void connect(std::string hostname, std::string service) throw (socket_connect_error);
  void connect(const void *address, size_t address_length) throw (socket_connect_error);
  void connect(Address *address) throw (socket_connect_error);
  Address getAddress();
  Address getAddressNumeric();
  int getDescriptor();
  int getError() throw (socket_getoptions_error);
  std::string getErrorDescription() throw (socket_getoptions_error);
  Family getFamily() throw (socket_getoptions_error);
  int getFamilyNumber() throw (socket_getoptions_error);
  void getName(void *address, size_t *address_length) throw (socket_getname_error);
  bool getOptionBroadcast() throw (socket_getoptions_error);
  bool getOptionDontRoute() throw (socket_getoptions_error);
  void getOptions(int level, int option_name, void *option_value, size_t *option_value_length) throw (socket_getoptions_error);
  Address getPeerAddress();
  Address getPeerAddressNumeric();
  void getPeerName(void *address, size_t *address_length) throw (socket_getpeername_error);
  Protocol getProtocol() throw (socket_getoptions_error);
  int getProtocolNumber() throw (socket_getoptions_error);
  Type getType() throw (socket_getoptions_error);
  int getTypeNumber() throw (socket_getoptions_error);
  void listen(int backlog) throw (socket_listen_error);
  ssize_t send(const void *buffer, size_t buffer_length) throw (socket_send_error);
  ssize_t send(const void *buffer, size_t buffer_length, int flags) throw (socket_send_error);
  ssize_t sendTo(const void *buffer, size_t buffer_length, const void *destination_address, size_t destination_address_length) throw (socket_sendto_error);
  ssize_t sendTo(const void *buffer, size_t buffer_length, const void *destination_address, size_t destination_address_length, int flags) throw (socket_sendto_error);
  void setOptionBroadcast(bool status) throw (socket_setoptions_error);
  void setOptionDontRoute(bool status) throw (socket_setoptions_error);
  void setOptions(int level, int option_nome, const void *option_value, size_t option_value_length) throw (socket_setoptions_error);
  ssize_t receive(void *buffer, size_t buffer_length) throw (socket_receive_error);
  ssize_t receive(void *buffer, size_t buffer_length, int flags) throw (socket_receive_error);
  ssize_t receiveFrom(void *buffer, size_t buffer_length, void *source_address, size_t *source_address_length) throw (socket_receivefrom_error);
  ssize_t receiveFrom(void *buffer, size_t buffer_length, void *source_address, size_t *source_address_length, int flags) throw (socket_receivefrom_error);
};
}
}
}

#endif
