// A mathematical game
// Copyright (C) 2004 by Christian von Schultz <schultz@linux.nu>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <wx/wx.h>

#include <config.h>
#include "checksetup.h"

#include "module.h"
#include "application.h"
#include "windowcontents.h"
#include "timeguard.h"

// Explicit template instantiation -- needed by g++ 3.3
template class Module<WindowContents>;
template class Module<TimeGuard>;
template class Module<RoomGuard>;

wxString Module<WindowContents>::object_type = "WindowContents";
wxString Module<TimeGuard>::object_type = "TimeGuard";
wxString Module<RoomGuard>::object_type = "RoomGuard";

template <class T> Module<T>::Module(const wxString &lib) throw(Exception) try:
  object(NULL),
  m_create(new create_t*),
  m_destroy(new destroy_t*),
  is_loaded(false),
  m_lib(lib)
{
  wxLogDebug("Constructing module.");

  LoadDynamicLibrary(lib);
  CheckVersion();
  LoadFactories();
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

template <class T> Module<T>::Module(const Module &mod) throw(NoisyBadAlloc) try:
  object(NULL),
  m_create(new create_t*),
  m_destroy(new destroy_t*),
  is_loaded(mod.is_loaded),
  m_lib(mod.m_lib)
{
  *m_create = *mod.m_create;
  *m_destroy = *mod.m_destroy;
  Ref(mod);
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

template <class T> Module<T>::~Module() throw()
{
  wxLogDebug("Destructing module.");
  wxLog::FlushActive();
  if(object != NULL)
    (*m_destroy)(object);

  delete m_destroy;
  delete m_create;

  if(GetRefData() != NULL)
  {
    wxLogDebug("  Unload()ing dynamic library.");
    UnRef();
  }
}

template <class T> Module<T>::DynLibRefData::DynLibRefData(const wxString &libfile)
  throw(NoisyBadAlloc) try:
  m_dynamic_library(new wxDynamicLibrary(libfile))
{
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

template <class T> Module<T>::DynLibRefData::~DynLibRefData() throw()
{
  delete m_dynamic_library;
}

template <class T> void Module<T>::LoadDynamicLibrary(const wxString &lib) throw(Exception)
try
{
  m_libfile = GetApp()->config->GetModuleFile(lib);

  if(GetRefData() != NULL)
  {
    UnRef();
  }
  SetRefData(new DynLibRefData(m_libfile));

  if(! GetDynLib()->IsLoaded())
  {
    throw NoisyException
      (wxString::Format(_("Could not load module (\"%s\"). Check that "
			  "the program is properly installed."),
			m_libfile.c_str()));
  }
  else
  {
    wxLogDebug("a new wxDynamicLibrary(\"%s\") has been created", m_libfile.c_str());
  }
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

template <class T> void Module<T>::CheckVersion() throw(Exception)
{
  wxASSERT(GetDynLib()->IsLoaded());

  char *(*version)(void);
  *(void**)(&version) = GetDynLib()->GetSymbol("version");

  if(version == NULL)
  {
    throw NoisyException
      (wxString::Format(_("This module (\"%s\") does not seem to be made for this "
			  "program. There is probably an error in the room list."),
			 m_libfile.c_str()));
  }

  if(wxString(version()) != wxString(PACKAGE_STRING))
  {
    throw NoisyException
      (wxString::Format(_("This module has the wrong version number. "
			  "All modules to be loaded by %s must have "
			  "the version %s."),
			PACKAGE_NAME, PACKAGE_STRING));
  }
 
  char *(*type)(void);
  *(void**)(&type) = GetDynLib()->GetSymbol("type");

  if(type == NULL)
  {
    throw NoisyException
      (wxString::Format(_("This module (\"%s\") seems to have an error. While it "
			  "says it's written for this version of this program, "
			  "it does not say which type of objects it defines, which "
			  "this version of this program requires."),
			 m_libfile.c_str()));
  }

  if(wxString(type()) != object_type)
  {
    throw NoisyException
      (wxString::Format(_("There seems to be an error in the room list. This "
			  "module\n(\"%s\") has been loaded as a %s module, "
			  "but it is in fact a %s module."),
			m_libfile.c_str(), object_type.c_str(), type()));
  }

 
  wxLogDebug("This module has the right version number and type.");
}

template <class T> void Module<T>::LoadFactories() throw(Exception)
{
  wxASSERT(GetDynLib()->IsLoaded());

  *m_create = (create_t*) GetDynLib()->GetSymbol("create");
  *m_destroy = (destroy_t*) GetDynLib()->GetSymbol("destroy");

  if(*m_create == NULL  ||  *m_destroy == NULL)
  {
    throw NoisyException
      (wxString::Format(_("Failed to load module \"%s\". Every module has to "
			  "define the functions \"create\" and \"destroy\"."),
			m_libfile.c_str()));
  }
  else
  {
    is_loaded = true;
  }
}

template <class T> void Module<T>::CreateObject() throw(Exception)
{
  wxASSERT(object == NULL);
  
  if(IsLoaded())
  {
    object = (*m_create)();
    if(object == NULL)
      throw NoisyBadAlloc();
  }
  else
  {
    throw NoisyException(_("Could not create object: "
			   "module not loaded"));
  }
}
