// A mathematical game
// Copyright (C) 2004 by Christian von Schultz <schultz@linux.nu>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


#include <wx/wx.h>

#include <config.h>
#include "../checksetup.h"

#include "../application.h"
#include "rgchange.h"

extern "C" char *version(void)
{
  return PACKAGE_STRING;
}

extern "C" char *type(void)
{
  return "RoomGuard";	// This is a RoomGuard object
}

void RGChange::Init(const wxExpr *expr) throw(Exception)
try
{
  RoomGuard::Init(expr);
  wxLogDebug("Entering RGChange::Init()");

  if(! m_expr->GetAttributeValue("change", m_change))
    throw NoisyException(_("The \"RGChange\" module requires "
			   "the attribute \"change\"."));
  
  try
  {
    const wxExpr *el = m_expr->AttributeValue("attributes");
    // el is a wxExpr list
    if(el == NULL)
      throw NullException("No list.");

    if(el->Type() != wxExprList)
      throw NoisyException(_("Error in room list. The room guards must "
			     "be entered in a list."));
    
    for(wxExpr *item = el->GetFirst(); item != NULL; item = item->GetNext())
    {
      if(item->Type() != wxExprString)
      {
	wxLogWarning(_("Each item in a room guard list must be a string."));
	continue;
      }

      m_hash[ item->StringValue() ] = m_expr->AttributeValue(item->StringValue())->Copy();
    }
  }
  catch(SilentException &ex)
  {
    throw NoisyException(_("The \"RGChange\" module requires "
			   "the attribute \"attributes\"."));
  }    
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

RGChange::~RGChange()
{  
  for(MyHash::iterator iter = m_hash.begin();
      iter != m_hash.end();
      ++iter)
  {
    if(iter->second != NULL)
    {
      delete iter->second;
      iter->second = NULL;
    }
  }
}


void RGChange::Replace(wxExpr *new_room) throw(Exception)
try
{
  if(new_room->Functor() == m_change)
  {
    wxLogDebug("This is the one we're supposed to change.");

    for(MyHash::iterator iter = m_hash.begin();
	iter != m_hash.end();
	++iter)
    {
      wxLogDebug("  Now considering attribute %s", iter->first.c_str());
      new_room->DeleteAttributeValue(iter->first);
      if(iter->second != NULL)
      {
	if(iter->second->Type() != wxExprNull)
	{
	  new_room->AddAttributeValue(iter->first, iter->second->Copy());
	}
      }
    }
  }
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

// The class factories

extern "C" RoomGuard* create(void)
{
  wxLogDebug("Creating an RGChange object.");
  return new(std::nothrow) RGChange();
}

extern "C" void destroy(RoomGuard *w)
{
  wxLogDebug("Deleting a RoomGuard object (i.e. an RGChange object)");
  delete w;
}
