// A mathematical game
// Copyright (C) 2004 by Christian von Schultz <schultz@linux.nu>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


#ifndef APPLICATION_H
#define APPLICATION_H

#include <wx/config.h>
#include <wx/hashmap.h>
#include <wx/dynlib.h>
#include <wx/wxexpr.h>
#include <new>
#include <list>
using namespace std;

#include <config.h>
#include "checksetup.h"

#include "exceptions.h"

class MainWindow;
inline MainWindow *GetTopWindow() throw(NullException);

#include "roomlist.h"
#include "mainwindow.h"
#include "configuration.h"
#include "module.h"
#include "windowcontents.h"
#include "timer.h"
#include "timeguard.h"
#include "roomguard.h"
#include "roomguard_datacontrol.h"

class EXPORT Application : public wxApp
{
public:
  Application() throw();

  // Instead of constructor/destructor
  virtual bool OnInit();
  virtual int OnExit();

  // Command line processing
  virtual void OnInitCmdLine(wxCmdLineParser &parser);
  virtual bool OnCmdLineParsed(wxCmdLineParser &parser);

  // Data access - the c in cGetData means that the function creates
  // something you must remember to delete (having it in the function
  // name makes it impossible to forget it)
  void AddLabelledData(const wxString &label, wxExpr *value) throw(NullException);
  wxExpr* cGetData(const wxString &label) const throw(NullException);
  void ForgetData(const wxString &label) throw();

public:
  // Save and restore session
  void Save(const wxString &filename) throw(Exception);
  void Restore(const wxString &filename) throw(Exception);
private:
  wxString m_file_to_open;

public:
  wxString LocaleCanonicalName() const throw();

  Config *config;

  WindowContents *GetCurrentWC() throw(SilentException);

  // Do not call Replace() from the modules. It will delete the current
  // module, so if you do, you'll return to a deleted place in memory
  // when Replace() has finished. Tell the main window to call Replace()
  // instead, and it will wait until it can be done safely.
  void Replace(const wxString &room_name) throw(Exception);
private:
  void Replace(const wxExpr *room) throw(Exception);
  void InitialReplace(const wxExpr *p_expr) throw(Exception);


public:
  friend Application *GetApp() throw(NullException);
  static Application *m_the_app;   // (se comment on GetApp() below)

  Timer *timer;

private:
  wxLocale *m_locale;

  RoomList *m_roomlist;
public:
  RoomList *GetRoomList() throw(NullException)
  { if(m_roomlist == NULL) throw NullException("no roomlist"); else return m_roomlist; }

private:
  MainWindow *m_main_window;

  Module<WindowContents> *m_current_module;



  //////////////// TimeGuard related stuff ////////////////
  list<TimeGuard*> m_time_guards;
  wxString m_timeout;		// the functor of the timeout clause

  void AddTimeGuards(wxExpr *expr_list) throw();
public:
  // TimeGuards will be deleted automatically when you call RemoveTimeGuard().
  // Do not delete them yourself.
  void AddTimeGuard(TimeGuard *tg) { m_time_guards.push_back(tg); }
  void LoadTimeGuard(const wxString &functor) throw(Exception);
  void RemoveTimeGuard(TimeGuard* &tg);

  // These are used by the Timer class.
  typedef list<TimeGuard*>::iterator TimeGuardIterator;

  TimeGuardIterator TimeGuardBegin() { return m_time_guards.begin(); }
  TimeGuardIterator TimeGuardEnd() { return m_time_guards.end(); }



private:
  //////////////// RoomGuard related stuff ////////////////
  list<RoomGuard*> m_room_guards;

  void AddRoomGuards(wxExpr *expr_list) throw();
public:
  // RoomGuards will be deleted automatically when you call RemoveRoomGuard().
  // Do not delete them yourself.
  void AddRoomGuard(RoomGuard *rg) { m_room_guards.push_back(rg); }
  void LoadRoomGuard(const wxString &functor) throw(Exception);
  void RemoveRoomGuard(RoomGuard* &rg);

  // Replace() uses these:
  typedef list<RoomGuard*>::iterator RoomGuardIterator;

  RoomGuardIterator RoomGuardBegin() { return m_room_guards.begin(); }
  RoomGuardIterator RoomGuardEnd() { return m_room_guards.end(); } 
};

// Usually one would use wxGetApp() to find the application. But that
// function is defined by IMPLEMENT_APP() in main.cc, and that's not
// part of this DLL. That would not be a problem with Linux, but it
// seems impossible for a DLL to access main application functions and
// variables under Windows. So, we need to implement our own GetApp().
Application *GetApp() throw(NullException);

inline MainWindow *GetTopWindow() throw(NullException)
{
  if(GetApp()->GetTopWindow() == NULL)
    throw NullException("No top window");

  wxASSERT(dynamic_cast<MainWindow*>(GetApp()->GetTopWindow()) != NULL);
  return dynamic_cast<MainWindow*>(GetApp()->GetTopWindow());
}

void GetLocalizedAttribute(const wxExpr *expr, const wxString &name,
			   wxString &dest) throw(Exception);

// You can enter the default colour wich will be returned if none is
// found in the room list.
wxColour GetColourAttribute(const wxExpr *expr, const wxString &name,
			    long red = 0, long green = 0, long blue = 0)
  throw(Exception);

inline wxString from_utf8(const wxString &s)
{
  return wxString(wxConvUTF8.cMB2WC(s), *wxConvCurrent);
}

#endif // APPLICATION_H

// 
// Local Variables:
// mode: c++
// End:
