// A mathematical game
// Copyright (C) 2004-2005 by Christian von Schultz <schultz@linux.nu>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <wx/config.h>
#include <wx/filename.h>
#include <wx/dynlib.h>
#include <wx/intl.h>
#include <wx/setup.h>
#include <wx/dirdlg.h>
#include <wx/msgdlg.h>
#include <wx/log.h>

#include <config.h>
#include "checksetup.h"

#include "configuration.h"
#include "application.h"
#include "exceptions.h"
#include "wordwrap.h"

// The name of the environment variable holding the path to the modules
const wxString
Config::module_path_env(wxString(PACKAGE_TARNAME).MakeUpper() + "_MODULES");

Config::Config() throw(NoisyBadAlloc) try:
  m_generate_config(false)
{
  wxLogDebug("Constructing Config object...");
  
  //////////////// Construct wxConfig object ////////////////
#if wxUSE_UNIX
  wxConfig::Set(new wxConfig(PACKAGE_NAME, PACKAGE_VENDOR,
			     "." PACKAGE_TARNAME "rc", PACKAGE_TARNAME));
#else
  wxConfig::Set(new wxConfig(PACKAGE_NAME, PACKAGE_VENDOR));
#endif


  //////////////// Find modules ////////////////
  if(wxDirExists(MODULEDIR))
    m_module_path.Add(MODULEDIR);
#if __WXDEBUG__
  else
    wxLogDebug("  Compiled-in module directory not found: %s", MODULEDIR);
#endif

  // We look at the environment, then in the config file
  wxGetEnv(module_path_env, &m_env);
  m_module_path.AddEnvList(module_path_env);
  wxString env;
  wxConfig::Get()->Read("/Modules/" + module_path_env, &env);
  wxSetEnv(module_path_env, env.c_str());
  m_module_path.AddEnvList(module_path_env);
 
  if(m_env == wxEmptyString)
    m_env = env;

  wxLogDebug("  %s = \"%s\"", module_path_env.c_str(), m_env.c_str());
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

Config::~Config() throw()
try
{
  wxLogDebug("Destructing Config object...");
  GenerateConfig();
  delete wxConfig::Set(NULL);
}
catch(...)
{
  wxLogDebug("Warning: The Config destructor caught an exception.");
}

void Config::FindLocaleDir(wxLocale *locale) throw(NullException)
{
  if(locale == NULL)
  {
    throw NullException("Config::FindLocaleDir(NULL)");
  }
  wxString localedir = wxEmptyString;
  if(!locale->IsLoaded(PACKAGE_TARNAME))
  {
    wxConfig::Get()->Read("/Locale/directory", &localedir);

    if(localedir != wxEmptyString)
    {
      locale->AddCatalogLookupPathPrefix(localedir);
      m_localedir = localedir;
      m_localedir.MakeAbsolute();
    }
    else
    {
      locale->AddCatalogLookupPathPrefix(LOCALEDIR);
    }
  }
}

wxString Config::GetRoomListFile() throw(Exception)
{
  if(m_roomlist_file.IsOk())
  {
    return m_roomlist_file.GetFullPath();
  }
  else
  {
    LookForValidRoomListFile();
    return m_roomlist_file.GetFullPath();
  }
}

void Config::SetRoomListFile(wxFileName filename) throw(Exception)
{
  if(!filename.FileExists() || filename.DirExists() || !filename.IsOk())
  {
    throw NoisyException(_("--roomlist error: the room list must be a file."));
  }
  else
  {
    filename.MakeAbsolute();
    m_roomlist_file = filename;
  }
}  

void Config::SetLocaleDir(wxString directory, wxLocale **locale) throw(Exception)
try
{
  wxASSERT(locale != NULL);
  wxASSERT(*locale != NULL);
  if(directory == wxEmptyString)
  {
    // Translating this string might seem superfluous, but there is a
    // tiny chance that a user wants to change the locale directory,
    // even though his/her translation files are properly loaded.
    directory = wxDirSelector(_("Where can the translation files be found?"),
			      wxEmptyString,	// default path
			      0,		// dialog style
			      wxDefaultPosition,
			      GetTopWindow());
  }

  if(directory == wxEmptyString || !wxDirExists(directory))
  {
    throw NoisyException(_("No valid directory specified. Using default "
			   "locale directory."));
  }

  m_localedir = directory;
  m_localedir.MakeAbsolute();
  delete *locale;
  *locale = new wxLocale(wxLANGUAGE_DEFAULT);
  
  (*locale)->AddCatalogLookupPathPrefix(directory);
  if(! (*locale)->AddCatalog(PACKAGE_TARNAME))
  {
    throw NoisyException(_("Sorry, no translations found in the specified "
			   "directory."));
  }
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

wxString Config::GetModuleFile(const wxString &module_name) throw(Exception)
try
{
  wxString module;
  wxString dir;

  while(module == "" || !wxFileExists(module))
  {
    module = m_module_path.FindAbsoluteValidPath(module_name);

    if(module == "" || !wxFileExists(module))
      module = m_module_path.FindAbsoluteValidPath(module_name +
						  wxDynamicLibrary::GetDllExt());

    if(module == "" || !wxFileExists(module))
    {
      wxASSERT(GetTopWindow() != NULL);
      // We have to ask the user about this one.
      int answer = wxMessageBox(
        wxString::Format(WordWrap(
			 _("I could not find the module \"%s\". This is "
			   "probably a fatal error. The program might "
			   "not be properly installed. You could try to "
			   "find the module yourself though. "
			   "If you find it, click \"yes\".\n\n"
			   "Have you found the module?")),
			 module_name.c_str()),
	_("Module not found!"),
	wxYES_NO | wxICON_ERROR, GetTopWindow());

      if(answer != wxYES)
	throw NoisyException(_("Module not found."));
      else
      {
	dir = wxDirSelector(
	      wxString::Format(_("Where can the module %s be found?"),
			       module_name.c_str()),
	      wxEmptyString,	// default path
	      0,		// dialog style
	      wxDefaultPosition,
	      GetTopWindow());
	if(dir == "" || !wxDirExists(dir))
	{
	  throw NoisyException(_("Sorry, no valid directory has been entered."));
	}
	else
	{
	  m_module_path.Add(dir);
	  module = m_module_path.FindAbsoluteValidPath(module_name);
	}
      }
    }
  }

  if(dir != wxEmptyString)
  {
    if(m_env == wxEmptyString)
    {
      m_env = dir;
    }
    else
    {
#ifdef wxUSE_UNIX
      m_env += ":" + dir;
#else
      m_env += ";" + dir;
#endif
    }
  }

  return module;
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}
catch(NullException &ex)
{
 // Promote any null exceptions fom GetTopWindow() to noisy ones.
  throw NoisyException(ex.what());
}

wxString Config::GetModuleDataFile(const wxString &filename) throw(Exception)
try
{
  wxString datafile;
  wxString dir;

  while(datafile == "" || !wxFileExists(datafile))
  {
    datafile = m_module_path.FindAbsoluteValidPath(filename);

    if(datafile == "" || !wxFileExists(datafile))
    {
      wxASSERT(GetTopWindow() != NULL);
      // We have to ask the user about this one.
      int answer = wxMessageBox
	(WordWrap
	 (wxString::Format
	  (_("I could not find the data file \"%s\". This is "
	     "probably a fatal error. The program might "
	     "not be properly installed. Any data needed by "
	     "the modules should be in the same directory as "
	     "the modules.\n\n"
	     "You could try to find the file yourself though. "
	     "If you find it, click \"yes\".\n\n"
	     "Have you found the file?"),
			 filename.c_str())),
	  _("File not found!"),
	  wxYES_NO | wxICON_ERROR, GetTopWindow());

      if(answer != wxYES)
      {
	throw NoisyException
	  (wxString::Format(_("The data file \"%s\" was not found."),
			    filename.c_str()));
      }
      else
      {
	dir = wxDirSelector(
	      wxString::Format(_("Where can the file \"%s\" be found?"),
			       filename.c_str()),
	      wxEmptyString,	// default path
	      0,		// dialog style
	      wxDefaultPosition,
	      GetTopWindow());
	if(dir == "" || !wxDirExists(dir))
	{
	  throw NoisyException(_("Sorry, no valid directory has been entered."));
	}
	else
	{
	  m_module_path.Add(dir);
	  datafile = m_module_path.FindAbsoluteValidPath(filename);
	}
      }
    }
  }

  if(dir != wxEmptyString)
  {
    if(m_env == wxEmptyString)
    {
      m_env = dir;
    }
    else
    {
#ifdef wxUSE_UNIX
      m_env += ":" + dir;
#else
      m_env += ";" + dir;
#endif
    }
  }

  return datafile;
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}
catch(NullException &ex)
{
 // Promote any null exceptions fom GetTopWindow() to noisy ones.
  throw NoisyException(ex.what());
}

void Config::SetGenerateConfig(bool gc) throw()
{
  wxLogDebug("Entering SetGenerateConfig(%s);", (gc ? "true" : "false"));
  m_generate_config = gc;
}

void Config::LookForValidRoomListFile() throw(Exception)
{
  wxString str = m_roomlist_file.GetFullPath();
  if(! m_roomlist_file.IsOk())
  {
    wxConfig::Get()->Read("/RoomList/filename", &str);
    if(str == "")
    {
      str = MODULEDIR + wxString("/roomlist.txt");
      wxLogDebug("Trying \"%s\"", str.c_str());
      m_roomlist_file = str;
      if(! m_roomlist_file.FileExists())
      {
	m_roomlist_file = str = wxString("roomlist.txt");
	if(! m_roomlist_file.FileExists())
	  throw NoisyException(_("Could not read room list from configuration "
				 "file. Empty filename!"));
      }
    }
  }

  m_roomlist_file = str;
  if(m_roomlist_file.DirExists())
  {
    throw NoisyException
      (wxString::Format(_("Could not read room list. \"%s\" is a directory!"),
			m_roomlist_file.GetFullPath().c_str()));
  }
  else 
  {
    if(! m_roomlist_file.FileExists())
    {
      throw NoisyException
	(wxString::Format(_("Could not read room list. \"%s\" does not exist!"),
			  m_roomlist_file.GetFullPath().c_str()));
    }
  }
}

void Config::GenerateConfig() throw()
{
  if(m_generate_config)
  {
    wxLogDebug("In Config::GenerateConfig(), generating configuration");

    if((!wxConfig::Get()->Write("/RoomList/filename",
				m_roomlist_file.GetFullPath()))
       ||
       (!wxConfig::Get()->Write("/Modules/" + module_path_env,
				m_env))
       ||
       (!wxConfig::Get()->Write("/Locale/directory",
				m_localedir.GetFullPath())))
    {
      wxLogError(_("Could not save configuration"));
    }
  }
}
