// A mathematical game
// Copyright (C) 2004-2005 by Christian von Schultz <schultz@linux.nu>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <wx/wx.h>
#include <wx/log.h>
#include <wx/frame.h>
#include <wx/intl.h>
#include <wx/bitmap.h>
#include <wx/menuitem.h>
#include <wx/menu.h>
#include <wx/sizer.h>
#include <wx/textdlg.h>
#include <wx/filedlg.h>
#include <wx/msgdlg.h>
#include <wx/cshelp.h>

#include "checksetup.h"

#include "mainwindow.h"
#include "windowcontents.h"
#include "application.h"

#include "icon.xpm"
#include "../stock-icons-xpm/stock_open_16.xpm"
#include "../stock-icons-xpm/stock_save_16.xpm"
#include "../stock-icons-xpm/stock_exit_16.xpm"

#ifdef __WXDEBUG__
# include "../stock-icons-xpm/stock_convert_16.xpm"
#endif

namespace
{
  const char *default_extension = "." PACKAGE_TARNAME;
  wxString wildcard;
}

MainWindow::MainWindow() throw(Exception) try:
  wxFrame(NULL, -1, PACKAGE_STRING, wxDefaultPosition, wxSize(600,400)),
  m_cp(NULL)
{
  wxLogDebug("Constructing MainWindow.");

  wildcard  = wxString::Format(_("%s files (%s)|%s"),
			       PACKAGE_NAME,
			       "*." PACKAGE_TARNAME,
			       "*." PACKAGE_TARNAME);
  
  //////////////// Icon ////////////////
  {
    wxIcon *icon = new wxIcon(icon_xpm);
    SetIcon(*icon);		// makes a copy of the icon.
    delete icon;
  }

  //////////////// Help provider ////////////////
  wxHelpProvider::Set(new wxSimpleHelpProvider);

  //////////////// Menu bar ////////////////
  wxLogDebug("  Creating a menu bar.");
  
  wxMenuItem *mi = NULL;

  //// File menu ////
  wxMenu *file_menu = new wxMenu();

  // Open
  mi = new wxMenuItem(file_menu, wxID_OPEN, _("&Open...\tCtrl-O"),
		      _("Open a previously saved session"));
  mi->SetBitmap(wxBitmap(stock_open_16));

  // The art provider gives us ugly icons, even on GTK+
  //  mi->SetBitmap(wxArtProvider::GetBitmap(wxART_FILE_OPEN, wxART_MENU));
  file_menu->Append(mi);

  // Save
  mi = new wxMenuItem(file_menu, wxID_SAVE, _("&Save...\tCtrl-S"),
		      _("Save this session"));
  mi->SetBitmap(wxBitmap(stock_save_16));
  file_menu->Append(mi);

  // Quit
  file_menu->AppendSeparator();
  mi = new wxMenuItem(file_menu, wxID_EXIT, _("&Quit\tCtrl-Q"), _("Quit this program"));
  mi->SetBitmap(wxBitmap(stock_exit_16));
  file_menu->Append(mi);

  //// Debug menu ////
#ifdef __WXDEBUG__
  wxMenu *debug_menu = new wxMenu();
  mi = new wxMenuItem(debug_menu, MENU_REPLACE, _("&Replace...\tCtrl-R"),
		      _("Replace the current WindowContents."));
  mi->SetBitmap(wxBitmap(stock_convert_16));
  debug_menu->Append(mi);
#endif

  //// Add everything to a menu bar ////
  wxMenuBar *menu_bar = new wxMenuBar();  
  menu_bar->Append(file_menu, _("&File"));  

#ifdef __WXDEBUG__
  menu_bar->Append(debug_menu, _("&Debug"));
#endif

  SetMenuBar(menu_bar);

  //////////////// Status bar ////////////////
  wxLogDebug("  Creating a status bar with 2 panes.");
  CreateStatusBar(2);
  SetStatusBarPane(0);
  int *widths = new int[2];
  widths[0] = -1;		// variable size
  widths[1] = 200;		// used to display the amount of time left
  SetStatusWidths(2, widths);
  delete [] widths;

  //////////////// Main panel ////////////////
  m_panel = new wxPanel(this, -1,
			wxDefaultPosition,
			GetClientSize(),
			0); 	// we don't want wxTAB_TRAVERSAL

  wxBoxSizer *sizer = new wxBoxSizer(wxHORIZONTAL);
  sizer->Add(m_panel, 1, wxEXPAND, 0);
  SetSizer(sizer);
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

void MainWindow::OnExit(wxCommandEvent &event) throw()
{
  wxLogDebug("Got Exit command...");
  Close();
}

void MainWindow::OnSave(wxCommandEvent &event) throw()
try
{
  wxLogDebug("Got Save command...");
  wxString filename = wxFileSelector(_("Where do you want to save the session?"),
				     wxEmptyString, // default_path
				     wxEmptyString, // default_filename
				     default_extension,
				     wildcard,
				     wxSAVE | wxOVERWRITE_PROMPT,
				     this);
  wxLogDebug("Got filename %s", filename.c_str());


  if(filename != wxEmptyString)
  {

    if(filename.Right(wxString(default_extension).length()) != default_extension)
    {
      int answer = wxMessageBox(wxString::Format(
			        _("The filename you entered, \"%s\", does not end in "
				  "\"%s\". You will not be able to open that file. "
				  "Would you like to name the file \"%s\" instead?"),
				filename.c_str(), default_extension,
				(filename + default_extension).c_str()),
				_("Invalid filename"),
				wxYES_NO | wxCANCEL | wxICON_QUESTION,
				this);
      if(answer == wxCANCEL)
	return;

      if(answer == wxYES)
	filename << default_extension;
    }

    try
    {
      GetApp()->Save(filename);
    }
    catch(Exception &ex)
    {
      wxLogDebug("  MainWindow::OnSave(): Exception: %s", ex.what());
    }
  }
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}
catch(...)
{
  wxLogError("An exception was caught!");
}

void MainWindow::OnOpen(wxCommandEvent &event) throw()
try
{
  wxLogDebug("Got Open command...");
  wxString filename = wxFileSelector(_("Which file do you want to open?"),
				     wxEmptyString, // default_path
				     wxEmptyString, // default_filename
				     default_extension,
				     wildcard,
				     wxOPEN | wxFILE_MUST_EXIST,
				     this);
  if(filename != wxEmptyString)
  {
    try
    {
      GetApp()->Restore(filename);
    }
    catch(Exception &ex)
    {
      wxLogDebug("  MainWindow::OnOpen(): Exception: %s", ex.what());
    }
    catch(...)
    {
      wxLogDebug("  Unknown exception in OnOpen()");
    }
  }
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}
catch(...)
{
  wxLogError("An exception was caught!");
}

void MainWindow::OnClose(wxCloseEvent &event) throw()
try
{
  wxLogDebug("In OnClose(): destroying the window.");

  try
  {
    GetApp()->GetCurrentWC()->Destroy();
  }
  catch(Exception &ex)
  {
    wxLogDebug("  MainWindow::OnClose(): Exception: %s", ex.what());
  }
  catch(...)
  {
    wxLogDebug("  Unknown exception in OnClose()");
  }

  Destroy();
  wxLogDebug("  wxWindow::Destroy() has been called on the main window.");
}
catch(...)
{
  abort();
}

#ifdef __WXDEBUG__
void MainWindow::OnMenuReplace(wxCommandEvent &event) throw()
{
  wxLogDebug("Got Replace command from menu...");
  ReplaceSoon(wxGetTextFromUser(_("Which room do you want to go to?")));
}
#endif

void MainWindow::ReplaceSoon(const wxString &room) throw()
{
  wxLogTrace("entering", "MainWindow::ReplaceSoon(): There is room to be replaced.");
  wxCommandEvent event(wxEVT_COMMAND_MENU_SELECTED, ID_REPLACE);
  event.SetString(room.c_str());
  AddPendingEvent(event);
}

void MainWindow::DoSoon(CommandProcessor &cp)
{
  wxLogTrace("entering", "MainWindow::DoSoon(): regestering command processor.");
  wxCommandEvent event(wxEVT_COMMAND_MENU_SELECTED, ID_DO);
  m_cp = &cp;
  AddPendingEvent(event);
}

void MainWindow::OnReplace(wxCommandEvent &event) throw()
try
{
  if(wxString(event.GetString()) != wxEmptyString)
  {
    wxLogTrace("entering", "MainWindow::OnReplace, replacing module.");
    GetApp()->Replace(event.GetString());
  }
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}
catch(Exception &ex)
{
  if(!ex.Close())
    Destroy();
}

void MainWindow::OnDo(wxCommandEvent &event) throw()
try
{
  wxLogTrace("entering", "MainWindow::OnDo(): activating command processor.");
  (*m_cp)();
  m_cp = NULL;
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}
catch(Exception &ex)
{
  if(!ex.Close())
    Destroy();
}

BEGIN_EVENT_TABLE(MainWindow, wxFrame)
  EVT_MENU(wxID_EXIT,  MainWindow::OnExit)
  EVT_MENU(wxID_SAVE,  MainWindow::OnSave)
  EVT_MENU(wxID_OPEN,  MainWindow::OnOpen)
  EVT_MENU(ID_REPLACE, MainWindow::OnReplace)
  EVT_MENU(ID_DO,      MainWindow::OnDo)
#ifdef __WXDEBUG__
  EVT_MENU(MENU_REPLACE, MainWindow::OnMenuReplace)
#endif
  EVT_CLOSE(MainWindow::OnClose)
END_EVENT_TABLE();
