// A mathematical game
// Copyright (C) 2004-2005 by Christian von Schultz <schultz@linux.nu>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <wx/wx.h>
#include <wx/cmdline.h>
#include <wx/intl.h>
#include <iostream>
#include <new>
using std::cout;
using std::cerr;
using std::endl;

#include "checksetup.h"

#include "mainwindow.h"
#include "application.h"
#include "configuration.h"
#include "roomlist.h"
#include "wordwrap.h"
#include "timeguard_timeout.h"
#include "roomguard_datacontrol.h"

Application *Application::m_the_app = NULL;

Application *GetApp() throw(NullException)
{
  if(Application::m_the_app == NULL)
    throw NullException("Application::m_the_app");

  return Application::m_the_app;
}

// The constructor only sets the pointers to NULL and m_the_app to this, real
// object construction is done in Application::OnInit()
Application::Application() throw():
  timer(NULL),
  config(NULL),
  m_locale(NULL),
  m_roomlist(NULL),
  m_main_window(NULL),
  m_current_module(NULL)
{
  m_the_app = this;
}

bool Application::OnInit()
try
{
  m_current_module = NULL;

  m_locale = new wxLocale();
  m_locale->Init();


  // Possible debug messages:
  //
  //  wxLog::AddTraceMask("image");    //  Image creation and related messages
  //  wxLog::AddTraceMask("DC");       //  Drawing related
  //  wxLog::AddTraceMask("encoding"); //  Encoding related
  wxLog::AddTraceMask("entering");       //  Entering and leaving functions
  wxLog::AddTraceMask("One");
  wxLog::AddTraceMask("ChooseOne");
  wxLog::AddTraceMask("save");           //  Saving files
  //  wxLog::AddTraceMask("module");         //  Module and DLL related


  int encoding = static_cast<int>(wxLocale::GetSystemEncoding());
  if(encoding < 0)
    encoding = wxFONTENCODING_MAX + abs(encoding);
  wxLogTrace("encoding", "encoding = %d", encoding);

  if(wxLocale::GetSystemEncoding() == wxFONTENCODING_SYSTEM)
  {
    int answer = wxMessageBox(
        WordWrap(
          _("I could not detect your font encoding. This is a "
	    "serious error. It means that the files used by the "
	    "program (which use the UTF-8 encoding) will not "
	    "be converted to you current font encoding. That "
	    "could mean that messages using non-ASCII characters "
	    "will not be displayed at all. But I guess you "
	    "could try and see if it works anyway. Do you "
	    "want to continue?")),
	_("Encoding not detected!"),
	  wxYES_NO | wxNO_DEFAULT | wxICON_ERROR, NULL);
	
    if(answer != wxYES)
      return false;
  }
 
  config = new Config;
 
  config->FindLocaleDir(m_locale);
  m_locale->AddCatalog(PACKAGE_TARNAME);

  SetAppName(PACKAGE_NAME);
  SetVendorName(PACKAGE_VENDOR);

  if(! wxApp::OnInit())
  {
    delete config;    config = NULL;
    return false;		// If our base class doesn't want us to
				// proceed, we don't.
  }

  try
  {
    m_roomlist = new RoomList();
  }
  catch(Exception &ex)
  {
    delete m_roomlist; m_roomlist = NULL;
    delete config;     config = NULL;
    return false;
  }

  m_main_window = new MainWindow();
  SetTopWindow(m_main_window);
  m_main_window->Show(true);
  m_main_window->ShowFullScreen(false,
				wxFULLSCREEN_NOBORDER |
				wxFULLSCREEN_NOCAPTION);

  if(!m_locale->IsLoaded(PACKAGE_TARNAME))
  {
    wxLogWarning
      (WordWrap
       (wxString::Format
	("Couldn't load catalog \"%s\" from the directory "
	 "\"%s\".\nThe translations are disabled. "
	 "This could mean that the program is not properly "
	 "installed, or that there is no translation for your "
	 "language. If you would like to translate this program, "
	 "please contact %s.",
	 PACKAGE, LOCALEDIR, PACKAGE_BUGREPORT)));
  }

 
  try
  {
    if(m_file_to_open == wxEmptyString)
    {
      const wxExpr *p_expr = m_roomlist->GetExpr("start");
      InitialReplace(p_expr);
    }
    else
    {
      Restore(m_file_to_open);
    }
  }
  catch(Exception &ex)
  {
    wxLogError("Failed to start, exception: %s", ex.what());
    delete config;        config = NULL;
    delete m_roomlist;    m_roomlist = NULL;
    throw;
  }

  return true;		     
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}
catch(Exception &ex)
{
  return false;
}

wxString Application::LocaleCanonicalName() const throw()
{
  return m_locale->GetCanonicalName();
}

inline void get_module(wxString &dest, const wxExpr &expr) throw(Exception)
{
  if(! expr.GetAttributeValue("module", dest))
  {
    NoisyException(_("Malformed item in room list: no module to load."));
  }
}

void Application::AddTimeGuards(wxExpr *el) throw() // el is a wxExpr list
try
{
  if(el == NULL)
    throw NullException("No list.");

  if(el->Type() != wxExprList)
    throw NoisyException(_("Error in room list. The time guards must "
			   "be entered in a list."));

  for(wxExpr *item = el->GetFirst(); item != NULL; item = item->GetNext())
  {
    if(item->Type() != wxExprString)
    {
      wxLogWarning(_("Each item in a time guard list must be a string."));
      continue;
    }

    try
    {
      LoadTimeGuard(item->StringValue());
    }
    catch(...)
    {
      wxLogDebug("LoadTimeGuard() threw an exception.");
    }
  }
}
catch(...)
{
  wxLogDebug("AddTimeGuards has caught an exception.");
}

void Application::LoadTimeGuard(const wxString &functor) throw(Exception)
try
{
  wxString module_name;
  const wxExpr *clause = m_roomlist->GetExpr(functor);

  get_module(module_name, *clause); 
  Module<TimeGuard> *module = new Module<TimeGuard>(module_name);

  module->CreateObject();
  module->object->SetModule(module);
  module->object->Init(clause);
  AddTimeGuard(module->object);
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

void Application::AddRoomGuards(wxExpr *el) throw() // el is a wxExpr list
try
{
  if(el == NULL)
    throw NullException("No list.");

  if(el->Type() != wxExprList)
    throw NoisyException(_("Error in room list. The room guards must "
			   "be entered in a list."));

  for(wxExpr *item = el->GetFirst(); item != NULL; item = item->GetNext())
  {
    if(item->Type() != wxExprString)
    {
      wxLogWarning(_("Each item in a room guard list must be a string."));
      continue;
    }

    try
    {
      LoadRoomGuard(item->StringValue());
    }
    catch(...)
    {
      wxLogDebug("LoadRoomGuard() threw an exception.");
    }
  }
}
catch(...)
{
  wxLogDebug("AddRoomGuards has caught an exception.");
}

void Application::LoadRoomGuard(const wxString &functor) throw(Exception)
try
{
  wxString module_name;
  const wxExpr *clause = m_roomlist->GetExpr(functor);

  if(clause->Functor() == wxString("data_control"))
  {
    RGDataControl *rgdc = new RGDataControl();
    rgdc->Init(clause);
    AddRoomGuard(rgdc);
  }
  else
  {
    get_module(module_name, *clause);
    Module<RoomGuard> *module = new Module<RoomGuard>(module_name);

    module->CreateObject();
    module->object->SetModule(module);
    module->object->Init(clause);
    AddRoomGuard(module->object);
  }
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

template <class Guard> void reset_and_delete(Guard *&pg)
{
  if(pg != NULL)
  {
    if(pg->HasModule())
      delete pg->GetModule();	// the Module destructor will delete the Guard
    else
      delete pg;
    pg = NULL;
  }
}

int Application::OnExit()
{
  wxLogTrace("entering", "In Application::OnExit()");
  wxLog::FlushActive();

  if(m_current_module != NULL)
    delete m_current_module;
  
  if(m_roomlist != NULL)
    delete m_roomlist;

  for(RoomGuardIterator iter = RoomGuardBegin(); iter != RoomGuardEnd(); ++iter)
    reset_and_delete(*iter);

  m_room_guards.clear();

  for(TimeGuardIterator iter = TimeGuardBegin(); iter != TimeGuardEnd(); ++iter)
    reset_and_delete(*iter);

  m_time_guards.clear();	// removes all elements from the list
  delete timer;

  delete config;

  delete m_locale;

  wxLogTrace("entering", "wxApp::OnExit()");
  return wxApp::OnExit();
}

void Application::OnInitCmdLine(wxCmdLineParser &parser)
{
  // SetLogo is not intended for description of environment variables,
  // but since I want --help to show them...
  parser.SetLogo(wxString::Format(_("The environment variable \"%s\" "
				    "is the path to the modules."),
				  Config::module_path_env.c_str()));

  parser.AddSwitch("", "help", _("outputs this help message"));
  parser.AddSwitch("", "version",
		   wxString::Format(_("outputs the version of %s"),
				    PACKAGE_NAME));
  parser.AddOption("", "roomlist", _("read roomlist from the specified file"));
  parser.AddOption("", "localedir", _("the directory in which to search for translations"));
  parser.AddSwitch("", "generate-config",
		   _("use with --roomlist or --localedir, "
		     "updates program configuration"));
  parser.AddParam(_("filename"), wxCMD_LINE_VAL_STRING, wxCMD_LINE_PARAM_OPTIONAL);
}

bool Application::OnCmdLineParsed(wxCmdLineParser &parser)
{
  wxString something = "";
  if(parser.Found("help"))
  {
    parser.Usage();
    return false;
  }
  else if(parser.Found("version"))
  {
    cout << PACKAGE_STRING << endl;
    return false;		// Terminate here
  }

  if(parser.Found("roomlist", &something))
  {
    config->SetRoomListFile(something);
  }

  if(parser.Found("localedir", &something))
  {
    try
    {
      config->SetLocaleDir(something, &m_locale);
    }
    catch(Exception &ex) {}
    // If we fail, we simply use the default. SetLocaleDir() relies on
    // this behaviour in its communication with the user.
  }

  if(parser.Found("generate-config"))
  {
    config->SetGenerateConfig(true);
  }

  if(parser.GetParamCount() == 1)
    m_file_to_open = parser.GetParam(0);

  return true;			// Continue execution
}


// Data access
void Application::AddLabelledData(const wxString &label, wxExpr *value) throw(NullException)
{
  if(RGDataControl::current == NULL)
    throw NullException("RGDataControl::current == NULL");
  RGDataControl::current->AddLabelledData(label, value);
}

wxExpr *Application::cGetData(const wxString &label) const throw(NullException)
{
  if(RGDataControl::current == NULL)
    throw NullException("RGDataControl::current == NULL");
  return RGDataControl::current->cGetData(label);
}

void Application::ForgetData(const wxString &label) throw()
{
  if(RGDataControl::current != NULL)
    RGDataControl::current->ForgetData(label);  
}

inline void start_timer(const wxExpr &expr, Timer &timer)
{
  if(! timer.IsRunning())
  {
    wxString dest;
    if(expr.GetAttributeValue("start_timer", dest))
    {
      if(dest != "later")
	timer.Start();
    }
    else
    {
      timer.Start();
    }
  }
}

inline void stop_timer(const wxExpr &expr, Timer &timer)
{
  if(timer.IsRunning())
  {
    wxString dest;
    if(expr.GetAttributeValue("stop_timer", dest))
    {
      if(dest == "yes")
	timer.Stop();
    }
  }
}

WindowContents *Application::GetCurrentWC() throw(SilentException)
{
  if(m_current_module == NULL)
  {
    throw NullException("m_current_module");
  }
  else if(m_current_module->IsOk())
  {
    return m_current_module->object;
  }
  else
  {
    throw SilentException("The current module has no WindowContents object.");
  }
}

void Application::InitialReplace(const wxExpr *p_expr) throw(Exception)
try
{
  wxLogTrace("entering", "Application::InitialReplace");
  int time;

  //////////////// Get the required attributes ////////////////
  if(! p_expr->GetAttributeValue("timeout", m_timeout))
  {
    throw NoisyException
      (wxString::Format(_("Malformed item in file: \"%s\" requires "
			  "the attribute \"timeout\"."),
			p_expr->Functor().c_str()));
  }
  if(! p_expr->GetAttributeValue("time", time))
  {
    throw NoisyException
      (wxString::Format(_("Malformed item in file: \"%s\" requires "
			  "the attribute \"time\" (an integer)."),
			p_expr->Functor().c_str()));
  } 

  //////////////// Reset any previous state ////////////////
  for(RoomGuardIterator iter = RoomGuardBegin(); iter != RoomGuardEnd(); ++iter)
    reset_and_delete(*iter);

  for(TimeGuardIterator iter = TimeGuardBegin(); iter != TimeGuardEnd(); ++iter)
    reset_and_delete(*iter);

  m_time_guards.clear();
  m_room_guards.clear();

  if(timer != NULL)
    delete timer;


  //////////////// Start the new session ////////////////
  timer = new Timer(time);
  AddTimeGuard(new TimeOut(m_timeout));
  AddTimeGuards(p_expr->AttributeValue("time_guards"));
    
  AddRoomGuards(p_expr->AttributeValue("room_guards"));

  // If there's an attribute "current", we go through that
  // door. Otherwise, we try to load the current clause.
  wxString current = p_expr->Functor();
  p_expr->GetAttributeValue("current", current);

  Replace(current);
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

void Application::Replace(const wxString &room_name) throw(Exception)
try
{
  wxExpr *p_expr = NULL;
  
  try
  {
    p_expr = m_roomlist->GetExpr(room_name)->Copy();
  }
  catch(NullException &ex)
  {
    throw NoisyException
      (wxString::Format(_("Room not found in room list: \"%s\""),
			room_name.c_str()));
  }

  try
  {
    for(RoomGuardIterator iter = RoomGuardBegin();
	iter != RoomGuardEnd();
	++iter)
    {
      (*iter)->Replace(p_expr);
    }
  }
  catch(NullException &ex)
  {
    throw NoisyException(_("The room could not be loaded"));
  }

  Replace(p_expr);

  delete p_expr;
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

void Application::Replace(const wxExpr *p_expr) throw(Exception)
try
{
  wxString module_name;
  
  get_module(module_name, *p_expr);
  
  Module<WindowContents> *module = NULL;
  if(m_current_module != NULL)
  {
    if(m_current_module->GetLibraryName() == module_name)
      module = new Module<WindowContents>(*m_current_module);
  }
  if(module == NULL)
    module = new Module<WindowContents>(module_name);
  
  if(! module->IsLoaded())
  {
    delete module;
    throw SilentException("Could not load module.");
  }

  module->CreateObject();
  if(! module->IsOk())
  {
    delete module;
    throw SilentException("Could not create WindowContents.");
  }

  start_timer(*p_expr, *timer); // Start timer?
  stop_timer(*p_expr, *timer);  // Stop timer?
  module->object->Init(p_expr, m_main_window->m_panel);

  // WC objects communicate

  // New WC object destroys the old WC object

  if(m_current_module != NULL) // We delete the old WC object...
    delete m_current_module;
 
  m_current_module = module;  // ... and replace it with the new one
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

void Application::RemoveTimeGuard(TimeGuard* &tg)
{
  m_time_guards.remove(tg);

  if(tg != NULL)
  {
    if(tg->HasModule())
      delete tg->GetModule();	// the Module destructor will delete the TG
    else
      delete tg;
    tg = NULL;
  }
}

void Application::RemoveRoomGuard(RoomGuard* &rg)
{
  m_room_guards.remove(rg);

  if(rg != NULL)
  {
    if(rg->HasModule())
      delete rg->GetModule();	// the Module destructor will delete the RG
    else
      delete rg;
    rg = NULL;
  }
}

void Application::Save(const wxString &filename) throw(Exception)
try
{
  wxLogTrace("entering", "Save(\"%s\")", filename.c_str());
  wxLogTrace("save", "Creating new wxExprDatabase");
  wxExprDatabase *db = new wxExprDatabase();

  wxExpr *clause = new wxExpr("restore");

  {
    wxLogTrace("save", "  Appending current WindowContents");
    wxExpr *wc = GetCurrentWC()->cHibernate();
    clause->AddAttributeValueString("current", wc->Functor());
    db->Append(wc);
  }

  {
    wxExpr *rglist = new wxExpr(wxExprList);
    wxExpr *rg = NULL;
    wxExpr *rgname = NULL;
    for(RoomGuardIterator iter = RoomGuardBegin();
	iter != RoomGuardEnd();
	++iter)
    {
      try
      {
	wxLogTrace("save", "  Appending a RoomGuard");
	rg = (*iter)->cHibernate();
	rgname = new wxExpr(wxExprString, rg->Functor());
	rglist->Append(rgname);
	db->Append(rg);
      }
      catch(...)
      {
	wxLogDebug("    Caught exception while adding RoomGuard during save.");
      }
    }
    clause->AddAttributeValue("room_guards", rglist);
  }

  {
    wxExpr *tglist = new wxExpr(wxExprList);
    wxExpr *tg = NULL;
    wxExpr *tgname = NULL;
    for(TimeGuardIterator iter = TimeGuardBegin();
	iter != TimeGuardEnd();
	++iter)
    {
      try
      {
	wxLogTrace("save", "  Appending a TimeGuard");
	tg = (*iter)->cHibernate();
	tgname = new wxExpr(wxExprString, tg->Functor());
	tglist->Append(tgname);
	db->Append(tg);
      } 
      catch(...)
      {
	wxLogDebug("    Caught exception while adding TimeGuard during save.");
      }
    }
    clause->AddAttributeValue("time_guards", tglist);
  }

  clause->AddAttributeValue("time", static_cast<long>(*timer));
  clause->AddAttributeValueString("timeout", m_timeout);

  db->Append(clause);

  wxLogTrace("save", "  Writing to file");
  if(!db->Write(filename))
    throw NoisyException(_("Could not save the file!"));

  delete db;
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

void Application::Restore(const wxString &filename) throw(Exception)
try
{
  wxLogTrace("entering", "Restore(\"%s\"): Restoring session...", filename.c_str());

  m_roomlist->OpenFile(filename);

  InitialReplace(m_roomlist->GetExpr("restore"));
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

void GetLocalizedAttribute(const wxExpr *expr, const wxString &name,
			   wxString &dest) throw(Exception)
{
  int enc = static_cast<int>(wxLocale::GetSystemEncoding());
  if(enc < 0)
    enc = wxFONTENCODING_MAX + abs(enc);
  wxString encoding = wxString::Format("%d", enc);

  if(expr->GetAttributeValue(name + "_" + GetApp()->LocaleCanonicalName() + encoding,
			     dest))
    return;			// Found e.g. name_sv_SE1

  if(expr->GetAttributeValue((name + "_"
			      + GetApp()->LocaleCanonicalName().substr(0,2)
			      + encoding),
			     dest))
    return;			// Found e.g. name_sv1

  if(expr->GetAttributeValue(name + encoding, dest))
    return;			// Found e.g. name1


  if(expr->GetAttributeValue(name + "_" + GetApp()->LocaleCanonicalName()
			     + "_" + encoding,
			     dest))
    return;			// Found e.g. name_sv_SE_1


  if(expr->GetAttributeValue(name + "_" + GetApp()->LocaleCanonicalName(),
			     dest))
    return;			// Found e.g. name_sv_SE


  if(expr->GetAttributeValue((name + "_"
			      + GetApp()->LocaleCanonicalName().substr(0,2)
			      + "_" + encoding),
			     dest))
    return;			// Found e.g. name_sv_1

  if(expr->GetAttributeValue((name + "_"
			      + GetApp()->LocaleCanonicalName().substr(0,2)),
			     dest))
    return;			// Found e.g. name_sv

  if(expr->GetAttributeValue((name + "_" + encoding),
			     dest))
    return;			// Found e.g. name_1


  if(expr->GetAttributeValue(name,
			     dest))
    return;			// Found e.g. name
  
  throw SilentException(_("This wxExpr has no such attribute."));
}

wxColour GetColourAttribute(const wxExpr *expr, const wxString &name,
			    long red, long green, long blue) throw(Exception)
try
{
  wxExpr *local_copy = expr->Copy();
  wxExpr *el = NULL;
  local_copy->GetAttributeValue(name, &el);
  if(el != NULL)
  {
    wxExpr *item = el->GetFirst();
    if(item != NULL)
    {
      red = item->IntegerValue();
      item = item->GetNext();
    }
    if(item != NULL)
    {
      green = item->IntegerValue();
      item = item->GetNext();
    }
    if(item != NULL)
    {
      blue = item->IntegerValue();
    }
  }
 
  delete local_copy;
  return wxColour((red > 255) ? 255 : red,
		  (green > 255) ? 255 : green,
		  (blue > 255) ? 255 : blue);
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}
