// A mathematical game
// Copyright (C) 2004-2005 by Christian von Schultz <schultz@linux.nu>

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


#include <wx/object.h>
#include <wx/wxexpr.h>
#include <wx/log.h>
#include <wx/panel.h>
#include <wx/sizer.h>
#include <wx/stattext.h>
#include <wx/textctrl.h>
#include <wx/layout.h>
#include <wx/intl.h>
#include <wx/statbmp.h>
#include <wx/listctrl.h>

#include <climits>

#include <config.h>
#include "../checksetup.h"

#include "../application.h"
#include "../windowcontents.h"
#include "one.h"
#include "onedata.h"
#include "onepanel.h"

namespace One {

OnePanel::OnePanel(One *one, OneData *od) throw(Exception) try:
   m_one(one),
   m_data(od),
   wxPanel(one->main_panel, ONEPANEL,
	   wxDefaultPosition, one->main_panel->GetClientSize(),
	   wxTAB_TRAVERSAL)
{
  wxLogTrace("One", "OnePanel Constructor");
  
  // wxWS_EX_TRANSIENT: This can be used to prevent a window from
  // being used as an implicit parent for the dialogs which were
  // created without a parent. It is useful for the windows which can
  // disappear at any moment as creating children of such windows
  // results in fatal problems.
  SetExtraStyle(wxWS_EX_TRANSIENT);
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

void OnePanel::Start() throw()
try
{
  wxLogTrace("One", "OnePanel::Start()");

  this->SetFocus();
  
  Screen *current = m_data->cGetScreen();
  
  wxBoxSizer *sizer = new wxBoxSizer(wxVERTICAL);
  
  // Preparing the control for the answer
  wxLogDebug("cMakeAnswerCtrl(current)");
  wxWindow *answer_ctrl = cMakeAnswerCtrl(current);
  wxLogDebug("END cMakeAnswerCtrl(current)");
  wxASSERT(answer_ctrl != NULL); // If it were, cMakeAnswer should have thrown an exception.

  // Depending on the height of the answer_ctrl, we can decide which image is best:
  int acwidth = 0, acheight = 0;
  answer_ctrl->GetClientSize(&acwidth, &acheight);
  wxBitmap *best_bitmap = NULL;
  int mainwidth = 0, mainheight = 0;
  GetTopWindow()->GetClientSize(&mainwidth, &mainheight);
  mainheight -= acheight;

  if(m_one->m_theme != NULL)
  {
    mainheight -= m_one->m_theme->GetTopBorder();
    mainheight -= m_one->m_theme->GetBottomBorder();
    mainwidth -= m_one->m_theme->GetLeftBorder();
    mainwidth -= m_one->m_theme->GetRightBorder();
  }
  for(Screen::image_iterator iter = current->images.begin();
      iter != current->images.end();
      ++iter)
  {
    wxLogTrace("image", "Examining a bitmap.");
    if(best_bitmap == NULL)
    {
      wxLogTrace("image", "  Current bitmap is NULL, accepting the new one.");
      best_bitmap = *iter;
    }
    else
    {
      if(*iter == NULL)
      {
	wxLogTrace("image", "  New bitmap is NULL, keeping the old one.");
	continue;
      }
      
      int w1 = best_bitmap->GetWidth();
      int h1 = best_bitmap->GetHeight();
      int w2 = (*iter)->GetWidth();
      int h2 = (*iter)->GetHeight();
      
      wxLogTrace("image", "  Current bitmap is %dx%d, new one is %dx%d, window is "
		 "%dx%d.", w1, h1, w2, h2, mainwidth, mainheight);
      
      if((mainheight < h1) && (h2 <  h1))
      {
	wxLogTrace("image",
		   "    The current bitmap is too high. The new one is better.");
	best_bitmap = *iter;
      }
      else if((mainwidth < w1) && (w2 < w1))
      {
	wxLogTrace("image",
		   "    The current bitmap is too wide. The new one is better.");
	best_bitmap = *iter;
      }
      else if((mainwidth < w2) || (mainheight < h2))
      {
	wxLogTrace("image",
		   "    The new bitmap is too large.");
	continue;
      }
      else
      {
	wxLogTrace("image",
		   "    The current bitmap and the new bitmap are both good (or both bad).");
	
	// If we arrive here, both best_bitmap and iter are
	// acceptable (either that, or both are terrible).
	if(h1 + w1 < h2 + w2)
	{
	  wxLogTrace("image",
		     "      %d + %d < %d + %d  =>  %d < %d", h1, w1, h2, w2, h1+w1, h2+w2);
	  best_bitmap = *iter;
	}
      }
    }
  }
  
  if(best_bitmap == NULL)
    throw NoisyException(_("Error in data file. Could not find a bitmap."));

  int bottom_padding = mainheight - best_bitmap->GetHeight();
  if(m_one->m_theme != NULL)
    bottom_padding -= m_one->m_theme->GetBottomBorder();
  bottom_padding /= 4;
  
  // Now that we have a good bitmap and a good answer_ctrl, we add them to the sizer.
  wxBoxSizer *bitmap_sizer = new wxBoxSizer(wxHORIZONTAL);

  bitmap_sizer->Add(new wxStaticBitmap(this, -1, *best_bitmap),
		    0, wxCENTRE);
  sizer->Add(bitmap_sizer, 1, wxCENTRE | wxNORTH,
	     ((m_one->m_theme != NULL)? m_one->m_theme->GetTopBorder(): 0));
  

  sizer->Add(answer_ctrl, 0, wxCENTRE | wxSOUTH,
	     ((m_one->m_theme != NULL)?
	      m_one->m_theme->GetBottomBorder() + bottom_padding: bottom_padding));
  sizer->SetItemMinSize(answer_ctrl, acwidth + 2, acheight);
  
  this->SetSizerAndFit(sizer);
  delete current;

  wxBoxSizer *main_hsizer = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer *main_vsizer = new wxBoxSizer(wxVERTICAL);
  main_hsizer->Add(this, 1, wxCENTRE | wxEXPAND, 10);
  main_vsizer->Add(main_hsizer, 1, wxCENTRE | wxEXPAND, 10);
  m_one->main_panel->SetSizer(main_vsizer);
  
  m_one->main_panel->Layout();
}
catch(std::bad_alloc &ex)
{
  throw NoisyBadAlloc();
}

OnePanel::~OnePanel()
{
  wxLogTrace("One", "OnePanel destructor");
}

void OnePanel::OnPaint(wxPaintEvent &event) throw()
try
{
  wxLogTrace("DC", "Creating a new wxPaintDC: Repainting...");
  wxPaintDC dc(this);
  Draw(dc);
}
catch(Exception &ex)
{
  if(!ex.Close())
    abort();
}

void OnePanel::Draw(wxDC &dc) throw()
{
  if(m_one->m_theme != NULL)
  {
    int width = 0, height = 0;
    GetSize(&width, &height);
  
    wxLogTrace("One", "Drawing theme");
    m_one->m_theme->Draw(dc, width, height);
  }
}

BEGIN_EVENT_TABLE(OnePanel, wxPanel)
  EVT_PAINT(          OnePanel::OnPaint)
END_EVENT_TABLE()

}
