#!/bin/bash
#=======================================================================
# * Version: $Id$
# * Project: MediaTex
# * Module : Reversibility use case
# *
# * This script provide facilities to work with postgres
#
# MediaTex is an Electronic Records Management System
# Copyright (C) 2014 2015 Nicolas Roche
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#=======================================================================
#set -x

# This script resume the operations to set up a view from
# several collection tables

# display usage
function usage {
    echo -e "\nUsage: "
    echo -e "$0 \tCREDENTIALS_PATH {"
    echo -e "\t\t  {create|drop} account"
    echo -e "\t\t| {create|drop} view"
    echo -e "\t\t|         drop  TABLE_PREFIX }"
    echo -e "\t\t|         run   SCRIPT_PATH }"
    exit
}

# load credentials
# $1: the credential file to load
function load_credentials {
    if [ ! -f $1 ]; then
	usage
    fi

    # do load the credential file
    . $1

    if [ -z $DB_HOST ]; then
    	echo "$DB_HOST"
    	echo "no DB_HOST variable found in credential file"
    	exit
    fi
    if [ -z $DB_PORT ]; then
	echo "no DB_PORT variable found in credential file"
	exit
    fi
    if [ -z $DB_BASE ]; then
	echo "no DB_BASE variable found in credential file"
	exit
    fi
    if [ -z $DB_USER ]; then
	echo "no DB_USER variable found in credential file"
	exit
    fi
    if [ -z $DB_PASSWD ]; then
	echo "no DB_PASSWD variable found in credential file"
	exit
    fi
    if [ -z $VIEW_PRE ]; then
	echo "no VIEW_PRE variable found in credential file"
	exit
    fi
    if [ -z "$TABLE_PRES" ]; then
	echo "no TABLE_PRES variable found in credential file"
	exit
    fi
    if [ -z "$TABLE_POSTS" ]; then
	echo "no TABLE_POSTS variable found in credential file"
	exit
    fi
}

# drop account
function drop_account {
    # supprime la base
    QUERY="drop database \\\"$DB_USER\\\";"
    su postgres -c "psql -d postgres -c \"$QUERY\"";

    # supprime un utilisateur postgres
    su postgres -c "dropuser $DB_USER"
}

# create account
function create_account {
    # ajoute un utilisateur postgres
    su postgres -c "createuser -SDR $DB_USER"

    # modifie son mot de passe
    QUERY="ALTER ROLE \\\"$DB_USER\\\" WITH PASSWORD '$DB_PASSWD'"
    su postgres -c "psql -d postgres -c \"$QUERY\"";

    # créer la base
    QUERY="create database \\\"$DB_BASE\\\" owner=\\\"$DB_USER\\\"\
 encoding='UTF8';"
    su postgres -c "psql -d postgres -c \"$QUERY\"";
}

# run the queries
function run_queries {
    PGPASSWORD="$DB_PASSWD" \
	psql -U $DB_USER -d $DB_BASE -h $DB_HOST -p $DB_PORT \
	-f manageDb.sql
}

# drop tables
# $1: the suffix for tables to drop
function drop_table {
    echo "" > manageDb.sql
    for POST in $TABLE_POSTS; do
	echo "DROP TABLE ${1}_${POST};" \
	    >> manageDb.sql
    done
    echo ";" >> manageDb.sql
    run_queries
}

# drop views
function drop_views {
    echo "" > manageDb.sql
    for POST in $TABLE_POSTS; do
	echo "DROP VIEW ${VIEW_PRE}_${POST};" \
	    >> manageDb.sql
    done
    echo ";" >> manageDb.sql
    run_queries
}

# create views
function create_views {
    echo "" > manageDb.sql
    for POST in $TABLE_POSTS; do
	echo -n "CREATE VIEW ${VIEW_PRE}_${POST} AS " \
	    >> manageDb.sql
	
	first=1
	for PREFIX in $TABLE_PRES; do
	    if [ $first -eq 0 ]; then
		echo -n " UNION " >> manageDb.sql
	    fi
	    echo -n "SELECT *, '${PREFIX}' as coll FROM ${PREFIX}_${POST}" \
		>> manageDb.sql
	    first=0
	done
	echo ";" >> manageDb.sql
    done
    run_queries
}

# main procedure
[ $# -eq 3 ] || usage
load_credentials $1
case $3 in

    # manage account
    account)
	echo "... Manage account"
        # need to be root
	if [ $(id -u) -ne 0 ]; then
	    echo "you need to be root"
	    exit;
	fi
	case $2 in
	    create)
		create_account
		;;
	    drop)
		drop_account
		;;
	    *)
		usage
		;;
	esac
	;;

    # manage view
    view)
	echo "... Manage view"
	case $2 in
	    create)
		create_views
		;;
	    drop)
		drop_views
		;;
	    *)
		usage
		;;
	esac
	;;

    # manage tables
    *)
	case $2 in
	    drop)
		echo "... Manage suffix: $3"
		drop_tables $3
		;;
	    run)
		echo "... Run script: $3"
		cp $3 manageDb.sql
		run_queries
		;;
	    *)
		usage
		;;
	esac
	;;
esac
