/*=======================================================================
 * Version: $$
 * Project: MediaTeX
 * Module : misc for seda02
 *
 * common functions

 MediaTex is an Electronic Records Management System
 Copyright (C) 2016  Nicolas Roche
 
 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 =======================================================================*/

#include "misc.h"
#include "misc/log.h"
#include "misc/md5sum.h"
#include "memory/archive.h"

/*=======================================================================
 * Function   : getArchiveIds
 * Description: Get the md5sum and size of a file
 * Synopsis   : int getArchiveIds(char* path, char* hash, off_t size)
 * Input      : char* path
 * Output     : char* hash: md5sum of the input file
 *              off_t size: size of the input file
 * Note       : copy/paste from mdtxUploadFile
 =======================================================================*/
int 
getArchiveIds(char* path, char* hash, off_t* size)
{
  int rc = FALSE;
  struct stat statBuffer;
  CheckData md5; 

   // get file attributes (size)
  if (stat(path, &statBuffer)) {
    logMain(LOG_ERR, "status error on %s: %s", path, strerror(errno));
    goto error;
  }

  // compute hash
  memset(&md5, 0, sizeof(CheckData));
  md5.path = path;
  md5.size = statBuffer.st_size;
  md5.opp = CHECK_CACHE_ID;
  if (!doChecksum(&md5)) goto error;

  strncpy(hash, md5.fullMd5sum, MAX_SIZE_MD5);
  hash[MAX_SIZE_MD5] = 0;
  *size = statBuffer.st_size;

  rc = TRUE;
 error:
  if (!rc) {
     logMain(LOG_ERR, "%s", "getArchiveIds fails");
  }
  return rc;
}


/*=======================================================================
 * Function   : sha1sum2string
 * Description: hexa[20] to char[40] convertion
 * Synopsis   : char* sha1sum2string(
 *                              unsigned char sha1sum[SHA_DIGEST_LENGTH], 
 *                              char rc[MAX_SIZE_SHA1 + 1])
 * Input      : unsigned char sha1sum[SHA_DIGEST_LENGTH]: 
 *               the sha1sum to convert
 * Output     : char rc[MAX_SIZE_SHA1 + 1]: the input array
 =======================================================================*/
char* 
sha1sum2string(unsigned char sha1sum[SHA_DIGEST_LENGTH], 
	      char rc[MAX_SIZE_SHA1 + 1])
{
  int n;

  for(n=0; n<SHA_DIGEST_LENGTH; n++) 
    sprintf(rc+(n<<1), "%02x", sha1sum[n]);

  rc[MAX_SIZE_SHA1] = (char)0;
  return rc;
}

/*=======================================================================
 * Function   : computeSha1
 * Description: Compute sha1sum 
 * Synopsis   : static int computeSha1(char* path, off_t size, 
 *                                    char fullSha1sum[MAX_SIZE_SHA1+1])
 * Input      : char* path: inpout file we compute the sha1sum
 *              off_t size: size of the input file
 * Output     : char chacksum[MAX_SIZE_SHA1 + 1]): the resuling sha1sum 
 =======================================================================*/
int 
computeSha1(char* path, off_t size, char checksum[MAX_SIZE_SHA1 + 1])
{
  int rc = FALSE;
  int fd = -1;
  SHA_CTX c;
  char buf[512];
  ssize_t bytes;
  ssize_t sum = 0;
  unsigned char sha1sum[SHA_DIGEST_LENGTH];

  logMain(LOG_DEBUG, "computeFullSha1 %s:%i", path, size);
  
  if (checksum == NULL) {
    logMain(LOG_ERR, "%s", "please allocate checksum argument");
    goto error;
  } 

  if ((fd = open(path, O_RDONLY)) == -1) {
    logMain(LOG_ERR, "open: %s", strerror(errno));
    goto error;
  }

  SHA1_Init(&c);
  if (&c == (SHA_CTX*)0) {
    logMain(LOG_ERR, "%s", "fails to allocate SHA_CTX argument");
    goto error;
  } 
  
  bytes=read(fd, buf, 512);
  while((!size || sum < size) && bytes > 0) {
    sum += bytes;
    SHA1_Update(&c, buf, bytes);
    bytes=read(fd, buf, 512);
  }
  
  SHA1_Final(sha1sum, &c);
  checksum = sha1sum2string(sha1sum, checksum);
  logMain(LOG_INFO, "%s  full sha1sum computed on %llu bytes", 
	  checksum, (long long unsigned int)sum);

  rc = TRUE;
 error:
  if (fd != -1 && close(fd) == -1) {
    logMain(LOG_ERR, "close: %s", strerror(errno));
    rc = FALSE;
  }
  if (!rc) {
     logMain(LOG_ERR, "%s", "computeSha1 fails");
  }
  return(rc);
}
