package net.percederberg.mibble;

import java.io.File;
import java.io.IOException;

import net.percederberg.mibble.value.ObjectIdentifierValue;

public class MibbleTester {

    public static void main(String[] args) {
        MibLoader  loader = new MibLoader();
        Mib        mib;
        Mib[]      mibs;
        File       file;
        int        errors = 0;
        int        warnings = 0;

        // Check command-line arguments
        if (args.length < 1) {
            printHelp("No file(s) specified");
            System.exit(1);
        }

        // Parse MIB files
        for (int i = 0; i < args.length; i++) {
            try {
                System.out.print("Reading " + args[i] + "... ");
                System.out.flush();
                file = new File(args[i]);
                loader.addDir(file.getParentFile());
                mib = loader.load(file);
                System.out.println("[OK]");
                if (mib.getLog().warningCount() > 0) {
                    mib.getLog().printTo(System.out);
                    warnings++;
                }
            } catch (IOException e) {
                System.out.println("[FAILED]");
                printError(args[i], e);
                errors++;
            } catch (MibLoaderException e) {
                System.out.println("[FAILED]");
                e.getLog().printTo(System.out);
                errors++;
            } catch (RuntimeException e) {
                System.out.println();
                printInternalError(e);
                System.exit(1);
            }
        }

        // Print OID children
        mib = loader.getMib("TCP-MIB");
        MibValueSymbol sym = (MibValueSymbol) mib.getSymbol("tcpConnEntry");
        ObjectIdentifierValue oid = (ObjectIdentifierValue) sym.getValue();
        System.out.println("Parent: " + oid);
        for (int i = 0; i < oid.getChildCount(); i++) {
            System.out.println("Child:  " + oid.getChild(i));
        }

        // Return error count
        System.exit(errors);
    }

    /**
     * Prints command-line help information.
     *
     * @param error          an optional error message, or null
     */
    private static void printHelp(String error) {
        if (error != null) {
            System.err.print("Error: ");
            System.err.println(error);
            System.err.println();
        }
    }

    /**
     * Prints an internal error message. This type of error should
     * only be reported when run-time exceptions occur, such as null
     * pointer and the likes. All these error should be reported as
     * bugs to the program maintainers.
     *
     * @param e              the exception to be reported
     */
    private static void printInternalError(Exception e) {
        e.printStackTrace();
    }

    /**
     * Prints a file not found error message.
     *
     * @param file           the file name not found
     * @param e              the detailed exception
     */
    private static void printError(String file, IOException e) {
        StringBuffer  buffer = new StringBuffer();

        buffer.append("Error: couldn't open file:");
        buffer.append("\n    ");
        buffer.append(file);
        System.out.println(buffer.toString());
    }
}
